# This code is modified by
# https://github.com/kuangliu/pytorch-cifar/blob/master/models/resnet.py
import torch
import torch.nn as nn
import torch.nn.functional as F
from collections import OrderedDict

import torch
import torch.nn as nn


class MaskBatchNorm2d(nn.BatchNorm2d):
    def __init__(self, num_features, eps=1e-5, momentum=0.1,
                 affine=True, track_running_stats=True):
        super(MaskBatchNorm2d, self).__init__(
            num_features, eps, momentum, affine, track_running_stats)

    def forward(self, input):
        input, weight = input[0], input[1].detach()
        self._check_input_dim(input)

        exponential_average_factor = 0.0

        if self.training and self.track_running_stats:
            if self.num_batches_tracked is not None:
                self.num_batches_tracked += 1
                if self.momentum is None:  # use cumulative moving average
                    exponential_average_factor = 1.0 / float(self.num_batches_tracked)
                else:  # use exponential moving average
                    exponential_average_factor = self.momentum

        # calculate running estimates
        if self.training:
            # weight = torch.ones((input.size(0),), dtype=input.dtype, device=input.device) * 1. / input.size(0)
            with torch.no_grad():
                inner_weight = weight / weight.sum()
            mean = (inner_weight[:, None] * input.mean([2, 3])).sum([0])
            mean_x2 = (inner_weight[:, None] * (input ** 2).mean([2, 3])).sum([0])
            var = mean_x2 - (mean ** 2)

            # mean = input.mean([0, 2, 3])
            # # use biased var in train
            # var = input.var([0, 2, 3], unbiased=False)

            n = input.numel() / input.size(1)
            with torch.no_grad():
                if self.track_running_stats:
                    self.running_mean = exponential_average_factor * mean \
                                        + (1 - exponential_average_factor) * self.running_mean
                    # update running_var with unbiased var
                    self.running_var = exponential_average_factor * var * n / (n - 1) \
                                       + (1 - exponential_average_factor) * self.running_var
        else:
            mean = self.running_mean
            var = self.running_var

        input = (input - mean[None, :, None, None]) / (torch.sqrt(var[None, :, None, None] + self.eps))
        if self.affine: # the watermark sample doesn't influence bn
            input = (input * self.weight[None, :, None, None] + self.bias[None, :, None, None]) * weight[:, None, None, None] +\
                (input * self.weight[None, :, None, None].detach() + self.bias[None, :, None, None].detach()) * (1-weight[:, None, None, None])

        return input, weight



class CustomConv2d(nn.Conv2d):
    def __int__(self, in_channels, out_channels, kernel_size, stride=1, bias=True):
        super(CustomConv2d, self).__int__(in_channels, out_channels, kernel_size, stride, bias=bias)

    def forward(self, inputs):
        x, index = inputs[0], inputs[1]
        return super(CustomConv2d, self).forward(x), index


class CustomReLU(nn.ReLU):
    def __init__(self):
        super(CustomReLU, self).__init__()

    def forward(self, inputs):
        x, index = inputs[0], inputs[1]
        return super(CustomReLU, self).forward(x), index


class MBasicBlock(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, stride=1, norm_layer=None):
        super(MBasicBlock, self).__init__()
        self.conv1 = CustomConv2d(
            in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn1 = norm_layer(planes)
        self.conv2 = CustomConv2d(planes, planes, kernel_size=3,
                                  stride=1, padding=1, bias=False)
        self.bn2 = norm_layer(planes)
        self.relu = CustomReLU()

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(OrderedDict([
                ('conv', CustomConv2d(in_planes, self.expansion*planes,
                             kernel_size=1, stride=stride, bias=False)),
                ('bn',norm_layer(self.expansion*planes))
            ]))

    def forward(self, x):
        out = self.relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        out = (out[0] + self.shortcut(x)[0], out[1])
        out = self.relu(out)
        return out


class Bottleneck(nn.Module):
    expansion = 4

    def __init__(self, in_planes, planes, stride=1, norm_layer=None):
        super(Bottleneck, self).__init__()
        self.conv1 = CustomConv2d(in_planes, planes, kernel_size=1, bias=False)
        self.bn1 = norm_layer(planes)
        self.conv2 = CustomConv2d(planes, planes, kernel_size=3,
                                  stride=stride, padding=1, bias=False)
        self.bn2 = norm_layer(planes)
        self.conv3 = CustomConv2d(planes, self.expansion *
                                  planes, kernel_size=1, bias=False)
        self.bn3 = norm_layer(self.expansion*planes)
        self.relu = CustomReLU()

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(
                CustomConv2d(in_planes, self.expansion*planes,
                             kernel_size=1, stride=stride, bias=False),
                norm_layer(self.expansion*planes)
            )

    def forward(self, x):
        out = self.relu(self.bn1(self.conv1(x)))
        out = self.relu(self.bn2(self.conv2(out)))
        out = self.bn3(self.conv3(out))
        out = (out[0] + self.shortcut(x)[0], out[1])
        out = self.relu(out)
        return out


class MResNet(nn.Module):
    def __init__(self, block, num_blocks, num_classes=10):
        super(MResNet, self).__init__()
        self._norm_layer = MaskBatchNorm2d
        self.in_planes = 64

        self.conv1 = CustomConv2d(3, 64, kernel_size=3,
                                  stride=1, padding=1, bias=False)
        self.bn1 = self._norm_layer(64)
        self.layer1 = self._make_layer(block, 64, num_blocks[0], stride=1)
        self.layer2 = self._make_layer(block, 128, num_blocks[1], stride=2)
        self.layer3 = self._make_layer(block, 256, num_blocks[2], stride=2)
        self.layer4 = self._make_layer(block, 512, num_blocks[3], stride=2)
        self.linear = nn.Linear(512*block.expansion, num_classes)
        self.relu = CustomReLU()

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1]*(num_blocks-1)
        layers = []
        for stride in strides:
            layers.append(block(self.in_planes, planes, stride, self._norm_layer))
            self.in_planes = planes * block.expansion
        return nn.Sequential(*layers)

    def forward(self, x, mask=None):
        if mask is None:
            mask = torch.zeros(len(x), dtype=torch.long).to(x.device)
        mask = 1. - mask
        inputs = (x, mask)
        out = self.relu(self.bn1(self.conv1(inputs)))
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.layer4(out)
        out = F.avg_pool2d(out[0], 4)
        out = out.view(out.size(0), -1)
        out = self.linear(out)
        return out


def MResNet18(num_classes=10):
    return MResNet(MBasicBlock, [2, 2, 2, 2], num_classes)


# def weighted_resnet34(num_classes=10):
#     return ResNet(WBasicBlock, [3, 4, 6, 3], num_classes)


# def weighted_resnet50(num_classes=10):
#     return ResNet(Bottleneck, [3, 4, 6, 3], num_classes)


# def weighted_resnet101(num_classes=10):
#     return ResNet(Bottleneck, [3, 4, 23, 3], num_classes)


# def weighted_resnet152(num_classes=10):
#     return ResNet(Bottleneck, [3, 8, 36, 3], num_classes)

# test()