'''ResNet in PyTorch.

BasicBlock and Bottleneck module is from the original ResNet paper:
[1] Kaiming He, Xiangyu Zhang, Shaoqing Ren, Jian Sun
    Deep Residual Learning for Image Recognition. arXiv:1512.03385

PreActBlock and PreActBottleneck module is from the later paper:
[2] Kaiming He, Xiangyu Zhang, Shaoqing Ren, Jian Sun
    Identity Mappings in Deep Residual Networks. arXiv:1603.05027
'''
import torch
import torch.nn as nn
import torch.nn.functional as F

from torch.autograd import Variable
from collections import OrderedDict

def conv3x3(in_planes, out_planes, stride=1):
    return nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride, padding=1, bias=False)


class BasicBlockGN(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, stride=1,num_groups=1):
        super(BasicBlockGN, self).__init__()
        self.conv1 = conv3x3(in_planes, planes, stride)
        self.bn1 = nn.GroupNorm(num_groups if num_groups>0 else planes, planes)
        self.conv2 = conv3x3(planes, planes)
        self.bn2 = nn.GroupNorm(num_groups if num_groups>0 else planes, planes)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(OrderedDict([
                ('conv', nn.Conv2d(in_planes, self.expansion*planes, kernel_size=1, stride=stride, bias=False)),
                ('bn', nn.GroupNorm(num_groups if num_groups>0 else self.expansion*planes, self.expansion*planes))
                ]))

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        out += self.shortcut(x)
        out = F.relu(out)
        return out

class DropoutBasicBlock(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, stride=1):
        super(DropoutBasicBlock, self).__init__()
        self.conv1 = conv3x3(in_planes, planes, stride)
        self.bn1 = nn.BatchNorm2d(planes)
        self.dropout = nn.Dropout2d(0.2)
        self.conv2 = conv3x3(planes, planes)
        self.bn2 = nn.BatchNorm2d(planes)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(OrderedDict([
                ('conv', nn.Conv2d(in_planes, self.expansion*planes, kernel_size=1, stride=stride, bias=False)),
                ('bn', nn.BatchNorm2d(self.expansion*planes)),
                ('dropout', nn.Dropout2d(0.2))
                ]))

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.dropout(self.bn2(self.conv2(out)))
        out += self.shortcut(x)
        out = F.relu(out)
        return out

class NSBlock(nn.Module):
    "No shortcut"
    expansion = 1

    def __init__(self, in_planes, planes, stride=1):
        super(NSBlock, self).__init__()
        self.conv1 = conv3x3(in_planes, planes, stride)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = conv3x3(planes, planes)
        self.bn2 = nn.BatchNorm2d(planes)


    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        out = F.relu(out)
        return out


class PreActBlock(nn.Module):
    '''Pre-activation version of the BasicBlock.'''
    expansion = 1

    def __init__(self, in_planes, planes, stride=1):
        super(PreActBlock, self).__init__()
        self.bn1 = nn.BatchNorm2d(in_planes)
        self.conv1 = conv3x3(in_planes, planes, stride)
        self.bn2 = nn.BatchNorm2d(planes)
        self.conv2 = conv3x3(planes, planes)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(OrderedDict([
                ('conv', nn.Conv2d(in_planes, self.expansion*planes, kernel_size=1, stride=stride, bias=False))
            ]))

    def forward(self, x):
        out = F.relu(self.bn1(x))
        shortcut = self.shortcut(out)
        out = self.conv1(out)
        out = self.conv2(F.relu(self.bn2(out)))
        out += shortcut
        return out


class Bottleneck(nn.Module):
    expansion = 4

    def __init__(self, in_planes, planes, stride=1):
        super(Bottleneck, self).__init__()
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)
        self.conv3 = nn.Conv2d(planes, self.expansion*planes, kernel_size=1, bias=False)
        self.bn3 = nn.BatchNorm2d(self.expansion*planes)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(OrderedDict([
                ('conv',nn.Conv2d(in_planes, self.expansion*planes, kernel_size=1, stride=stride, bias=False)),
                ('bn',nn.BatchNorm2d(self.expansion*planes))
            ]))

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = F.relu(self.bn2(self.conv2(out)))
        out = self.bn3(self.conv3(out))
        out += self.shortcut(x)
        out = F.relu(out)
        return out


class PreActBottleneck(nn.Module):
    '''Pre-activation version of the original Bottleneck module.'''
    expansion = 4

    def __init__(self, in_planes, planes, stride=1):
        super(PreActBottleneck, self).__init__()
        self.bn1 = nn.BatchNorm2d(in_planes)
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn3 = nn.BatchNorm2d(planes)
        self.conv3 = nn.Conv2d(planes, self.expansion*planes, kernel_size=1, bias=False)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(OrderedDict([
                ('conv', nn.Conv2d(in_planes, self.expansion*planes, kernel_size=1, stride=stride, bias=False))
            ]))

    def forward(self, x):
        out = F.relu(self.bn1(x))
        shortcut = self.shortcut(out)
        out = self.conv1(out)
        out = self.conv2(F.relu(self.bn2(out)))
        out = self.conv3(F.relu(self.bn3(out)))
        out += shortcut
        return out


class ResNetGN(nn.Module):
    def __init__(self, block, num_blocks, num_classes=10, num_groups=1):
        super(ResNetGN, self).__init__()
        self.in_planes = 64

        self.conv1 = conv3x3(3,64)
        self.bn1 = nn.GroupNorm(num_groups if num_groups>0 else 64, 64)
        self.layer1 = self._make_layer(block, 64, num_blocks[0], 1, num_groups)
        self.layer2 = self._make_layer(block, 128, num_blocks[1], 2, num_groups)
        self.layer3 = self._make_layer(block, 256, num_blocks[2], 2, num_groups)
        self.layer4 = self._make_layer(block, 512, num_blocks[3], 2, num_groups)
        self.linear = nn.Linear(512*block.expansion, num_classes)

    def _make_layer(self, block, planes, num_blocks, stride, num_groups):
        strides = [stride] + [1]*(num_blocks-1)
        layers = []
        for stride in strides:
            layers.append(block(self.in_planes, planes, stride, num_groups))
            self.in_planes = planes * block.expansion
        return nn.Sequential(*layers)

    def forward(self, x, mask=None, lin=0, lout=6):
        out = x
        if lin < 1 and lout > -1:
            out = self.conv1(out)
            out = self.bn1(out)
            out = F.relu(out)
        if lin < 2 and lout > 0:
            out = self.layer1(out)
        if lin < 3 and lout > 1:
            out = self.layer2(out)
        if lin < 4 and lout > 2:
            out = self.layer3(out)
        if lin < 5 and lout > 3:
            out = self.layer4(out)
        # if lout > 4:
        #     out = F.avg_pool2d(out, 4)
        #     out = out.view(out.size(0), -1)
        #     out = self.linear(out)
        if lin < 6 and lout > 4:
            out = F.avg_pool2d(out, 4)
            out = out.view(out.size(0), -1)
        if lout > 5:
            out = self.linear(out)
        return out

def ResNetLN18(num_classes=10):
    return ResNetGN(BasicBlockGN, [2,2,2,2], num_classes, 1)

def ResNetIN18(num_classes=10):
    return ResNetGN(BasicBlockGN, [2,2,2,2], num_classes, -1)

def ResNetGN18(num_classes=10):
    return ResNetGN(BasicBlockGN, [2,2,2,2], num_classes, 32)

def ResNetGN1618(num_classes=10):
    return ResNetGN(BasicBlockGN, [2,2,2,2], num_classes, 16)

def ResNetGN818(num_classes=10):
    return ResNetGN(BasicBlockGN, [2,2,2,2], num_classes, 8)

# def DoubleResNet18(num_classes=10):
#     return DoubleResNet(BasicBlock, [2,2,2,2], num_classes)

# def DropResNet18(num_classes=10):
#     return ResNet(DropoutBasicBlock, [2,2,2,2], num_classes)
    
# def NSResNet18(num_classes=10):
#     return ResNet(NSBlock, [2,2,2,2], num_classes)

# def ResNet34(num_classes):
#     return ResNet(BasicBlock, [3,4,6,3], num_classes)

# def ResNet50(num_classes):
#     return ResNet(Bottleneck, [3,4,6,3], num_classes)

# def ResNet101(num_classes):
#     return ResNet(Bottleneck, [3,4,23,3], num_classes)

# def ResNet152(num_classes):
#     return ResNet(Bottleneck, [3,8,36,3], num_classes)

# def PreActResNet18(num_classes=10):
#     return ResNet(PreActBlock, [2,2,2,2], num_classes)

# test()

if __name__ == "__main__":
    model = ResNetLN18()
    print(model.bn1)