import torch
import torch.nn as nn


class CustomBatchNorm2d(nn.BatchNorm2d):
    def __init__(self, num_features, eps=1e-5, momentum=0.1,
                 affine=True, track_running_stats=True):
        super(CustomBatchNorm2d, self).__init__(
            num_features, eps, momentum, affine, track_running_stats)

    def forward(self, input):
        input, weight = input[0], input[1].detach()
        self._check_input_dim(input)

        exponential_average_factor = 0.0

        if self.training and self.track_running_stats:
            if self.num_batches_tracked is not None:
                self.num_batches_tracked += 1
                if self.momentum is None:  # use cumulative moving average
                    exponential_average_factor = 1.0 / float(self.num_batches_tracked)
                else:  # use exponential moving average
                    exponential_average_factor = self.momentum

        # calculate running estimates
        if self.training:
            # weight = torch.ones((input.size(0),), dtype=input.dtype, device=input.device) * 1. / input.size(0)
            with torch.no_grad():
                weight = weight / weight.sum()
            mean = (weight[:, None] * input.mean([2, 3])).sum([0])
            mean_x2 = (weight[:, None] * (input ** 2).mean([2, 3])).sum([0])
            var = mean_x2 - (mean ** 2)

            # mean = input.mean([0, 2, 3])
            # # use biased var in train
            # var = input.var([0, 2, 3], unbiased=False)

            n = input.numel() / input.size(1)
            with torch.no_grad():
                if self.track_running_stats:
                    self.running_mean = exponential_average_factor * mean \
                                        + (1 - exponential_average_factor) * self.running_mean
                    # update running_var with unbiased var
                    self.running_var = exponential_average_factor * var * n / (n - 1) \
                                       + (1 - exponential_average_factor) * self.running_var
        else:
            mean = self.running_mean
            var = self.running_var

        input = (input - mean[None, :, None, None]) / (torch.sqrt(var[None, :, None, None] + self.eps))
        if self.affine:
            input = input * self.weight[None, :, None, None] + self.bias[None, :, None, None]

        return input, weight
