# This code is modified by
# https://github.com/kuangliu/pytorch-cifar/blob/master/models/resnet.py
import torch
import torch.nn as nn
import torch.nn.functional as F
from models import CustomBatchNorm2d
from collections import OrderedDict


class CustomConv2d(nn.Conv2d):
    def __int__(self, in_channels, out_channels, kernel_size, stride=1, bias=True):
        super(CustomConv2d, self).__int__(in_channels, out_channels, kernel_size, stride, bias=bias)

    def forward(self, inputs):
        x, index = inputs[0], inputs[1]
        return super(CustomConv2d, self).forward(x), index


class CustomReLU(nn.ReLU):
    def __init__(self):
        super(CustomReLU, self).__init__()

    def forward(self, inputs):
        x, index = inputs[0], inputs[1]
        return super(CustomReLU, self).forward(x), index


class WBasicBlock(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, stride=1, norm_layer=None):
        super(WBasicBlock, self).__init__()
        self.conv1 = CustomConv2d(
            in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn1 = norm_layer(planes)
        self.conv2 = CustomConv2d(planes, planes, kernel_size=3,
                                  stride=1, padding=1, bias=False)
        self.bn2 = norm_layer(planes)
        self.relu = CustomReLU()

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(OrderedDict([
                ('conv', CustomConv2d(in_planes, self.expansion*planes,
                             kernel_size=1, stride=stride, bias=False)),
                ('bn',norm_layer(self.expansion*planes))
            ]))

    def forward(self, x):
        out = self.relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        out = (out[0] + self.shortcut(x)[0], out[1])
        out = self.relu(out)
        return out


class Bottleneck(nn.Module):
    expansion = 4

    def __init__(self, in_planes, planes, stride=1, norm_layer=None):
        super(Bottleneck, self).__init__()
        self.conv1 = CustomConv2d(in_planes, planes, kernel_size=1, bias=False)
        self.bn1 = norm_layer(planes)
        self.conv2 = CustomConv2d(planes, planes, kernel_size=3,
                                  stride=stride, padding=1, bias=False)
        self.bn2 = norm_layer(planes)
        self.conv3 = CustomConv2d(planes, self.expansion *
                                  planes, kernel_size=1, bias=False)
        self.bn3 = norm_layer(self.expansion*planes)
        self.relu = CustomReLU()

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(
                CustomConv2d(in_planes, self.expansion*planes,
                             kernel_size=1, stride=stride, bias=False),
                norm_layer(self.expansion*planes)
            )

    def forward(self, x):
        out = self.relu(self.bn1(self.conv1(x)))
        out = self.relu(self.bn2(self.conv2(out)))
        out = self.bn3(self.conv3(out))
        out = (out[0] + self.shortcut(x)[0], out[1])
        out = self.relu(out)
        return out


class WResNet(nn.Module):
    def __init__(self, block, num_blocks, num_classes=10):
        super(WResNet, self).__init__()
        self._norm_layer = CustomBatchNorm2d
        self.in_planes = 64

        self.conv1 = CustomConv2d(3, 64, kernel_size=3,
                                  stride=1, padding=1, bias=False)
        self.bn1 = self._norm_layer(64)
        self.layer1 = self._make_layer(block, 64, num_blocks[0], stride=1)
        self.layer2 = self._make_layer(block, 128, num_blocks[1], stride=2)
        self.layer3 = self._make_layer(block, 256, num_blocks[2], stride=2)
        self.layer4 = self._make_layer(block, 512, num_blocks[3], stride=2)
        self.linear = nn.Linear(512*block.expansion, num_classes)
        self.relu = CustomReLU()

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1]*(num_blocks-1)
        layers = []
        for stride in strides:
            layers.append(block(self.in_planes, planes, stride, self._norm_layer))
            self.in_planes = planes * block.expansion
        return nn.Sequential(*layers)

    def forward(self, x, mask=None):
        if mask is None:
            mask = torch.zeros(len(x), dtype=torch.long).to(x.device)
        mask = 1. - mask
        inputs = (x, mask)
        out = self.relu(self.bn1(self.conv1(inputs)))
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.layer4(out)
        out = F.avg_pool2d(out[0], 4)
        out = out.view(out.size(0), -1)
        out = self.linear(out)
        return out


def WResNet18(num_classes=10):
    return WResNet(WBasicBlock, [2, 2, 2, 2], num_classes)


# def weighted_resnet34(num_classes=10):
#     return ResNet(WBasicBlock, [3, 4, 6, 3], num_classes)


# def weighted_resnet50(num_classes=10):
#     return ResNet(Bottleneck, [3, 4, 6, 3], num_classes)


# def weighted_resnet101(num_classes=10):
#     return ResNet(Bottleneck, [3, 4, 23, 3], num_classes)


# def weighted_resnet152(num_classes=10):
#     return ResNet(Bottleneck, [3, 8, 36, 3], num_classes)


def test():
    net = WResNet18()
    y = net(torch.randn(1, 3, 32, 32))
    print(y.size())

# test()