from abc import ABC, abstractmethod
import pandas as pd
import numpy as np

class BaseColumnTransformer(ABC):
    def fit(self, X):
        """
        Optional fit method to fit the transformer to the data.
        By default, it does nothing.
        """
        return self

    def transform(self, X):
        """
        Applies the transformation by calling the _transform method.
        """
        return self._transform(X)

    def inverse_transform(self, X):
        """
        Applies the inverse transformation by calling the _inverse_transform method.
        """
        return self._inverse_transform(X)

    def set_fit(self, config):
        """
        Optional method to set fitted parameters from a config.
        By default, it does nothing.

        Args:
            config (dict): Configuration containing fitted parameters
        """
        return self

    @abstractmethod
    def _transform(self, X):
        """
        Abstract method to define how the transformation is applied.
        Must be implemented by the child class.
        """
        pass

    @abstractmethod
    def _inverse_transform(self, X):
        """
        Abstract method to define how the inverse transformation is applied.
        Must be implemented by the child class.
        """
        pass