# AutoSAT - Adaptive SAT Solver Framework

## Overview

AutoSAT is an adaptive SAT solver framework that automatically selects the best heuristic combination for each SAT instance based on instance features. The framework implements the DASHCO (Data-driven Adaptive SAT Heuristic Combination Optimization) approach.

## Core Components

### 1. Feature Extraction (`autosat/feature_extraction/`)

- **CompactSATFeatureExtractor**: Extracts 21 independent features from SAT instances
- **BatchFeatureExtractor**: Processes large-scale datasets efficiently
- **GlobalFeatureNormalizer**: Normalizes features across datasets

### 2. Adaptive Solver Selection (`adaptive_solver_selection.py`)

- Implements instance space partitioning via performance-based clustering
- Finds optimal heuristic combinations for each instance cluster
- Provides adaptive selection for new instances

### 3. Evaluation Framework

- **`evaluate.py`**: Base evaluation tool for individual solvers
- **`evaluate_combinations.py`**: Evaluates multiple solver combinations
- **`adaptive_eval.py`**: Adaptive evaluation using trained models

### 4. Heuristic Combination Generation (`generate_combinations.py`)

- Generates solver combinations from LLM-generated heuristics
- Creates C++ solver files with different restart conditions, bump functions, and rephase strategies

### 5. Multi-Agent Training (`main_MultiAgent.py`)

- Implements the multi-agent training framework
- Trains agents to generate effective heuristic combinations

## Usage

### 1. Feature Extraction

```bash
# Extract features from a dataset
python extract_all_features.sh

# Or use the Python script directly
python -c "from autosat.feature_extraction import BatchFeatureExtractor; BatchFeatureExtractor().extract_all_features()"
```

### 2. Generate Solver Combinations

```bash
# Generate combinations from LLM results
python generate_combinations.py --results-dir ./llm_results --template-path ./solver_template.cpp --output-dir ./combinations
```

### 3. Evaluate Combinations

```bash
# Evaluate all combinations
python evaluate_combinations.py --combinations-dir ./combinations --dataset-dir ./dataset/test --results-dir ./eval_results
```

### 4. Train Adaptive Selector

```bash
# Train the adaptive solver selector
python adaptive_solver_selection.py
```

### 5. Adaptive Evaluation

```bash
# Run adaptive evaluation
python adaptive_eval.py --eval_data_dir ./dataset/test --results_save_path ./adaptive_results --model_path ./adaptive_selector_train_new_fixed.pkl
```

### 6. Multi-Agent Training

```bash
# Run multi-agent training
python main_MultiAgent.py --config config.yaml
```

## Project Structure

```
autosat_clean/
├── autosat/                          # Core library
│   ├── feature_extraction/           # Feature extraction modules
│   ├── heuristic_combination/        # Heuristic combination tools
│   ├── evaluation/                   # Evaluation utilities
│   ├── execution/                    # Execution framework
│   ├── llm_api/                     # LLM API integration
│   ├── trainer/                      # Training modules
│   └── agent/                       # Multi-agent components
├── combinations/                     # Generated solver combinations
├── adaptive_eval.py                 # Adaptive evaluation script
├── adaptive_solver_selection.py     # Adaptive selector training
├── evaluate_combinations.py         # Combination evaluation
├── generate_combinations.py         # Combination generation
├── evaluate.py                      # Base evaluation tool
├── main_MultiAgent.py              # Multi-agent training
├── requirements.txt                 # Python dependencies
├── setup.py                        # Package setup
└── *.sh                           # Utility scripts
```

## Installation

```bash
pip install -r requirements.txt
```

## Key Features

1. **Adaptive Selection**: Automatically selects the best solver for each instance
2. **Feature-Based**: Uses 21 compact, non-redundant features
3. **Multi-Agent**: LLM-based heuristic generation
4. **Scalable**: Batch processing for large datasets
5. **Reproducible**: Complete evaluation and training pipeline

## Dependencies

- Python 3.8+
- NumPy, Pandas, Scikit-learn
- PyTorch (for multi-agent training)
- SAT solver compilation tools (gcc, make)

## License

This project is part of the DaSATcho framework.