#!/usr/bin/env python3
"""
Evaluate all generated solver combinations and analyze their performance.
This script directly uses evaluate.py for evaluation and adds analysis capabilities.
"""

import os
import sys
import argparse
import subprocess
from datetime import datetime
from collections import defaultdict
import json
import csv
import glob

def evaluate_solver_using_evaluate_py(solver_path, args):
    """Evaluate a single solver using the existing evaluate.py with organized directory structure"""
    print(f"Evaluating solver: {os.path.basename(solver_path)}")
    
    solver_name = os.path.basename(solver_path).replace('.cpp', '')
    
    # Copy necessary header files to the same directory as the solver
    solver_dir = os.path.dirname(solver_path)
    original_dir = './examples/EasySAT/original_EasySAT'
    for file in ['EasySAT.hpp', 'heap.hpp']:
        src_file = os.path.join(original_dir, file)
        if os.path.exists(src_file):
            dst_file = os.path.join(solver_dir, file)
            if not os.path.exists(dst_file):
                import shutil
                shutil.copy2(src_file, dst_file)
                print(f"  Copied {file} to solver directory")
    
    # Use organized directory structure: output_dir/dataset_name/method_name/
    dataset_name = os.path.basename(os.path.normpath(args.eval_data_dir))
    print(f"  Results will be saved to: {args.output_dir}/{dataset_name}/{solver_name}/")
    
    # Run evaluation using evaluate.py directly with organized structure
    cmd = [
        'python', 'evaluate.py',
        '--SAT_solver_file_path', solver_path,
        '--eval_data_dir', args.eval_data_dir,
        '--results_save_path', args.output_dir,  # Use the base results path
        '--eval_parallel_size', str(args.eval_parallel_size),
        '--eval_timeout', str(args.eval_timeout),
        '--method_name', solver_name,
        '--keep_intermediate_results', 'True'
        # No --config parameter - let evaluate.py use its default behavior with command line args
    ]
    
    try:
        print(f"  Running: {' '.join(cmd)}")
        # Use Popen for real-time output with timeout
        process = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, 
                                  text=True, bufsize=1, universal_newlines=True)
        
        # Print output in real-time with timeout
        try:
            for line in process.stdout:
                print(f"    {line.rstrip()}")
        except KeyboardInterrupt:
            print("  Process interrupted by user")
            process.terminate()
            return None
        
        # Wait for completion with timeout
        try:
            return_code = process.wait(timeout=args.external_timeout)
        except subprocess.TimeoutExpired:
            print(f"  Timeout after {args.external_timeout}s, terminating process")
            process.kill()
            return None
        
        if return_code != 0:
            print(f"  Error: Process returned code {return_code}")
            return None
        
        print(f"  Evaluation completed successfully")
        # Return the evaluation directory path using organized structure
        eval_session_dir = os.path.join(args.output_dir, dataset_name, solver_name)
        return eval_session_dir
        
    except subprocess.TimeoutExpired:
        print(f"  Timeout during evaluation")
        process.kill()
        return None
    except Exception as e:
        print(f"  Exception: {str(e)}")
        return None

def find_results_file(eval_dir, solver_name):
    """Find the results file generated by evaluate.py in organized directory structure"""
    if not eval_dir or not os.path.exists(eval_dir):
        return None
    
    # Look for results files with multiple possible patterns
    search_patterns = [
        os.path.join(eval_dir, 'results_*.txt'),           # Direct in eval_dir
        os.path.join(eval_dir, '*.txt'),                   # Any txt file in eval_dir
        os.path.join(eval_dir, '**', 'results_*.txt'),     # Recursive search
    ]
    
    for pattern in search_patterns:
        import glob
        results_files = glob.glob(pattern, recursive=True)
        if results_files:
            # Return the most recent one
            most_recent = max(results_files, key=os.path.getctime)
            print(f"  Found results file: {most_recent}")
            return most_recent
    
    # If no results_*.txt found, try to check if evaluation completed
    print(f"  Searching in directory: {eval_dir}")
    all_files = []
    try:
        for root, dirs, files in os.walk(eval_dir):
            for file in files:
                if file.endswith('.txt'):
                    all_files.append(os.path.join(root, file))
        print(f"  Found txt files: {all_files[:5]}...")  # Show first 5
    except Exception as e:
        print(f"  Error walking directory: {e}")
    
    print(f"  No results file found in {eval_dir}")
    return None

def parse_results_file(results_file):
    """Parse the results file and return instance-level results"""
    instance_results = {}
    
    if not os.path.exists(results_file):
        print(f"  Results file not found: {results_file}")
        return instance_results
    
    with open(results_file, 'r') as f:
        lines = f.readlines()
    
    # Skip header lines
    for line in lines:
        if line.startswith('File name') or line.startswith('{'):
            continue
        if line.strip() == '':
            continue
        
        parts = line.strip().split('\t')
        if len(parts) >= 3:
            cnf_file = os.path.basename(parts[0])  # Extract just the filename
            try:
                duration = float(parts[1])
            except ValueError:
                duration = float('inf')  # Handle timeout cases
            situation = parts[2].lower()
            
            # Convert timeout to a large number for comparison
            if situation == 'timeout':
                duration = float('inf')
            
            instance_results[cnf_file] = {
                'duration': duration,
                'situation': situation
            }
    
    return instance_results

def get_solver_strategy_info(solver_name):
    """Get strategy composition information for a solver based on actual configuration"""
    # Extract combination number from solver name
    if 'solver_combination_' in solver_name:
        combo_num = solver_name.replace('solver_combination_', '')
        try:
            combo_num = int(combo_num)
            
            # Read configuration from combination_config.sh
            import subprocess
            result = subprocess.run(['bash', '-c', 'source combination_config.sh > /dev/null && echo "$SAMPLE_RESTART:$SAMPLE_BUMP_VAR:$SAMPLE_REPHASE"'], 
                                  capture_output=True, text=True, cwd='.')
            if result.returncode != 0:
                raise RuntimeError(f"Failed to read combination_config.sh: {result.stderr}")
            
            restart_count, bump_count, rephase_count = map(int, result.stdout.strip().split(':'))
            
            # Generate strategy names based on configuration
            restart_strategies = ['baseline'] + [f'top{i}' for i in range(1, restart_count)]
            bump_strategies = ['baseline'] + [f'top{i}' for i in range(1, bump_count)]
            rephase_strategies = ['baseline'] + [f'top{i}' for i in range(1, rephase_count)]
            
            # Generate all combinations in the same order as generate_combinations.py
            strategies = {}
            combination_index = 0
            
            from itertools import product
            for restart_strat in restart_strategies:
                for bump_strat in bump_strategies:
                    for rephase_strat in rephase_strategies:
                        strategies[combination_index] = f"restart:{restart_strat}, bump:{bump_strat}, rephase:{rephase_strat}"
                        combination_index += 1
            
            return strategies.get(combo_num, f"combination_{combo_num}")
        except ValueError:
            return "unknown_strategy"
    return "unknown_strategy"

def analyze_instance_performance(all_results):
    """Analyze performance of each instance across all solvers"""
    instance_analysis = {}
    
    # Get all unique instances
    all_instances = set()
    for solver_name, results in all_results.items():
        all_instances.update(results.keys())
    
    # Analyze each instance
    for instance in all_instances:
        instance_data = {
            'solver_rankings': [],
            'best_solver': None,
            'best_time': float('inf'),
            'worst_solver': None,
            'worst_time': 0,
            'avg_time': 0,
            'solved_count': 0,
            'timeout_count': 0,
            'total_solvers': len(all_results)
        }
        
        # Collect all solver results for this instance
        solver_times = []
        for solver_name, results in all_results.items():
            if instance in results:
                duration = results[instance]['duration']
                situation = results[instance]['situation']
                
                solver_times.append({
                    'solver': solver_name,
                    'time': duration,
                    'situation': situation
                })
                
                if situation != 'timeout':
                    instance_data['solved_count'] += 1
                else:
                    instance_data['timeout_count'] += 1
                
                if duration < instance_data['best_time']:
                    instance_data['best_time'] = duration
                    instance_data['best_solver'] = solver_name
                
                if duration > instance_data['worst_time']:
                    instance_data['worst_time'] = duration
                    instance_data['worst_solver'] = solver_name
        
        # Sort by time and create rankings
        solver_times.sort(key=lambda x: x['time'])
        for rank, solver_data in enumerate(solver_times, 1):
            solver_data['rank'] = rank
            instance_data['solver_rankings'].append(solver_data)
        
        # Calculate average time (excluding timeouts)
        solved_times = [s['time'] for s in solver_times if s['situation'] != 'timeout']
        if solved_times:
            instance_data['avg_time'] = sum(solved_times) / len(solved_times)
        
        instance_analysis[instance] = instance_data
    
    return instance_analysis

def analyze_solver_performance(all_results):
    """Analyze overall performance of each solver"""
    solver_analysis = {}
    
    for solver_name, results in all_results.items():
        solver_data = {
            'total_instances': len(results),
            'solved_instances': 0,
            'timeout_instances': 0,
            'total_time': 0,
            'avg_time': 0,
            'best_time': float('inf'),
            'worst_time': 0,
            'instance_wins': 0,
            'instance_rankings': [],
            'strategy': get_solver_strategy_info(solver_name)
        }
        
        solved_times = []
        for instance, result in results.items():
            duration = result['duration']
            situation = result['situation']
            
            if situation != 'timeout':
                solver_data['solved_instances'] += 1
                solved_times.append(duration)
                solver_data['total_time'] += duration
                
                if duration < solver_data['best_time']:
                    solver_data['best_time'] = duration
                
                if duration > solver_data['worst_time']:
                    solver_data['worst_time'] = duration
            else:
                solver_data['timeout_instances'] += 1
        
        if solved_times:
            solver_data['avg_time'] = sum(solved_times) / len(solved_times)
        
        solver_analysis[solver_name] = solver_data
    
    return solver_analysis

def find_best_solver_per_instance(all_results):
    """Find the best solver for each instance"""
    instance_best = {}
    solver_stats = defaultdict(lambda: {'wins': 0, 'total_time': 0, 'solved': 0})
    
    # Get all unique instances
    all_instances = set()
    for solver_name, results in all_results.items():
        all_instances.update(results.keys())
    
    # For each instance, find the best solver
    for instance in all_instances:
        best_solver = None
        best_duration = float('inf')
        
        for solver_name, results in all_results.items():
            if instance in results:
                duration = results[instance]['duration']
                if duration < best_duration:
                    best_duration = duration
                    best_solver = solver_name
        
        if best_solver:
            instance_best[instance] = best_solver
            solver_stats[best_solver]['wins'] += 1
            solver_stats[best_solver]['total_time'] += best_duration
            if best_duration != float('inf'):
                solver_stats[best_solver]['solved'] += 1
    
    return instance_best, solver_stats

def create_detailed_ranking_report(instance_analysis, solver_analysis, output_dir):
    """Create detailed ranking reports"""
    
    # 1. Instance-level ranking report
    with open(os.path.join(output_dir, 'instance_rankings.csv'), 'w', newline='') as f:
        writer = csv.writer(f)
        writer.writerow(['Instance', 'Best_Solver', 'Best_Time', 'Worst_Solver', 'Worst_Time', 
                        'Avg_Time', 'Solved_Count', 'Timeout_Count', 'Total_Solvers'])
        
        for instance, data in sorted(instance_analysis.items()):
            writer.writerow([
                instance,
                data['best_solver'],
                data['best_time'] if data['best_time'] != float('inf') else 'timeout',
                data['worst_solver'],
                data['worst_time'] if data['worst_time'] != float('inf') else 'timeout',
                f"{data['avg_time']:.2f}" if data['avg_time'] > 0 else 'N/A',
                data['solved_count'],
                data['timeout_count'],
                data['total_solvers']
            ])
    
    # 2. Solver-level ranking report
    with open(os.path.join(output_dir, 'solver_rankings.csv'), 'w', newline='') as f:
        writer = csv.writer(f)
        writer.writerow(['Solver', 'Strategy', 'Total_Instances', 'Solved_Instances', 'Timeout_Instances',
                        'Success_Rate', 'Avg_Time', 'Best_Time', 'Worst_Time', 'Total_Time'])
        
        for solver_name, data in sorted(solver_analysis.items()):
            success_rate = (data['solved_instances'] / data['total_instances']) * 100
            writer.writerow([
                solver_name,
                data['strategy'],
                data['total_instances'],
                data['solved_instances'],
                data['timeout_instances'],
                f"{success_rate:.1f}%",
                f"{data['avg_time']:.2f}" if data['avg_time'] > 0 else 'N/A',
                data['best_time'] if data['best_time'] != float('inf') else 'N/A',
                data['worst_time'] if data['worst_time'] != float('inf') else 'N/A',
                f"{data['total_time']:.2f}" if data['total_time'] > 0 else 'N/A'
            ])
    
    # 3. Detailed solver rankings per instance
    with open(os.path.join(output_dir, 'detailed_instance_rankings.csv'), 'w', newline='') as f:
        writer = csv.writer(f)
        writer.writerow(['Instance', 'Rank', 'Solver', 'Time', 'Situation', 'Strategy'])
        
        for instance, data in sorted(instance_analysis.items()):
            for ranking in data['solver_rankings']:
                strategy = get_solver_strategy_info(ranking['solver'])
                time_str = f"{ranking['time']:.2f}" if ranking['time'] != float('inf') else 'timeout'
                writer.writerow([
                    instance,
                    ranking['rank'],
                    ranking['solver'],
                    time_str,
                    ranking['situation'],
                    strategy
                ])
    
    # 4. Performance summary report
    with open(os.path.join(output_dir, 'performance_summary.txt'), 'w') as f:
        f.write("=" * 60 + "\n")
        f.write("SOLVER COMBINATION EVALUATION SUMMARY\n")
        f.write("=" * 60 + "\n\n")
        
        # Overall statistics
        total_instances = len(instance_analysis)
        total_solvers = len(solver_analysis)
        f.write(f"Total Instances: {total_instances}\n")
        f.write(f"Total Solvers: {total_solvers}\n\n")
        
        # Best performing solvers
        f.write("TOP PERFORMING SOLVERS (by solved instances):\n")
        f.write("-" * 50 + "\n")
        sorted_solvers = sorted(solver_analysis.items(), 
                               key=lambda x: x[1]['solved_instances'], reverse=True)
        for i, (solver_name, data) in enumerate(sorted_solvers[:5], 1):
            success_rate = (data['solved_instances'] / data['total_instances']) * 100
            f.write(f"{i}. {solver_name} ({data['strategy']})\n")
            f.write(f"   Solved: {data['solved_instances']}/{data['total_instances']} ({success_rate:.1f}%)\n")
            f.write(f"   Avg Time: {data['avg_time']:.2f}s\n\n")
        
        # Instance difficulty analysis
        f.write("INSTANCE DIFFICULTY ANALYSIS:\n")
        f.write("-" * 50 + "\n")
        easy_instances = [inst for inst, data in instance_analysis.items() 
                         if data['solved_count'] == data['total_solvers']]
        hard_instances = [inst for inst, data in instance_analysis.items() 
                         if data['solved_count'] == 0]
        medium_instances = [inst for inst, data in instance_analysis.items() 
                           if 0 < data['solved_count'] < data['total_solvers']]
        
        f.write(f"Easy instances (all solvers solved): {len(easy_instances)}\n")
        f.write(f"Medium instances (some solvers solved): {len(medium_instances)}\n")
        f.write(f"Hard instances (no solver solved): {len(hard_instances)}\n\n")
        
        # Strategy effectiveness
        f.write("STRATEGY EFFECTIVENESS:\n")
        f.write("-" * 50 + "\n")
        strategy_stats = defaultdict(lambda: {'total_solved': 0, 'total_instances': 0})
        for solver_name, data in solver_analysis.items():
            strategy = data['strategy']
            strategy_stats[strategy]['total_solved'] += data['solved_instances']
            strategy_stats[strategy]['total_instances'] += data['total_instances']
        
        for strategy, stats in strategy_stats.items():
            success_rate = (stats['total_solved'] / stats['total_instances']) * 100
            f.write(f"{strategy}: {stats['total_solved']}/{stats['total_instances']} ({success_rate:.1f}%)\n")

def main():
    parser = argparse.ArgumentParser(description='Evaluate solver combinations using evaluate.py and analyze results')
    parser.add_argument('--combinations_dir', type=str, default='./combinations',
                       help='Directory containing the generated solver combinations')
    parser.add_argument('--eval_data_dir', type=str, default='./dataset/train',
                       help='Directory containing CNF files for evaluation')
    parser.add_argument('--eval_parallel_size', type=int, default=16,
                       help='Number of parallel processes for evaluation')
    parser.add_argument('--eval_timeout', type=int, default=60,
                       help='Timeout for each instance evaluation (seconds)')
    parser.add_argument('--external_timeout', type=int, default=3600,
                       help='External timeout for the entire evaluation process (seconds)')
    parser.add_argument('--max_combinations', type=int, default=3,
                       help='Maximum number of combinations to evaluate (for testing)')
    parser.add_argument('--keep_temp_files', action='store_true', default=False,
                       help='Keep temporary files after evaluation (default: False)')
    parser.add_argument('--output_dir', type=str, default='./evaluation_results',
                       help='Directory to save evaluation results')
    parser.add_argument('--report_dir', type=str, default='./combination_reports',
                       help='Directory to save independent combination analysis reports')
    parser.add_argument('--start_from_combination', type=int, default=0,
                       help='Start evaluation from this combination number (0-based)')
    
    args = parser.parse_args()
    
    # Create output directories
    os.makedirs(args.output_dir, exist_ok=True)
    os.makedirs(args.report_dir, exist_ok=True)
    
    # Find all solver combination files
    solver_files = []
    for file in os.listdir(args.combinations_dir):
        if file.startswith('solver_combination_') and file.endswith('.cpp'):
            solver_files.append(os.path.join(args.combinations_dir, file))
    
    solver_files.sort()  # Ensure consistent ordering
    
    # Start from specified combination if specified
    if args.start_from_combination > 0:
        start_index = args.start_from_combination
        if start_index < len(solver_files):
            solver_files = solver_files[start_index:]
            print(f"Starting from combination {start_index}: {len(solver_files)} combinations remaining")
        else:
            print(f"Warning: start_from_combination {start_index} is beyond available combinations")
            solver_files = []
    
    # Limit the number of combinations if specified
    if args.max_combinations:
        solver_files = solver_files[:args.max_combinations]
        print(f"Limited to first {args.max_combinations} combinations for testing")
    
    print(f"Found {len(solver_files)} solver combinations to evaluate")
    
    # Evaluate each solver using evaluate.py
    all_results = {}
    failed_solvers = []
    temp_dirs = []  # Track temporary directories for cleanup
    
    for solver_file in solver_files:
        solver_name = os.path.basename(solver_file).replace('.cpp', '')
        print(f"\n{'='*50}")
        print(f"Evaluating {solver_name}")
        print(f"{'='*50}")
        
        try:
            print(f"Starting evaluation for {solver_name}...")
            eval_dir = evaluate_solver_using_evaluate_py(solver_file, args)
            print(f"Evaluation completed for {solver_name}, eval_dir: {eval_dir}")
            
            if eval_dir:
                temp_dirs.append(eval_dir)  # Track for cleanup
                print(f"Looking for results file in: {eval_dir}")
                results_file = find_results_file(eval_dir, solver_name)
                print(f"Results file found: {results_file}")
                
                if results_file:
                    print(f"Parsing results file: {results_file}")
                    instance_results = parse_results_file(results_file)
                    all_results[solver_name] = instance_results
                    print(f"Successfully evaluated {solver_name}: {len(instance_results)} instances")
                else:
                    print(f"No results file found for {solver_name}")
                    failed_solvers.append(solver_name)
            else:
                print(f"Failed to evaluate {solver_name}")
                failed_solvers.append(solver_name)
        except Exception as e:
            print(f"Error evaluating {solver_name}: {str(e)}")
            failed_solvers.append(solver_name)
            continue
    
    # Clean up temporary files unless --keep_temp_files is specified
    if not args.keep_temp_files and temp_dirs:
        print(f"\nCleaning up {len(temp_dirs)} temporary directories...")
        for temp_dir in temp_dirs:
            try:
                import shutil
                shutil.rmtree(temp_dir)
                print(f"  Removed: {temp_dir}")
            except Exception as e:
                print(f"  Warning: Could not remove {temp_dir}: {e}")
    
    if failed_solvers:
        print(f"\nWarning: {len(failed_solvers)} solvers failed to evaluate:")
        for solver in failed_solvers:
            print(f"  - {solver}")
    
    print(f"\nEvaluation Summary:")
    print(f"  Total solvers processed: {len(solver_files)}")
    print(f"  Successful evaluations: {len(all_results)}")
    print(f"  Failed evaluations: {len(failed_solvers)}")
    print(f"  All results keys: {list(all_results.keys())}")
    
    if not all_results:
        print("No successful evaluations!")
        return
    
    # Perform detailed analysis
    print(f"\n{'='*50}")
    print("Performing detailed performance analysis...")
    print(f"{'='*50}")
    
    print("Step 1: Analyzing instance performance...")
    instance_analysis = analyze_instance_performance(all_results)
    print(f"  Completed: {len(instance_analysis)} instances analyzed")
    
    print("Step 2: Analyzing solver performance...")
    solver_analysis = analyze_solver_performance(all_results)
    print(f"  Completed: {len(solver_analysis)} solvers analyzed")
    
    # Find best solver per instance
    instance_best, solver_stats = find_best_solver_per_instance(all_results)
    
    # Create detailed reports in the independent report directory
    create_detailed_ranking_report(instance_analysis, solver_analysis, args.report_dir)
    
    # Print summary results
    print(f"\n{'='*50}")
    print("Solver Strategy Composition")
    print(f"{'='*50}")
    for solver_name in sorted(all_results.keys()):
        strategy = get_solver_strategy_info(solver_name)
        stats = solver_stats[solver_name]
        print(f"{solver_name}: {strategy}")
        print(f"  Wins: {stats['wins']}, Solved: {stats['solved']}, Avg time: {stats['total_time']/stats['wins']:.2f}s" if stats['wins'] > 0 else "  No wins")
    
    print(f"\n{'='*50}")
    print("Best solver per instance (sorted by wins)")
    print(f"{'='*50}")
    # Sort solvers by number of wins
    sorted_solvers = sorted(solver_stats.items(), key=lambda x: x[1]['wins'], reverse=True)
    for solver_name, stats in sorted_solvers:
        strategy = get_solver_strategy_info(solver_name)
        print(f"{solver_name}: {strategy} - {stats['wins']} wins")
    
    # Generate timestamp for this evaluation run
    formatted_date_time = datetime.now().strftime("%Y%m%d_%H%M%S")
    dataset_name = os.path.basename(os.path.normpath(args.eval_data_dir))
    
    # Save detailed results for independent analysis
    results_summary = {
        'instance_best': instance_best,
        'solver_stats': dict(solver_stats),
        'strategy_info': {solver: get_solver_strategy_info(solver) for solver in all_results.keys()},
        'instance_analysis': instance_analysis,
        'solver_analysis': solver_analysis,
        'evaluation_timestamp': formatted_date_time,
        'dataset': dataset_name,
        'individual_results_location': args.output_dir,
        'total_combinations_evaluated': len(all_results),
        'failed_solvers': failed_solvers if 'failed_solvers' in locals() else []
    }
    
    # Save independent analysis reports
    with open(os.path.join(args.report_dir, f'combination_analysis_{dataset_name}_{formatted_date_time}.json'), 'w') as f:
        json.dump(results_summary, f, indent=2)
    
    # Save CSV for easy analysis
    with open(os.path.join(args.report_dir, f'best_combinations_{dataset_name}_{formatted_date_time}.csv'), 'w', newline='') as f:
        writer = csv.writer(f)
        writer.writerow(['Instance', 'Best_Solver', 'Strategy'])
        for instance, best_solver in sorted(instance_best.items()):
            strategy = get_solver_strategy_info(best_solver)
            writer.writerow([instance, best_solver, strategy])
    
    print(f"\n📊 Individual evaluation results saved to: {args.output_dir}")
    print(f"   └── Organized by {dataset_name}/[method_name]/ structure for each combination")
    print(f"\n📈 Independent analysis reports saved to: {args.report_dir}")
    print("Generated analysis files:")
    print("  - instance_rankings.csv: Ranking of solvers for each instance")
    print("  - solver_rankings.csv: Overall performance of each solver") 
    print("  - detailed_instance_rankings.csv: Complete ranking details per instance")
    print("  - performance_summary.txt: Human-readable performance summary")
    print(f"  - combination_analysis_{dataset_name}_{formatted_date_time}.json: Complete analysis data")
    print(f"  - best_combinations_{dataset_name}_{formatted_date_time}.csv: Best solver for each instance")

if __name__ == '__main__':
    main() 