#!/bin/bash

# Complete workflow: Generate combinations -> Evaluate -> Analyze
# This script automates the entire heuristic combination evaluation process

# ================================================
# Load Configuration
# ================================================
CONFIG_FILE="combination_config.sh"

# Load configuration if exists, otherwise use defaults
if [ -f "$CONFIG_FILE" ]; then
    echo "📋 Loading configuration from $CONFIG_FILE..."
    source "$CONFIG_FILE"
else
    echo "📋 Using default configuration (create $CONFIG_FILE to customize)..."
    # Default configuration
    COMBINATIONS_DIR="./combinations"             # Output directory for generated combinations
    EVAL_DATA_DIR="./dataset/train"              # Dataset for evaluation
    EVAL_RESULTS_DIR="./eval_results"            # Individual evaluation results
    REPORT_DIR="./combination_reports"           # Analysis reports
    EVAL_PARALLEL_SIZE=16                        # Parallel processes for evaluation
    EVAL_TIMEOUT=60                              # Timeout per instance (seconds)
    MAX_COMBINATIONS=8                           # Maximum combinations to evaluate
    KEEP_TEMP_FILES=true                         # Keep temporary files
    FORCE_REGENERATE=false                       # Force regeneration of combinations
    SAMPLE_RESTART=2                             # Default restart samples
    SAMPLE_BUMP_VAR=2                            # Default bump_var samples
    SAMPLE_REPHASE=2                             # Default rephase samples
fi

# ================================================
# Print Configuration
# ================================================
echo "=================================================="
echo "🚀 AutoSAT Combination Generation & Evaluation"
echo "=================================================="
echo "Search results dir: $RESULTS_JSON_DIR"
echo "Combinations dir: $COMBINATIONS_DIR"
echo "Evaluation data: $EVAL_DATA_DIR"
echo "Results dir: $EVAL_RESULTS_DIR"
echo "Reports dir: $REPORT_DIR"
echo "Parallel size: $EVAL_PARALLEL_SIZE"
echo "Timeout: $EVAL_TIMEOUT seconds"
echo "Max combinations: $MAX_COMBINATIONS"
echo "Sample counts: restart=$SAMPLE_RESTART, bump_var=$SAMPLE_BUMP_VAR, rephase=$SAMPLE_REPHASE"
echo "=================================================="

# ================================================
# Step 1: Auto-detect Training Folders & Generate Combinations
# ================================================
echo ""
echo "📝 Step 1: Auto-detecting training folders and generating combinations..."
echo "=================================================="

# Auto-detect the latest training folders for different tasks
echo "🔍 Auto-detecting training folders for different tasks..."

# Look for task-specific training directories
TASK_FOLDERS=()
TASK_TYPES=("restart_condition" "bump_var_function" "rephase_function")

for task in "${TASK_TYPES[@]}"; do
    echo "  Looking for $task training folders..."
    
    # Find directories matching task patterns (back to working version)
    MATCHING_DIRS=$(find . -maxdepth 1 -type d \( -name "${task}_results_d*" -o -name "${task}_search_d*" -o -name "${task}_session_d*" \) 2>/dev/null | sort -r)
    
    if [ -n "$MATCHING_DIRS" ]; then
        # Get the latest directory
        LATEST_DIR=$(echo "$MATCHING_DIRS" | head -1)
        
        # Check if it has top_k_results (required by generate_combinations.py)
        if [ -d "$LATEST_DIR/top_k_results" ]; then
            TASK_FOLDERS+=("$LATEST_DIR")
            echo "    ✓ Found: $LATEST_DIR"
        else
            echo "    ⚠️  Found $LATEST_DIR but missing top_k_results/"
        fi
    else
        echo "    ❌ No training folder found for $task"
    fi
done

# Check if we have at least some training folders
if [ ${#TASK_FOLDERS[@]} -eq 0 ]; then
    echo ""
    echo "❌ Error: No valid training folders found!"
    echo "   Expected patterns: *_results_d*, *_search_d*, or *_session_d*"
    echo "   Each folder should contain top_k_results/ subdirectory"
    echo ""
    echo "   Available directories:"
    ls -d *_results_d* *_search_d* *_session_d* 2>/dev/null || echo "   (none found)"
    echo ""
    echo "   Please run training first for the required tasks:"
    for task in "${TASK_TYPES[@]}"; do
        echo "   - $task"
    done
    exit 1
fi

echo ""
echo "✅ Found ${#TASK_FOLDERS[@]} valid training folders:"
for folder in "${TASK_FOLDERS[@]}"; do
    echo "  📂 $folder"
done

# Check if combinations already exist
if [ "$FORCE_REGENERATE" = true ] || [ ! -d "$COMBINATIONS_DIR" ] || [ -z "$(ls -A "$COMBINATIONS_DIR"/*.cpp 2>/dev/null)" ]; then
    echo ""
    echo "🔧 Generating new combinations..."
    
    # Prepare search_folders argument for generate_combinations.py
    SEARCH_FOLDERS_ARG=""
    for folder in "${TASK_FOLDERS[@]}"; do
        SEARCH_FOLDERS_ARG="$SEARCH_FOLDERS_ARG $folder"
    done
    
    # Generate combinations using the detected folders
    echo "  Running: python generate_combinations.py --search_folders$SEARCH_FOLDERS_ARG --output_dir $COMBINATIONS_DIR"
    
    if [ ${#TASK_FOLDERS[@]} -eq 3 ]; then
        # We have all three task types, use specific folders
        python generate_combinations.py \
            --search_folders $SEARCH_FOLDERS_ARG \
            --output_dir "$COMBINATIONS_DIR" \
            --sample_restart "$SAMPLE_RESTART" \
            --sample_bump_var "$SAMPLE_BUMP_VAR" \
            --sample_rephase "$SAMPLE_REPHASE"
    else
        # Let generate_combinations.py auto-detect (it might find more folders)
        echo "  Only found ${#TASK_FOLDERS[@]}/3 task types, letting generate_combinations.py auto-detect..."
        python generate_combinations.py \
            --output_dir "$COMBINATIONS_DIR" \
            --sample_restart "$SAMPLE_RESTART" \
            --sample_bump_var "$SAMPLE_BUMP_VAR" \
            --sample_rephase "$SAMPLE_REPHASE"
    fi

    if [ $? -ne 0 ]; then
        echo "❌ Error: Failed to generate combinations"
        echo "   This might happen if:"
        echo "   - Training results are incomplete"
        echo "   - Missing top_k_results in training folders"
        echo "   - Not all required task types have been trained"
        exit 1
    fi
    
    echo "✅ Successfully generated combinations in $COMBINATIONS_DIR"
else
    echo ""
    echo "✓ Using existing combinations in $COMBINATIONS_DIR"
    echo "  (Set FORCE_REGENERATE=true in config to regenerate)"
fi

# ================================================
# Step 2: Evaluate Combinations
# ================================================
echo ""
echo "🔬 Step 2: Evaluating solver combinations..."
echo "=================================================="

# Create output directories
mkdir -p "$EVAL_RESULTS_DIR"
mkdir -p "$REPORT_DIR"

# Check if combinations exist
if [ ! -d "$COMBINATIONS_DIR" ] || [ -z "$(ls -A "$COMBINATIONS_DIR"/*.cpp 2>/dev/null)" ]; then
    echo "❌ Error: No combination files found in $COMBINATIONS_DIR"
    exit 1
fi

# Count available combinations
COMBINATION_COUNT=$(ls -1 "$COMBINATIONS_DIR"/solver_combination_*.cpp 2>/dev/null | wc -l)
echo "✓ Found $COMBINATION_COUNT combination files"

# Use Python script for comprehensive evaluation
echo "  Running comprehensive evaluation with analysis..."

# Build evaluation command
EVAL_CMD="python evaluate_combinations.py \
    --combinations_dir \"$COMBINATIONS_DIR\" \
    --eval_data_dir \"$EVAL_DATA_DIR\" \
    --output_dir \"$EVAL_RESULTS_DIR\" \
    --report_dir \"$REPORT_DIR\" \
    --eval_parallel_size \"$EVAL_PARALLEL_SIZE\" \
    --eval_timeout \"$EVAL_TIMEOUT\""

# Add max_combinations if specified
if [ "$MAX_COMBINATIONS" -gt 0 ]; then
    EVAL_CMD="$EVAL_CMD --max_combinations \"$MAX_COMBINATIONS\""
fi

# Add keep_temp_files if enabled
if [ "$KEEP_TEMP_FILES" = true ]; then
    EVAL_CMD="$EVAL_CMD --keep_temp_files"
fi

echo "  Command: $EVAL_CMD"
eval $EVAL_CMD

if [ $? -ne 0 ]; then
    echo "❌ Error: Evaluation failed"
    exit 1
fi

echo "✅ Successfully completed evaluation"

# ================================================
# Step 3: Summary
# ================================================
echo ""
echo "📊 Step 3: Summary"
echo "=================================================="

# Count results
DATASET_NAME=$(basename "$EVAL_DATA_DIR")
EVAL_RESULTS_COUNT=$(find "$EVAL_RESULTS_DIR" -name "results_*.txt" 2>/dev/null | wc -l)
REPORT_FILES_COUNT=$(find "$REPORT_DIR" -name "*.csv" -o -name "*.json" -o -name "*.txt" 2>/dev/null | wc -l)

echo "✅ Workflow completed successfully!"
echo ""
echo "📁 Generated Files:"
echo "  📂 Combinations: $COMBINATIONS_DIR ($COMBINATION_COUNT files)"
echo "  📂 Individual Results: $EVAL_RESULTS_DIR ($EVAL_RESULTS_COUNT result files)"
echo "  📂 Analysis Reports: $REPORT_DIR ($REPORT_FILES_COUNT report files)"
echo ""
echo "🔍 Quick Access:"
echo "  📈 Performance Summary: $REPORT_DIR/performance_summary.txt"
echo "  📊 Best Combinations: $REPORT_DIR/best_combinations_${DATASET_NAME}_*.csv"
echo "  📋 Detailed Analysis: $REPORT_DIR/combination_analysis_${DATASET_NAME}_*.json"
echo ""
echo "🎯 Next Steps:"
echo "  1. Review the performance summary for overall results"
echo "  2. Check individual results in $EVAL_RESULTS_DIR/$DATASET_NAME/"
echo "  3. Use the best combinations for further testing"
echo ""
echo "=================================================="
echo "🎉 AutoSAT Combination Evaluation Complete!"
echo "==================================================" 