from processing.fourierize import empirical_cf_2d
from processing.music_2d import music_2d
from mixture.gmm import GMM
from reduction.pca import PCA
from mixture.em import EM
import numpy as np
import os
import time
from multiprocessing import Pool
from scipy.stats import wasserstein_distance_nd


def process_trial(n_samples, center):
    ndim = 100
    model = GMM(2, ndim, [-center, center], np.eye(ndim), [1/2, 1/2])
    em = EM(2, eps=1e-6)
    em_pca = EM(2, eps=1e-6)
    samples = model.sample(n_samples)

    music_start = time.time()
    pca = PCA(2)
    pca.fit(samples)
    reduced_samples = pca.transform(samples)
    Omega = np.sqrt(2)
    domain_x = domain_y = np.linspace(-Omega, Omega, 5)    
    fourier_data = empirical_cf_2d(reduced_samples, domain_x, domain_y, modulate_term=np.eye(2))
    results, music_weights = music_2d(domain_x, domain_y, fourier_data, 2, (2,2), n_omegas=101, freq_bound=[-3,3], show_plot=False, centralShift=True, weight_threshold=0.2, min_distance=0.5, returnWeights=True)
    music_centers = pca.inverse_transform(np.asarray(results))
    music_end = time.time()
    print(f"music finished in {music_end - music_start} seconds, with error {wasserstein_distance_nd(music_centers, [-center, center], music_weights, [1/2, 1/2])}")

    em_start = time.time()
    model_em = em.estimate_fixed_sigma(samples, sigma=np.eye(ndim))
    em_end = time.time()
    em_centers, em_weights = model_em.centers, model_em.weights
    print(f"em finished in {em_end - em_start} seconds")

    em_pca_start = time.time()
    pca = PCA(2)
    pca.fit(samples)
    reduced_samples = pca.transform(samples)
    model_em_pca = em_pca.estimate_fixed_sigma(reduced_samples, sigma=np.eye(2))
    em_centers_pca, em_weights_pca = model_em_pca.centers, model_em_pca.weights
    em_centers_pca = pca.inverse_transform(em_centers_pca)
    em_pca_end = time.time()
    print(f"em_pca finished in {em_pca_end - em_pca_start} seconds")

    print(f"Sample size: {n_samples}===========================")
    return wasserstein_distance_nd(music_centers, [-center, center], music_weights, [1/2, 1/2]),\
          wasserstein_distance_nd(em_centers, [-center, center], em_weights, [1/2, 1/2]),\
             wasserstein_distance_nd(em_centers_pca, [-center, center], em_weights_pca, [1/2, 1/2]),\
              music_end - music_start, em_end - em_start, em_pca_end - em_pca_start

if __name__ == "__main__":
    n_trials = 96 
    sample_sizes = np.arange(10000, 210000, 10000)
    records = np.zeros((len(sample_sizes), n_trials, 6))
    

    for i, n_samples in enumerate(sample_sizes):
        radius = 1
        centers = np.random.uniform(low=-1, high=1, size=(n_trials, 100))
        centers = centers / np.linalg.norm(centers, axis=1).reshape(-1, 1) * radius
        pool = Pool()
        inputs = [(n_samples, centers[i]) for i in range(n_trials)]
        results = pool.starmap(process_trial, inputs)
        pool.close()
        pool.join()

        for j, result in enumerate(results):
            records[i, j] = result
        print(np.mean(records[i, :, 0], axis=0), np.mean(records[i, :, 1], axis=0), np.mean(records[i, :, 2], axis=0), np.mean(records[i, :, 3], axis=0), np.mean(records[i, :, 4], axis=0), np.mean(records[i, :, 5], axis=0))
    
        np.save("comparisonEM/records_two_1_1_pca_test.npy", records)




    


    
    