#!/usr/bin/env python3
"""
Baseline experiment script that replicates the functionality of:
bash scripts/run_experiment.sh data imagenet vit_b16_ep50 end 16 1 False 0.25 200
"""

import argparse
import os
import sys
from experiment import main as experiment_main

def run_baseline_experiment(output_dir_path: str) -> None:
    """
    Run the baseline experiment with predefined parameters.
    
    This function replicates the shell script:
    bash scripts/run_experiment.sh data imagenet vit_b16_ep50 end 16 1 False 0.25 200
    
    Parameters from the shell script:
    - DATA: data
    - DATASET: imagenet
    - CFG: vit_b16_ep50 (config file)
    - CTP: end (class token position)
    - NCTX: 16 (number of context tokens)
    - SHOTS: 1 (number of shots)
    - CSC: False (class-specific context)
    - lambda: 0.25
    - topk: 200
    - METHOD: locoop (default)
    """
    
    # Set up arguments that match the shell script behavior
    args = argparse.Namespace()
    
    # Core arguments from shell script
    args.root = "/datasets/LoCoOp"  # DATA parameter
    args.seed = 0
    args.trainer = "LoCoOp"  # TRAINER
    args.dataset_config_file = "configs/datasets/imagenet.yaml"  # DATASET
    args.config_file = "configs/trainers/LoCoOp/vit_b16_ep30.yaml"  # CFG
    
    # Output directory based on shell script logic
    # OUTPUT_DIR=output/${DATASET}/${TRAINER}/${CFG}_${SHOTS}shots/nctx${NCTX}_csc${CSC}_ctp${CTP}/seed0
    dataset = "imagenet"
    cfg = "vit_b16_ep30"
    shots = 1
    nctx = 16
    csc = False
    ctp = "end"
    
    args.output_dir = output_dir_path
    
    # LoCoOp specific parameters
    args.lambda_value = 0.25  # lambda parameter
    args.topk = 200  # topk parameter
    args.method = "locoop"  # METHOD parameter (default)
    
    # Additional parameters from experiment.py
    args.resume = ""
    args.backbone = ""
    args.head = ""
    args.eval_only = False
    args.model_dir = ""
    args.load_epoch = None
    args.no_train = False
    args.T = 1.0
    args.sample_size = 500
    
    # Options list for configuration overrides (from shell script)
    args.opts = [
        "TRAINER.LOCOOP.N_CTX", str(nctx),           # 16
        "TRAINER.LOCOOP.CSC", str(csc),              # False
        "TRAINER.LOCOOP.CLASS_TOKEN_POSITION", ctp,   # end
        "DATASET.NUM_SHOTS", str(shots)               # 1
    ]
    
    print("Running baseline experiment with parameters:")
    print(f"  Root: {args.root}")
    print(f"  Dataset: imagenet")
    print(f"  Config: {cfg}")
    print(f"  Shots: {shots}")
    print(f"  Context tokens: {nctx}")
    print(f"  Class token position: {ctp}")
    print(f"  Class-specific context: {csc}")
    print(f"  Lambda: {args.lambda_value}")
    print(f"  Top-k: {args.topk}")
    print(f"  Method: {args.method}")
    print(f"  Output dir: {args.output_dir}")
    print()
    
    # Create output directory if it doesn't exist
    os.makedirs(args.output_dir, exist_ok=True)
    # Run the main experiment
    experiment_main(args)


def main():
    """Main entry point for baseline script."""
    parser = argparse.ArgumentParser(
        description="Baseline experiment that replicates shell script behavior"
    )
    parser.add_argument(
        "--output-dir", 
        type=str, 
        required=True,
        help="Path to output directory where results will be saved"
    )
    
    args = parser.parse_args()
    
    # Run the baseline experiment
    run_baseline_experiment(args.output_dir)


if __name__ == "__main__":
    main()