import numpy as np
from pycocotools import mask as masker


def convert_array_to_rle(attention: np.ndarray) -> dict:
    """Converts an attention in numpy array format to RLE format.

    RLE format compresses the attention lossless.

    Args:
        attention (np.ndarray): attention of an object

    Returns:
        dict: the attention in RLE format.
    """
    # pycocotools requires the array to be F(ortran)-CONTIGUOUS
    attention = np.asfortranarray(attention)
    RLE_mask: dict = masker.encode(attention)  # type: ignore
    RLE_mask["counts"] = RLE_mask["counts"].decode("utf-8")
    return RLE_mask


def calculate_mid_mask(RLE_mask: dict) -> list[float]:
    """Calculates given a binary mask the middle point of the mask."""
    binary_mask = masker.decode(RLE_mask)
    y_nonzero, x_nonzero = np.nonzero(binary_mask)
    x_nonzero_arr = list(set(x_nonzero))
    x_mid: float = x_nonzero_arr[int(len(x_nonzero_arr) / 2)]
    y_nonzero_arr = list(set(y_nonzero))
    y_mid: float = y_nonzero_arr[int(len(y_nonzero_arr) / 2)]
    return [x_mid, y_mid]
