import pandas as pd
from plot.utils import (apply_average_across_series,
                        apply_window_over_each_series, reduce_data)
from tqdm import tqdm


def read_lisp(path):
    data_path = f"{path}.lisp"
    with open(data_path, "r") as f:
        data = f.read().splitlines()
    data = data[5][3:-3].split(") (")
    data = [i.split(" ") for i in data]
    data = [[float(j) for j in i] for i in data]
    data = pd.DataFrame(data)
    data = data.T
    data.loc[-1] = data.loc[0]
    data.index = data.index + 1
    data.sort_index(inplace=True)
    return data


def collect_data_for_evo_plot(
    raw_file_paths,
    windows=None,
    window_type=None,
    only_x_last_interactions=None,
    start=None,
    end=None,
    series_numbers=None,
    average_mode="mean",
    error_mode="stdev",
    percentiles=[5, 95],
    per_agent=None,
    every_x_interactions=None,
):
    """
    Collect all data for the evo plot.

    :param raw_file_paths: The paths to the data
    :param file_type: The file type of the data files. Default='dat'
    :param windows: The window for the rolling mean,
        specified for each data file.
    :param only_x_last_interactions: Only keep the 'x' last interactions.
    :param start: Only keep interactions after 'start'
    :param end: Only keep interactions before 'end'
    :param series_numbers: Only keep these series.
        Series start counting from 1.
    :param average_mode: Use this average mode across the series.
    :param error_mode: Use this error mode across the series.
    :param percentiles: Use this percentiles for the error mode (if necessary).
    :returns: A list of Pandas DataFrames containing all data for the evo plot.
    """
    result = []
    for path, window in tqdm(zip(raw_file_paths, windows), total=len(raw_file_paths)):
        data = read_lisp(path)
        series_length = len(data)

        if start is not None or end is not None:
            s = start if start is not None else 0
            e = end if end is not None else len(data)
            data = data.iloc[s:e, :]
            data = data.reset_index(drop=True)

        if window_type == "standard":
            if window is not None and path.split("/")[-1] != "unique-form-usage":
                data = apply_window_over_each_series(data, window)

        if only_x_last_interactions is not None:
            x = series_length - only_x_last_interactions
            data = data.iloc[x:, :]
            data = data.reset_index(drop=True)

        if series_numbers is not None:
            if type(series_numbers) is not list:
                raise Exception(
                    f"""series_numbers should be a list,
                                    got {series_numbers}"""
                )
            else:
                series_indexes = [s - 1 for s in series_numbers]
                data = data.iloc[:, series_indexes]

        if len(data.columns) > 1:
            data = apply_average_across_series(
                data, average_mode, error_mode, percentiles
            )

        if every_x_interactions is not None:
            data = reduce_data(data, every_x_interactions)

        if per_agent:
            # select every xth row in the dataframe data
            divide_by_index = per_agent // 2
            data = data.iloc[::divide_by_index, :]
            data = data.reset_index(drop=True)
        result.append(data)
    return result
