import json
import os

import cv2 as cv
import numpy as np
from detectron2 import model_zoo
from detectron2.config import get_cfg
from detectron2.engine import DefaultPredictor

from dm.dotdict import DotDict


def load_json(path: str) -> dict:
    with open(path, "r") as f:
        return json.load(f)


def load_module_config(opt) -> None:
    """Loads the configuration of a given module into the general configuration."""
    print("Reading module config file...")
    cfg_path = f"models/{opt.module_config}"
    data = load_json(cfg_path)
    opt.module_info = DotDict(data)
    print(f"Module config {cfg_path} loaded.")


def load_detectron(use_cuda: bool, module_info: DotDict) -> DefaultPredictor:
    cfg = get_cfg()
    cfg.merge_from_file(model_zoo.get_config_file(module_info["architecture_path"]))
    cfg.DATASETS.TRAIN = ("clevr_mini_train",)
    cfg.DATASETS.TEST = ()
    cfg.DATALOADER.NUM_WORKERS = 0
    cfg.MODEL.DEVICE = "cuda:0" if use_cuda else "cpu"
    cfg.MODEL.WEIGHTS = module_info.weights
    cfg.INPUT.MASK_FORMAT = module_info.mask_format
    cfg.MODEL.ROI_HEADS.NUM_CLASSES = module_info.n_classes
    cfg.MODEL.ROI_HEADS.SCORE_THRESH_TEST = module_info.threshold
    return DefaultPredictor(cfg)


def load_img(dataset, split, filename) -> np.ndarray:
    img_path = os.path.join(f"data/{dataset}/images/{split}", filename)
    img = cv.imread(img_path)
    img = cv.cvtColor(img, cv.COLOR_RGBA2RGB)
    return img
