import cv2 as cv
import numpy as np
from pycocotools import mask
from scipy.spatial.distance import hamming
from scipy.stats import circmean, circstd
from skimage import color


def circular_std(values):
    values = np.deg2rad(values)
    std = circstd(values)
    return np.rad2deg(std)


def circular_mean(values):
    values = np.deg2rad(values)
    mean = circmean(values)
    return np.rad2deg(mean)


def get_color_for_mask(im, mask):
    hsv = cv.cvtColor(im, cv.COLOR_BGR2HSV)
    h, s, v = cv.split(hsv)
    h_mean = circular_mean(h[mask != 0] * 2.0)
    s_mean = (np.mean(s[mask != 0]) / 255.0) * 100.0
    v_mean = (np.mean(v[mask != 0]) / 255.0) * 100.0
    mean = [h_mean, s_mean, v_mean]
    _, std = cv.meanStdDev(hsv, mask=mask)
    _, s_std, v_std = std
    h_std = circular_std(h[mask != 0] * 2.0) / 2.0
    std = [h_std, s_std[0], v_std[0]]
    return mean, std


def get_hamming_distance_to_enclosing_circle(mask, cnt, center, radius):
    obj2 = np.zeros(mask.shape, dtype=np.uint8)
    cv.circle(obj2, center, radius, color=255, thickness=cv.FILLED)
    circleCont, _ = cv.findContours(obj2, cv.RETR_EXTERNAL, cv.CHAIN_APPROX_SIMPLE)
    obj1 = np.zeros(mask.shape, dtype=np.uint8)
    cv.drawContours(obj1, [cnt], -1, 1, cv.FILLED)
    x, y, w, h = cv.boundingRect(cnt)
    obj1 = obj1[y : y + h, x : x + w]
    obj2 = obj2[y : y + h, x : x + w]
    return hamming(obj1.flatten(), obj2.flatten())


def get_white_and_black_percentage(image, mask, area):
    hsv = cv.cvtColor(image, cv.COLOR_BGR2HSV)
    h, s, v = cv.split(hsv)
    s = s[mask != 0]
    v = v[mask != 0]
    low_s = s < 0.5 * 255
    high_v = v > 0.7 * 255
    low_v = v < 0.2 * 255
    whites = np.count_nonzero(np.bitwise_and(low_s, high_v))
    blacks = np.count_nonzero(low_v)
    return whites / float(area), blacks / float(area)

def hsv2rgb(hsv):
    h = hsv[0] / 360
    s = hsv[1] / 100
    v = hsv[2] / 100
    return color.hsv2rgb([h, s, v])


def extract_feature(img, RLE_mask):
    binary_mask = mask.decode(RLE_mask)

    # Extract contours
    binary_mask_copy = binary_mask.copy()
    contours, _ = cv.findContours(
        binary_mask_copy, cv.RETR_EXTERNAL, cv.CHAIN_APPROX_TC89_KCOS
    )

    # If multiple are found, take the biggest
    if len(contours) > 1:
        contour_areas = [cv.contourArea(c) for c in contours]
        max_idx = contour_areas.index(max(contour_areas))
        cnt = contours[max_idx]
    else:
        cnt = contours[0]
    # Extract size related features
    (_, _), (width, height), angle = cv.minAreaRect(cnt)
    area = cv.contourArea(cnt)
    bb_area = width * height
    bb_area_ratio = area / bb_area
    wh_ratio = width / height
    image_area = img.shape[0] * img.shape[1]
    image_area_ratio = area / image_area
    # Extract XY-coordinates and Hamming distance
    (x, y), radius = cv.minEnclosingCircle(cnt)
    center = (int(x), int(y))
    radius = int(radius)
    circle_distance = get_hamming_distance_to_enclosing_circle(
        binary_mask, cnt, center, radius
    )
    # Extract number of corners
    approx_cnt = cv.approxPolyDP(cnt, 0.05 * cv.arcLength(cnt, True), True)
    number_of_corners = len(approx_cnt)
    # Extract white and black levels
    whites, blacks = get_white_and_black_percentage(img, mask, area)

    # Extract colours
    hsv_mean, hsv_std = get_color_for_mask(img, binary_mask)
    rgb_mean = hsv2rgb(hsv_mean)
    rgb_std = hsv2rgb(hsv_std)

    # Return the data
    features = {
        "xpos": x,
        "ypos": y,
        "angle": angle,
        # size
        "width": width,
        "height": height,
        "area": area,
        "relative-area": image_area_ratio,
        "bb-area": bb_area,
        "bb-area-ratio": bb_area_ratio,
        # shape
        "corners": number_of_corners,
        "wh-ratio": wh_ratio,
        "circle-distance": circle_distance,
        # material
        "white-level": whites,
        "black-level": blacks,
        # color
        "rgb-mean-r": rgb_mean[0],
        "rgb-mean-g": rgb_mean[1],
        "rgb-mean-b": rgb_mean[2],
        "rgb-std-r": rgb_std[0],
        "rgb-std-g": rgb_std[1],
        "rgb-std-b": rgb_std[2],
    }
    return features
