import os
import random
import re
from math import floor, log10

import pandas as pd

# The same gnuplot colors as used in plot-raw-data
global GREAT_GNUPLOT_COLORS
GREAT_GNUPLOT_COLORS = [
    "darkcyan",
    "darkgoldenrod",
    "darkred",
    "navy",
    "darkgreen",
    "gray",
    "r",
    "green",
    "darkorange",
    "royalblue",
    "seagreen",
    "deeppink",
    "purple",
    "orangered",
    "darkgray",
    "darkkhaki",
    "darkturquoise",
    "salmon",
    "darkmagenta",
    "y",
    "violet",
    "lightgreen",
]


def pathname_name(path):
    """
    Extracts the filename of a pathname

    :param path: a pathname
    :returns: a filename
    """
    return os.path.splitext(os.path.basename(path))[0]


def pathname_type(path):
    """
    Extracts the file type of a pathname

    :param path: a pathname
    :returns: a file type
    """
    return os.path.splitext(path)[1]


def get_only_subdirectory(path):
    subdirectories = [f.path for f in os.scandir(path) if f.is_dir()]
    return subdirectories[0]


def pathname_directory(path):
    """
    Extracts the directory of a pathname

    :param path: a pathname
    :returns: a directory pathname
    """
    return os.path.dirname(path)


def get_error_bar_distance(n):
    """
    Compute an appropriate spacing for the error bars, depending
    on the number of data points

    :param n: Number of data points
    :returns: The spacing for the error bars
    """
    for x in range(0, 10000000, 1000):
        if n <= x + 1:
            return round(x / 10)


def apply_window_over_each_series(data, window):
    """
    Apply a rolling mean over each serie in the data.

    :param data: a Pandas DataFrame
    :param window: the size of the window for the rolling mean
    :returns: the Pandas DataFrame with rolling mean applied
    """
    return data.rolling(window=window, min_periods=1).mean()


def apply_window_over_each_series_half(data, window):
    """
    Apply a rolling mean over each serie in the data.

    :param data: a Pandas DataFrame
    :param window: the size of the window for the rolling mean
    :returns: the Pandas DataFrame with rolling mean applied
    """
    half_1 = data.iloc[: len(data) // 2 - 1]
    half_2 = data.iloc[len(data) // 2 :]
    rolling_half_1 = half_1.rolling(window=window, min_periods=1).mean()
    rolling_half_2 = half_2.rolling(window=window, min_periods=1).mean()
    rolling_data = pd.concat([rolling_half_1, rolling_half_2])
    return rolling_data


def apply_average_across_series(data, average_mode, error_mode, percentiles=[5, 95]):
    """
    Apply some averaging operation across the series in the data.

    :param data: a Pandas DataFrame
    :param average_mode: What operation to use for averaging across the series.
    :param error_mode: What operation to use to indicate the variation across
        the series.
    :param percentiles: When using the 'percentile' error_mode, the quantiles
        used for the low and high end can be specified using this argument.
        Values specified should be between 0 and 100.
    :returns: a Pandas DataFrame with three extra columns, one for specifying
        the average value and the other two for specifying the low and high
        end of the error bar.
    """
    if average_mode == "mean":
        data["average"] = data.mean(axis=1)
    elif average_mode == "median":
        data["average"] = data.median(axis=1)
    else:
        raise Exception(
            f"""Unknown average_mode {average_mode}. Supported
                            modes are: 'mean' and 'median'"""
        )

    if error_mode == "stdev":
        std = data.std(axis=1)
        data["error_low"] = std * 2
        data["error_high"] = std * 2

    return data


def reduce_data(data, n):
    """
    Reduce the number of datapoints in the data.

    :param data: a Pandas DataFrame
    :param n: the number of datapoints to reduce to
    :returns: a Pandas DataFrame with the number of datapoints reduced to n
    """
    return data.iloc[::n, :]


def random_value(seed, a, b):
    random.seed(seed)
    return round(random.uniform(a, b))


def custom_sort(strings):
    def sorting_key(s):
        match = re.match(r"^(.*?)(-(\d+))?$", s)
        text_part = match.group(1)
        num_part = match.group(3)

        # If there's a number part, convert it to integer; otherwise, set it to a high value
        num_value = int(num_part) if num_part is not None else float("inf")

        return (text_part, num_value)

    return sorted(strings, key=sorting_key)


def format_func(value, tick_number=None):
    num_thousands = 0 if abs(value) < 1000 else floor(log10(abs(value)) / 3)
    value = round(value / 1000**num_thousands, 2)
    return f"{value:g}" + " KMGTPEZY"[num_thousands]
