import numpy as np
from robosuite.controllers import load_controller_config
from robosuite.environments.base import make
from robosuite.utils.placement_samplers import UniformRandomSampler


def make_lift_env(robots, env_kwargs, image_observation=False, z_offset=0.0):
    controller_config = load_controller_config(
        default_controller="OSC_POSITION")
    object_center = np.array([0, 0])
    placement_initializer = UniformRandomSampler(
        name="ObjectSampler",
        x_range=object_center[0] + np.array([-0.001, 0.001]),
        y_range=object_center[1] + np.array([-0.001, 0.001]),
        rotation=0,
        ensure_object_boundary_in_range=False,
        ensure_valid_placement=True,
        reference_pos=[0, 0, 0.8],
        z_offset=z_offset + 0.01,
    )

    default_kwargs = {
        "controller_configs": controller_config,
        "placement_initializer": placement_initializer,
        "gripper_types": "default",
        "control_freq": 30,
        "horizon": 120,
        "initialization_noise": {
            "type": "uniform",
            "magnitude": 0.2,
        },
        "has_renderer": False,
        "has_offscreen_renderer": True,
        "render_camera": "agentview",
        "use_object_obs": True,
        "use_camera_obs": True,
        "camera_heights": 128,
        "camera_widths": 128,
        "camera_names": ["agentview", "sideview"],
        "reward_shaping": True,
    }
    if env_kwargs:
        default_kwargs.update(env_kwargs)

    env = make(
        env_name="Lift",
        robots=robots,
        **default_kwargs,
    )

    return env


def set_lift_task_id(env, task_id):
    x, y, z = env.task_id_to_pos(task_id)
    env.env.placement_initializer.x_range = x + env.position_noise_range
    env.env.placement_initializer.y_range = y + env.position_noise_range
    env.env.placement_initializer.z_offset = z + 0.01
    env.set_table_offset(z)
    env.reset()
    return [x, y]
