import numpy as np
from robosuite.controllers import load_controller_config
from robosuite.environments.base import make
from robosuite.utils.placement_samplers import (SequentialCompositeSampler,
                                                UniformRandomSampler)


def make_placement_initializer(env, task_id: int):
    object_center = env.task_id_to_pos(task_id=task_id)
    placement_initializerA = UniformRandomSampler(
        name="ObjectSamplerA",
        x_range=object_center[0] + np.array([-0.001, 0.001]),
        y_range=object_center[1] + np.array([-0.001, 0.001]),
        rotation=0,
        ensure_object_boundary_in_range=False,
        ensure_valid_placement=True,
        reference_pos=[0, 0, 0.8],
        z_offset=0.01,
    )

    placement_initializerB = UniformRandomSampler(
        name="ObjectSamplerB",
        x_range=np.array([0.0, 0.0]),
        y_range=np.array([0.0, 0.0]),
        rotation=0,
        ensure_object_boundary_in_range=False,
        ensure_valid_placement=True,
        reference_pos=[0, 0, 0.8],
        z_offset=0.01,
    )

    placement_initializer = SequentialCompositeSampler("CompositeSampler")
    placement_initializer.append_sampler(placement_initializerA)
    placement_initializer.append_sampler(placement_initializerB)

    if isinstance(env.env.placement_initializer, UniformRandomSampler):
        cubes = env.env.placement_initializer.mujoco_objects
    elif isinstance(env.env.placement_initializer, SequentialCompositeSampler):
        cubes = [
            env.env.placement_initializer.samplers["ObjectSamplerA"].
            mujoco_objects[0],
            env.env.placement_initializer.samplers["ObjectSamplerB"].
            mujoco_objects[0],
        ]
    placement_initializer.add_objects_to_sampler("ObjectSamplerA", cubes[0])
    placement_initializer.add_objects_to_sampler("ObjectSamplerB", cubes[1])
    return placement_initializer


def make_stack_env(robots, env_kwargs, image_observation=False):
    controller_config = load_controller_config(
        default_controller="OSC_POSITION")

    default_kwargs = {
        "controller_configs": controller_config,
        "gripper_types": "default",
        "control_freq": 30,
        "horizon": 180,
        "initialization_noise": {
            "type": "uniform",
            "magnitude": 0.2,
        },
        "has_renderer": False,
        "has_offscreen_renderer": True,
        "render_camera": "agentview",
        "use_object_obs": True,
        "use_camera_obs": True,
        "camera_heights": 128,
        "camera_widths": 128,
        "camera_names": ["agentview", "sideview"],
        "reward_shaping": True,
        "hard_reset": False,
    }
    if env_kwargs:
        default_kwargs.update(env_kwargs)

    env = make(
        env_name="Stack",
        robots=robots,
        **default_kwargs,
    )
    return env


def set_stack_task_id(env, task_id: int):
    if task_id >= 4:
        task_id += 1
    env.env.placement_initializer = make_placement_initializer(
        env,
        task_id=task_id,
    )
    return np.array([0, 0])
