import torch
import torch.nn as nn
from transformers.models.bert.modeling_bert import (
    BertAttention,
    BertPreTrainedModel,
    BertEmbeddings,
    BertIntermediate,
    BertOutput,
    apply_chunking_to_forward
)
from transformers.modeling_outputs import BaseModelOutputWithPoolingAndCrossAttentions
from typing import Optional, Union, Tuple
from outputs import MoEBertOutputWithPastAndCrossAttentions, MoEBertOutputWithPoolingAndCrossAttentions
from moe_blocks import *
from typing import List, Optional, Tuple, Union
from transformers.modeling_attn_mask_utils import _prepare_4d_attention_mask_for_sdpa
from transformers.activations import ACT2FN


class GroupPooling(nn.Module):
    def __init__(self, config):
        super().__init__()
        self.config = config

    def forward(self, hidden_states, attention_mask, router_labels, padding_embedding):
        batch_size, seq_len, hidden_dim = hidden_states.shape
        device = hidden_states.device
        dtype = hidden_states.dtype

        # Step 1: Mask router labels where attention is not allowed
        masked_router_labels = router_labels.masked_fill(attention_mask == 0, -1)

        # Step 2: Detect group boundaries and assign group IDs
        padded = torch.nn.functional.pad(masked_router_labels, (1, 0), value=-2)
        boundaries = padded[:, 1:] != padded[:, :-1]
        group_ids = boundaries.cumsum(dim=1)

        # Step 3: Determine max number of groups across the batch
        max_groups = group_ids.max().item() + 1

        # Step 4: Flatten and mask tensors
        valid_mask = attention_mask.bool().flatten()  # (B*S,)
        valid_group_ids = group_ids.flatten()[valid_mask]  # (K,)
        valid_hidden = hidden_states.flatten(0, 1)[valid_mask]  # (K, H)

        # Step 5: Create unique group IDs across the batch
        batch_indices = torch.arange(batch_size, device=device).view(-1, 1).expand(-1, seq_len).flatten()[valid_mask]
        unique_group_ids = batch_indices * max_groups + valid_group_ids

        # Step 6: Scatter-add for sum and count
        total_groups = batch_size * max_groups
        pooled_sum = torch.zeros((total_groups, hidden_dim), device=device, dtype=dtype)
        pooled_sum.scatter_add_(0, unique_group_ids.unsqueeze(1).expand(-1, hidden_dim), valid_hidden)

        counts = torch.zeros((total_groups, 1), device=device, dtype=torch.float)
        ones = torch.ones_like(unique_group_ids.unsqueeze(1), dtype=torch.float)
        counts.scatter_add_(0, unique_group_ids.unsqueeze(1), ones)

        # Step 7: Reshape and normalize
        pooled_sum = pooled_sum.view(batch_size, max_groups, hidden_dim)
        counts = counts.view(batch_size, max_groups, 1)

        counts_clamped = counts.clamp_(min=1)
        pooled = pooled_sum.div_(counts_clamped)

        # Step 8: Fill empty groups with padding embedding
        empty_group_mask = (counts.squeeze(-1) == 0)
        if empty_group_mask.any():
            pooled[empty_group_mask] = padding_embedding.to(pooled.dtype)

        # Step 9: Create attention mask and exclude dummy group
        new_attention_mask = (counts.squeeze(-1) > 0).to(attention_mask.dtype)
        return pooled[:, 1:], new_attention_mask[:, 1:]


class BertExpert(nn.Module):
    def __init__(self, config):
        super().__init__()
        self.intermediate_up = nn.Linear(config.hidden_size, config.intermediate_size_expert) # BertIntermediate dense
        self.intermediate_down = nn.Linear(config.intermediate_size_expert, config.hidden_size) # BertOutput dense
        self.dropout = nn.Dropout(config.hidden_dropout_prob)
        self.act = ACT2FN[config.hidden_act]
    
    def forward(self, hidden_states):
        x = self.act(self.intermediate_up(hidden_states))
        x = self.dropout(self.intermediate_down(x))
        return x + hidden_states


class BertPooler(nn.Module):
    def __init__(self, config):
        super().__init__()
        self.dense = nn.Linear(config.hidden_size, config.hidden_size)
        self.activation = nn.Tanh()

    def forward(self, hidden_states: torch.Tensor) -> torch.Tensor:
        first_token_tensor = hidden_states[:, 0]
        pooled_output = self.dense(first_token_tensor)
        pooled_output = self.activation(pooled_output)
        return pooled_output


class BertLayer(nn.Module):
    def __init__(self, config):
        super().__init__()
        self.chunk_size_feed_forward = config.chunk_size_feed_forward
        self.seq_len_dim = 1
        self.attention = BertAttention(config)
        self.is_decoder = config.is_decoder
        self.add_cross_attention = config.add_cross_attention
        if self.add_cross_attention:
            if not self.is_decoder:
                raise ValueError(f"{self} should be used as a decoder model if cross attention is added")
            self.crossattention = BertAttention(config, position_embedding_type="absolute")
        self.intermediate = BertIntermediate(config)
        self.output = BertOutput(config)

    def forward(
        self,
        hidden_states: torch.Tensor,
        attention_mask: Optional[torch.FloatTensor] = None,
        head_mask: Optional[torch.FloatTensor] = None,
        encoder_hidden_states: Optional[torch.FloatTensor] = None,
        encoder_attention_mask: Optional[torch.FloatTensor] = None,
        past_key_value: Optional[Tuple[Tuple[torch.FloatTensor]]] = None,
        output_attentions: Optional[bool] = False,
        *args,
        **kwargs,
    ) -> Tuple[torch.Tensor]:
        # decoder uni-directional self-attention cached key/values tuple is at positions 1,2
        self_attn_past_key_value = past_key_value[:2] if past_key_value is not None else None
        self_attention_outputs = self.attention(
            hidden_states,
            attention_mask,
            head_mask,
            output_attentions=output_attentions,
            past_key_value=self_attn_past_key_value,
        )
        attention_output = self_attention_outputs[0]

        # if decoder, the last output is tuple of self-attn cache
        if self.is_decoder:
            outputs = self_attention_outputs[1:-1]
            present_key_value = self_attention_outputs[-1]
        else:
            outputs = self_attention_outputs[1:]  # add self attentions if we output attention weights

        cross_attn_present_key_value = None
        if self.is_decoder and encoder_hidden_states is not None:
            if not hasattr(self, "crossattention"):
                raise ValueError(
                    f"If `encoder_hidden_states` are passed, {self} has to be instantiated with cross-attention layers"
                    " by setting `config.add_cross_attention=True`"
                )

            # cross_attn cached key/values tuple is at positions 3,4 of past_key_value tuple
            cross_attn_past_key_value = past_key_value[-2:] if past_key_value is not None else None
            cross_attention_outputs = self.crossattention(
                attention_output,
                attention_mask,
                head_mask,
                encoder_hidden_states,
                encoder_attention_mask,
                cross_attn_past_key_value,
                output_attentions,
            )
            attention_output = cross_attention_outputs[0]
            outputs = outputs + cross_attention_outputs[1:-1]  # add cross attentions if we output attention weights

            # add cross-attn cache to positions 3,4 of present_key_value tuple
            cross_attn_present_key_value = cross_attention_outputs[-1]
            present_key_value = present_key_value + cross_attn_present_key_value

        layer_output = apply_chunking_to_forward(
            self.feed_forward_chunk, self.chunk_size_feed_forward, self.seq_len_dim, attention_output
        )
        outputs = (layer_output,) + outputs

        # if decoder, return the attn key/values as the last output
        if self.is_decoder:
            outputs = outputs + (present_key_value,)

        return outputs

    def feed_forward_chunk(self, attention_output):
        intermediate_output = self.intermediate(attention_output)
        layer_output = self.output(intermediate_output, attention_output)
        return layer_output


class BertEncoder(nn.Module):
    def __init__(self, config):
        super().__init__()
        self.config = config
        self.moe_idx = config.num_hidden_layers // 2
        self.layer = nn.ModuleList()
        for i in range(self.config.num_hidden_layers):
            self.layer.append(BertLayer(config))
        self.gradient_checkpointing = config.gradient_checkpointing

    def forward(
        self,
        hidden_states,
        original_attention_mask=None,
        attention_mask=None,
        padding_embedding=None,
        token_type_ids=None,
        router_labels=None,
        head_mask=None,
        encoder_hidden_states=None,
        encoder_attention_mask=None,
        past_key_values=None,
        use_cache=None,
        output_attentions=False,
        output_hidden_states=False,
        return_dict=True,
    ):
        all_hidden_states = () if output_hidden_states else None
        all_self_attentions = () if output_attentions else None
        all_cross_attentions = () if output_attentions and self.config.add_cross_attention else None
        all_router_logits = ()

        next_decoder_cache = () if use_cache else None
        for i, layer_module in enumerate(self.layer):

            if output_hidden_states:
                all_hidden_states = all_hidden_states + (hidden_states,)

            layer_head_mask = head_mask[i] if head_mask is not None else None
            past_key_value = past_key_values[i] if past_key_values is not None else None

            if self.gradient_checkpointing and self.training:
                layer_outputs = self._gradient_checkpointing_func(
                    layer_module.__call__,
                    hidden_states,
                    attention_mask,
                    layer_head_mask,
                    encoder_hidden_states,
                    encoder_attention_mask,
                    past_key_value,
                    output_attentions,
                    use_reentrant=False,
                )
            else:
                layer_outputs = layer_module(
                    hidden_states,
                    attention_mask,
                    layer_head_mask,
                    encoder_hidden_states,
                    encoder_attention_mask,
                    past_key_value,
                    output_attentions,
                )

            hidden_states = layer_outputs[0]
            if use_cache:
                next_decoder_cache = next_decoder_cache + (layer_outputs[-1],)
            if output_attentions:
                all_self_attentions = all_self_attentions + (layer_outputs[1],)
                if self.config.add_cross_attention:
                    all_cross_attentions = all_cross_attentions + (layer_outputs[2],)

        if output_hidden_states:
            all_hidden_states = all_hidden_states + (hidden_states,)

        if not return_dict:
            return tuple(
                v
                for v in [
                    hidden_states,
                    next_decoder_cache,
                    all_hidden_states,
                    all_self_attentions,
                    all_cross_attentions,
                ]
                if v is not None
            )

        return MoEBertOutputWithPastAndCrossAttentions(
            last_hidden_state=hidden_states,
            hidden_states=all_hidden_states,
            attentions=original_attention_mask,
            router_logits=all_router_logits
        )


class MoETEmbedding(BertEmbeddings):
    def __init__(self, config):
        super().__init__(config)
        self.config = config
        self.experts = TokenHashMoeBlock(config, expert=BertExpert)
        self.word_embeddings = nn.Embedding(config.vocab_size, config.hidden_size, padding_idx=config.pad_token_id)
        self.position_embeddings = nn.Embedding(config.max_position_embeddings, config.hidden_size)
        self.LayerNorm = nn.LayerNorm(config.hidden_size, eps=config.layer_norm_eps)
        self.dropout = nn.Dropout(config.hidden_dropout_prob)
        self.GroupPool = GroupPooling(config)
        self.pad_token_id = config.pad_token_id

        self.position_embedding_type = getattr(config, "position_embedding_type", "absolute")
        self.register_buffer(
            "position_ids", torch.arange(config.max_position_embeddings).expand((1, -1)), persistent=False
        )

    def forward(
        self,
        input_ids: Optional[torch.LongTensor] = None,
        attention_mask: Optional[torch.LongTensor] = None,
        router_labels: Optional[torch.LongTensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        position_ids_orig: Optional[torch.LongTensor] = None,
        inputs_embeds: Optional[torch.FloatTensor] = None,
        past_key_values_length: int = 0,
    ) -> torch.Tensor:
        if input_ids is not None:
            input_shape = input_ids.size()
        else:
            input_shape = inputs_embeds.size()[:-1]

        inputs_embeds = self.word_embeddings(input_ids)

        transformed_input_embeds, _ = self.experts(inputs_embeds, router_labels=router_labels)
        transformed_input_embeds, attention_mask = self.GroupPool(
            transformed_input_embeds, attention_mask, router_labels=router_labels, padding_embedding=self.word_embeddings.weight[self.pad_token_id]
        )

        seq_length = transformed_input_embeds.size()[1]

        if position_ids is None:
            position_ids = self.position_ids[:, past_key_values_length : seq_length + past_key_values_length]

        if self.position_embedding_type == "absolute":
            position_embeddings = self.position_embeddings(position_ids)
            transformed_input_embeds += position_embeddings
        transformed_input_embeds = self.LayerNorm(transformed_input_embeds)
        transformed_input_embeds = self.dropout(transformed_input_embeds)
        return transformed_input_embeds, attention_mask


class MoEBertModel(BertPreTrainedModel):

    def __init__(self, config, add_pooling_layer=True):
        super().__init__(config)
        self.config = config

        self.embeddings = MoETEmbedding(config)
        self.encoder = BertEncoder(config)

        self.pooler = BertPooler(config) if add_pooling_layer else None

        self.attn_implementation = config._attn_implementation
        self.position_embedding_type = config.position_embedding_type

        # Initialize weights and apply final processing
        self.post_init()

    def get_input_embeddings(self):
        return self.embeddings.word_embeddings

    def set_input_embeddings(self, value):
        self.embeddings.word_embeddings = value

    def _prune_heads(self, heads_to_prune):
        for layer, heads in heads_to_prune.items():
            self.encoder.layer[layer].attention.prune_heads(heads)

    def forward(
        self,
        input_ids: Optional[torch.Tensor] = None,
        attention_mask: Optional[torch.Tensor] = None,
        token_type_ids: Optional[torch.Tensor] = None,
        router_labels: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.Tensor] = None,
        head_mask: Optional[torch.Tensor] = None,
        inputs_embeds: Optional[torch.Tensor] = None,
        encoder_hidden_states: Optional[torch.Tensor] = None,
        encoder_attention_mask: Optional[torch.Tensor] = None,
        past_key_values: Optional[List[torch.FloatTensor]] = None,
        use_cache: Optional[bool] = None,
        output_attentions: Optional[bool] = None,
        output_hidden_states: Optional[bool] = None,
        return_dict: Optional[bool] = None,
    ) -> Union[Tuple[torch.Tensor], BaseModelOutputWithPoolingAndCrossAttentions]:

        output_attentions = output_attentions if output_attentions is not None else self.config.output_attentions
        output_hidden_states = (
            output_hidden_states if output_hidden_states is not None else self.config.output_hidden_states
        )
        return_dict = return_dict if return_dict is not None else self.config.use_return_dict

        if self.config.is_decoder:
            use_cache = use_cache if use_cache is not None else self.config.use_cache
        else:
            use_cache = False

        if input_ids is not None and inputs_embeds is not None:
            raise ValueError("You cannot specify both input_ids and inputs_embeds at the same time")
        elif input_ids is not None:
            self.warn_if_padding_and_no_attention_mask(input_ids, attention_mask)
            input_shape = input_ids.size()
        elif inputs_embeds is not None:
            input_shape = inputs_embeds.size()[:-1]
        else:
            raise ValueError("You have to specify either input_ids or inputs_embeds")

        # past_key_values_length
        past_key_values_length = past_key_values[0][0].shape[2] if past_key_values is not None else 0

        embedding_output, attention_mask = self.embeddings(
            input_ids=input_ids,
            attention_mask=attention_mask,
            position_ids=position_ids,
            router_labels=router_labels,
            inputs_embeds=inputs_embeds,
            past_key_values_length=past_key_values_length,
        )

        batch_size, seq_length = embedding_output.size()[:2]
        device = input_ids.device if input_ids is not None else inputs_embeds.device

        if token_type_ids is None:
            if hasattr(self.embeddings, "token_type_ids"):
                buffered_token_type_ids = self.embeddings.token_type_ids[:, :seq_length]
                buffered_token_type_ids_expanded = buffered_token_type_ids.expand(batch_size, seq_length)
                token_type_ids = buffered_token_type_ids_expanded
            else:
                token_type_ids = torch.zeros(input_shape, dtype=torch.long, device=device)

        if attention_mask is None:
            attention_mask = torch.ones((batch_size, seq_length + past_key_values_length), device=device)

        use_sdpa_attention_masks = (
            self.attn_implementation == "sdpa"
            and self.position_embedding_type == "absolute"
            and head_mask is None
            and not output_attentions
        )

        # Expand the attention mask
        if use_sdpa_attention_masks and attention_mask.dim() == 2:
            # Expand the attention mask for SDPA.
            # [bsz, seq_len] -> [bsz, 1, seq_len, seq_len]
            if self.config.is_decoder:
                extended_attention_mask = _prepare_4d_causal_attention_mask_for_sdpa(
                    attention_mask,
                    input_shape,
                    embedding_output,
                    past_key_values_length,
                )
            else:
                extended_attention_mask = _prepare_4d_attention_mask_for_sdpa(
                    attention_mask, embedding_output.dtype, tgt_len=seq_length
                )
        else:
            # We can provide a self-attention mask of dimensions [batch_size, from_seq_length, to_seq_length]
            # ourselves in which case we just need to make it broadcastable to all heads.
            extended_attention_mask = self.get_extended_attention_mask(attention_mask, input_shape)

        # If a 2D or 3D attention mask is provided for the cross-attention
        # we need to make broadcastable to [batch_size, num_heads, seq_length, seq_length]
        if self.config.is_decoder and encoder_hidden_states is not None:
            encoder_batch_size, encoder_sequence_length, _ = encoder_hidden_states.size()
            encoder_hidden_shape = (encoder_batch_size, encoder_sequence_length)
            if encoder_attention_mask is None:
                encoder_attention_mask = torch.ones(encoder_hidden_shape, device=device)

            if use_sdpa_attention_masks and encoder_attention_mask.dim() == 2:
                # Expand the attention mask for SDPA.
                # [bsz, seq_len] -> [bsz, 1, seq_len, seq_len]
                encoder_extended_attention_mask = _prepare_4d_attention_mask_for_sdpa(
                    encoder_attention_mask, embedding_output.dtype, tgt_len=seq_length
                )
            else:
                encoder_extended_attention_mask = self.invert_attention_mask(encoder_attention_mask)
        else:
            encoder_extended_attention_mask = None

        # Prepare head mask if needed
        # 1.0 in head_mask indicate we keep the head
        # attention_probs has shape bsz x n_heads x N x N
        # input head_mask has shape [num_heads] or [num_hidden_layers x num_heads]
        # and head_mask is converted to shape [num_hidden_layers x batch x num_heads x seq_length x seq_length]
        head_mask = self.get_head_mask(head_mask, self.config.num_hidden_layers)

        encoder_outputs = self.encoder(
            embedding_output,
            router_labels=router_labels,
            padding_embedding=self.embeddings.word_embeddings.weight[0],
            original_attention_mask=attention_mask,
            attention_mask=extended_attention_mask,
            head_mask=head_mask,
            encoder_hidden_states=encoder_hidden_states,
            encoder_attention_mask=encoder_extended_attention_mask,
            past_key_values=past_key_values,
            use_cache=use_cache,
            output_attentions=output_attentions,
            output_hidden_states=output_hidden_states,
            return_dict=return_dict,
        )
        sequence_output = encoder_outputs[0]
        pooled_output = self.pooler(sequence_output) if self.pooler is not None else None

        if not return_dict:
            return (sequence_output, pooled_output) + encoder_outputs[1:]
        
        sequence_output = encoder_outputs[0]
        pooled_output = self.pooler(sequence_output) if self.pooler is not None else None

        return MoEBertOutputWithPoolingAndCrossAttentions(
            last_hidden_state=sequence_output,
            pooler_output=pooled_output,
            hidden_states=encoder_outputs.hidden_states,
            attentions=encoder_outputs.attentions,
            router_logits=encoder_outputs.router_logits
        )


if __name__ == "__main__":
    B, L, H = 2, 10, 4  # for example, 2 samples, 10 tokens, 4 hidden dimensions
    hidden_states = torch.randn(B, L, H)
    
    # Simulate token_type_ids (e.g. groups change when the token type changes)
    token_type_ids = torch.tensor([
        [0, 0, 1, 1, 1, 0, 0, 0, 0, 0],
        [1, 1, 1, 4, 9, 1, 0, 0, 0, 0]
    ])
    
    # Simulate attention mask (1 for real tokens, 0 for padding)
    attention_mask = torch.tensor([
        [1, 1, 1, 1, 1, 0, 0, 0, 0, 0],
        [1, 1, 1, 1, 1, 1, 0, 0, 0, 0]
    ])

    padding_embedding = torch.ones(H)

    print("Hidden states:")
    print(hidden_states)
    
    config = {}  # any config options if needed
    pooling_naive = GroupPoolingBest(config)
    pooling = GroupPooling(config)
    pooled_hidden, new_attention_mask = pooling(hidden_states, attention_mask, token_type_ids, padding_embedding)
    
    print("Pooled hidden states:")
    print(pooled_hidden)
    print("New attention mask:")
    print(new_attention_mask)

    pooled_hidden, new_attention_mask = pooling_naive(hidden_states, attention_mask, token_type_ids, padding_embedding)

    print("Pooled hidden states:")
    print(pooled_hidden)
    print("New attention mask:")
    print(new_attention_mask)