import os
from dataclasses import dataclass
from pathlib import Path
from typing import Optional

import paramiko
from hydra.core.config_store import ConfigStore

def upload_files(
    source_files: list[str],
    target_files: list[str],
) -> None:
    if len(source_files) != len(target_files):
        raise ValueError(
            "The number of source files and target files does not match."
        )
    upload_config = load_upload_config()

    with paramiko.SSHClient() as ssh:
        ssh.load_host_keys(os.path.expanduser(os.path.join(
            "~", ".ssh", "known_hosts"
        )))
        # TODO: this is not secure
        #ssh.set_missing_host_key_policy(paramiko.AutoAddPolicy())

        try:
            ssh.connect(
                upload_config.host_address,
                username=upload_config.username,
                # password=credentials[1]
            )
        except paramiko.AuthenticationException:
            print('Error authenticating to server')
            return

        results_root = Path(upload_config.results_root_dir)
        with ssh.open_sftp() as sftp:
            # Delete the old experiment results directory to avoid cluttering
            # if clear_dir_postfix is not None:
            #     clear_dir = os.path.realpath(os.path.join(
            #             target_dir, clear_dir_postfix))
            #     try:
            #         sftp.rmdir(clear_dir)
            #         print('Deleting directory "{}"'.format(clear_dir))
            #     except (FileNotFoundError, OSError):
            #         pass

            # for src_file, target_file in tqdm.tqdm(zip(
            for src_file, target_file in zip(source_files, target_files):
                target_file = results_root / target_file
                # target_dir = target_file.parent
                # TODO: try to group file together by directories to avoid
                # mkdir calls on existing directories
                # remote_mkdir_recursive(sftp, file_dir, target_dir)
                sftp.put(src_file, str(target_file))

    print("Uploaded files to server.")

                
# def remote_mkdir_recursive(sftp: paramiko.SFTP, directory: Path):
#     cur_dir = base_dir
#     for path_component in path_components:
#         cur_dir = os.path.join(cur_dir, path_component)
#         try:
#             sftp.mkdir(cur_dir)
#         except IOError:
#             # The directory already exists
#             pass


USERNAME_VAR = "UPLOAD_USERNAME"
RESULTS_HOST_VAR = "UPLOAD_RESULTS_HOST"
RESULTS_ROOT_DIR_VAR = "UPLOAD_ROOT_DIR"

@dataclass
class UploadConfig:
    username: str
    host_address: str
    results_root_dir: str

    def to_env_vars(self) -> dict[str, str]:
        return {
            USERNAME_VAR: self.username,
            RESULTS_HOST_VAR: self.host_address,
            RESULTS_ROOT_DIR_VAR: self.results_root_dir,
        }

def register_upload_config() -> None:
    cs = ConfigStore.instance()
    cs.store(
        name="upload",
        node=UploadConfig,
    )

_upload_config: Optional[UploadConfig] = None

def set_upload_config(config: UploadConfig) -> None:
    global _upload_config
    _upload_config = config

def load_upload_config() -> UploadConfig:
    global _upload_config
    if _upload_config is None:
        username = os.environ.get(USERNAME_VAR)
        host_address = os.environ.get(RESULTS_HOST_VAR)
        results_root_dir = os.environ.get(RESULTS_ROOT_DIR_VAR)

        if username is None:
            raise ValueError(
                "Username for upload is not defined. "
                "Make sure you set the environment variable "
                f"'{USERNAME_VAR}'"
            )
        if host_address is None:
            raise ValueError(
                "Host address for upload is not defined. "
                "Make sure you set the environment variable "
                f"'{RESULTS_HOST_VAR}'"
            )
        if results_root_dir is None:
            raise ValueError(
                "Results root directory for upload is not defined. "
                "Make sure you set the environment variable "
                f"'{RESULTS_ROOT_DIR_VAR}'"
            )

        _upload_config = UploadConfig(
            username=username,
            host_address=host_address,
            results_root_dir=results_root_dir,
        )
    return _upload_config
