import google.generativeai as genai
import base64
import os
import yaml
import json
import time
from baseline import Baseline
from typing import List, Tuple, Dict
from itertools import chain, combinations
import base64
from survey_loader import load_survey_questions_independent, load_survey_questions_cot, load_survey_questions_cot_with_gt
import numpy as np
import re
import tqdm

from api_baseline import APIBaseline

class Gemini(APIBaseline):
    def __init__(self, model_name:str='gemini-2.0-flash-exp', api_key_env_var=None):
        super().__init__(model_name=model_name, api_key_env_var=api_key_env_var)
        self.model = genai.GenerativeModel(model_name=model_name)
    
    def encode_image(self, image_path: str):
        """
        Encode an image to base64.

        :param image_path: The path to the image file.
        :return: The base64 encoded string of the image.
        """
        with open(image_path, "rb") as image_file:
            return base64.b64encode(image_file.read()).decode('utf-8')

    def generate_text_individual(self, 
                                 text: str, 
                                 image_filepaths: List[str] = []):
        """
        Generate text based on the provided text and images.
        """
        return self.generate_text_api_call(text, encoded_images=image_filepaths)

    def generate_text_using_past_conversations(self,
                                               text: str,
                                               image_filepaths: List[str] = []):
        """
        Generate text based on the provided text, images, and past conversations.
        """
        return self.generate_text_api_call(text, 
                                           encoded_images=image_filepaths, 
                                           past_conversations=self.past_conversations)

    def generate_text_api_call(self, 
                               text: str, 
                               encoded_images: List[str] = [],
                               past_conversations: List[Tuple[str, str]] = [],
                               include_past_imgs_as_context: bool = True):
        """
        Generate text using the Gemini model with optional images and past conversations.
        This mirrors the structure found in the GPT4o snippet, but uses the Gemini API.
        """
        # Prepare the images in the format expected by Gemini
        images = []
        if encoded_images:
            for base64_image in encoded_images:
                images.append({'mime_type': 'image/jpeg', 'data': base64_image})

        # Combine images and text into a single list (Gemini expects them together)
        input_content = [*images, text]

        try:
            response = self.model.generate_content(input_content)
            
            text = response.text
                    
            def extract_json_string(response):
                # Match and extract JSON-like content
                match = re.search(r'\{.*\}', response, re.DOTALL)
                if match:
                    return match.group()  # Return the JSON string
                else:
                    raise ValueError("No JSON-like content found in response.")
            
            return extract_json_string(text)
        except Exception as e:
            print("Invalid Gemini response (or rate limited), trying again in 10 seconds...", e)
            # wait 10 seconds
            time.sleep(10)
            return self.generate_text_api_call(text, encoded_images, past_conversations)
