import os
import base64
import requests
from api_baseline import APIBaseline
from typing import List, Tuple
import json
import time

class GPT4o(APIBaseline):
    def encode_image(self, 
                     image_path: str):
        """
        Encode an image to base64.

        :param image_path: The path to the image file.
        :return: The base64 encoded string of the image.
        """
        with open(image_path, "rb") as image_file:
            return base64.b64encode(image_file.read()).decode('utf-8')

    def generate_text_individual(self, 
                                 text: str, 
                                 image_filepaths: List[str] = []):
        """
        Generate text based on the provided text and images.

        :param text: The input text to process.
        :param image_filepaths: The optional input image to process.
        :return: The generated text.
        """
        return self.generate_text_api_call(text, image_filepaths)
    
    def generate_text_using_past_conversations(self,
                                                  text: str,
                                                  image_filepaths: List[str] = []):
        """
        Generate text based on the provided text, images, and past conversations.

        :param text: The input text to process.
        :param image_filepaths: The optional input image to process.
        :return: The generated text.
        """
        return self.generate_text_api_call(text, image_filepaths, self.past_conversations)

    def generate_text_api_call(self, 
                      text: str, 
                      encoded_images = [],
                      past_conversations: List[Tuple[str, str]] = [],
                      include_past_imgs_as_context: bool = True):
        """
        Generate text using the OpenAI GPT-4 model with optional images.

        :param text: The input text to process.
        :param images: A list of image file paths to include in the request.
        :return: The generated text from GPT-4.
        """
        # Prepare the headers for the API request
        headers = {
            "Content-Type": "application/json",
            "Authorization": f"Bearer {self.api_key}"
        }

        # Construct the message content
        message_content = [{"type": "text", "text": text}]
        
        if encoded_images:
            for base64_image in encoded_images:
                message_content.append({
                    "type": "image_url",
                    "image_url": {"url": f"data:image/jpeg;base64,{base64_image}"}
                })
                    
        current_message = {"role": "user", "content": message_content}
        
        messages = [current_message] 

        # Prepare the payload for the API request
        payload = {
            "model": f"{self.model_name}",
            "messages": messages,
            # "max_tokens": 50,
            "response_format": {"type": "json_object"}
        }

        # Send the request to the OpenAI API
        response = requests.post(
            "https://api.openai.com/v1/chat/completions",
            headers=headers,
            json=payload
        )
        
        try:
            # Check for errors in the response
            response_data = response.json()
            json.loads(response.json()['choices'][0]['message']['content'])
            if response.status_code != 200:
                error_message = response_data.get('error', {}).get('message', 'Unknown error')
                raise Exception(f"API request failed with status {response.status_code}: {error_message}")
            return response_data['choices'][0]['message']['content']
        except:
            print("Non-JSON response:", response)
            print("Trying again in 10 seconds...")
            time.sleep(10)
            return self.generate_text_api_call(text, encoded_images, past_conversations)

