import os
import re
import json
from typing import List, Tuple, Optional
from PIL import Image, ImageSequence, UnidentifiedImageError


def postprocess_questions_and_fps(
    question_groups: List[List[dict]],
    folderpath: str,
    image_filepaths: List[str],
    main_folder: str
) -> Tuple[List[List[dict]], List[str]]:
    """
    Update question IDs with a running index and normalize image file paths.

    Args:
        question_groups: A list of question group lists, each containing dicts with an 'id' key.
        folderpath: Directory containing the images.
        image_filepaths: List of image file paths to normalize.
        main_folder: Base folder path to be removed from image paths.

    Returns:
        A tuple containing:
            - Updated question_groups with unique IDs.
            - Normalized image_filepaths relative to main_folder.

    Raises:
        AssertionError: If folderpath or main_folder does not exist, or if no image_filepaths provided.
    """
    assert os.path.exists(folderpath), f"Folderpath {folderpath} does not exist."
    assert image_filepaths, "No image filepaths provided."
    assert os.path.exists(main_folder), f"Main folder {main_folder} does not exist."

    # Append a unique running index to each question ID
    counter = 0
    for group in question_groups:
        for q in group:
            q['id'] = f"{q['id']}_{counter}"
            counter += 1

    # Normalize image paths relative to main_folder and remove 'static/' prefix
    normalized_paths = []
    for path in image_filepaths:
        filename = os.path.basename(path)
        rel_path = os.path.relpath(folderpath, main_folder)
        new_path = os.path.join(rel_path, filename).lstrip(os.sep)
        if new_path.startswith('static' + os.sep):
            new_path = new_path[len('static' + os.sep):]
        normalized_paths.append(new_path)

    return question_groups, normalized_paths


def preprocess_composable(
    folderpath: str,
    main_folder: str
) -> Tuple[List[str], str, List[str], str]:
    """
    Collect image file paths, build GIF path, and extract instructions from a description file.

    Args:
        folderpath: Directory containing '.png' frames and a 'descriptions' subfolder in main_folder.
        main_folder: Base folder that contains the 'descriptions' directory.

    Returns:
        - image_filepaths: List of '.png' file paths within folderpath.
        - gif_filepath: Path to 'seq.gif' within folderpath.
        - instructions: List of instruction strings parsed from the corresponding .txt file.
        - question_num: Identifier extracted from the last segment of folderpath.

    Raises:
        AssertionError: If folderpath or instruction file does not exist.
    """
    assert os.path.exists(folderpath), f"Folderpath {folderpath} does not exist."

    # Gather .png images
    image_filepaths = [os.path.join(folderpath, f)
                       for f in os.listdir(folderpath) if f.endswith('.png')]
    gif_filepath = os.path.join(folderpath, 'seq.gif')
    video_filepath = os.path.join(folderpath, 'seq.mp4')

    # Parse folder structure for description lookup
    parts = os.path.normpath(folderpath).split(os.sep)
    # Example: [..., \"avoid_preferright_yield_dynamic\", \"default\", \"11\"]
    instruction_key = parts[-3]
    direction = parts[-2] # 'left', 'right', or 'default'
    question_num = parts[-1]

    desc_path = os.path.join(main_folder, 'descriptions', f"{instruction_key}_{direction}.txt")
    assert os.path.exists(desc_path), f"Instruction file {desc_path} does not exist."

    # Extract instruction lines matching pattern
    instructions: List[str] = []
    pattern = re.compile(r'instruction\s+\d+:\s*(.+)', re.IGNORECASE)
    with open(desc_path, 'r') as f:
        for line in f:
            match = pattern.match(line)
            if match:
                instructions.append(match.group(1).strip())

    return image_filepaths, gif_filepath, video_filepath, instructions, question_num


def preprocess_and_get_imagepaths_and_n_objects(
    folderpath: str,
    main_folder: str
) -> Tuple[List[str], int]:
    """
    Extract frames from a combined GIF and read the number of objects from a JSON file.

    Args:
        folderpath: Directory containing 'sample_with_bev.gif' and 'n_people.json'.
        main_folder: Base folder (unused but kept for signature compatibility).

    Returns:
        - image_filepaths: List of extracted frame file paths.
        - n_objects: Number of people read from 'n_people.json'.

    Raises:
        AssertionError: If folderpath or JSON file does not exist.
    """
    assert os.path.exists(folderpath), f"Folderpath {folderpath} does not exist."

    gif_fp = os.path.join(folderpath, 'sample_with_bev.gif')
    image_filepaths = export_frames_and_get_filepaths(gif_fp)

    n_objects = get_n_objects(folderpath)
    return image_filepaths, n_objects


def get_n_objects(folderpath: str) -> int:
    """
    Read the number of people from 'n_people.json' in the specified folder.

    Args:
        folderpath: Directory containing 'n_people.json'.

    Returns:
        The integer number of people.

    Raises:
        AssertionError: If 'n_people.json' does not exist.
    """
    json_fp = os.path.join(folderpath, 'n_people.json')
    assert os.path.exists(json_fp), f"File {json_fp} does not exist."
    with open(json_fp, 'r') as f:
        data = json.load(f)
    return data.get('n_people', 0)


def export_frames_and_get_filepaths(gif_filepath: str, n_frames: int = 10) -> List[str]:
    """
    Extract up to 'n_frames' frames from an animated GIF, saving them as PNGs if not already present.

    Args:
        gif_filepath: Path to the source GIF file.
        n_frames: Number of frames to extract (default: 10).

    Returns:
        List of file paths for the extracted frames.

    Raises:
        AssertionError: If the GIF file does not exist or is invalid.
    """
    base = gif_filepath.rsplit('.gif', 1)[0]
    frame_paths = [f"{base}_{i}.png" for i in range(n_frames)]
    if not os.path.exists(gif_filepath):
        # load the frames
        frames = []
        for fp in frame_paths:
            frame_full_filepath = os.path.join(os.path.dirname(gif_filepath), fp)
            assert os.path.exists(frame_full_filepath), f"Frame {frame_full_filepath} does not exist."
            frame = Image.open(frame_full_filepath)
            frames.append(frame)
        # Save the frames as a GIF
        frames[0].save(
            gif_filepath,
            save_all=True,
            append_images=frames[1:],
            optimize=True,
            duration=100,
            loop=0
        )

    # Check if frames already exist
    if all(os.path.exists(fp) for fp in frame_paths):
        return frame_paths

    # Extract frames
    with Image.open(gif_filepath) as gif:
        for i, frame in enumerate(ImageSequence.Iterator(gif)):
            if i >= n_frames:
                break
            frame.convert('RGBA').save(frame_paths[i])

    return frame_paths


def merge_imgs_top_down(
    annotated_fp: str,
    topdown_fp: str,
    folderpath: str
) -> Optional[str]:
    """
    Vertically merge an annotated image and a top-down image into one JPG.

    Args:
        annotated_fp: Filename of the annotated image (must end with '.jpg').
        topdown_fp: Filename of the top-down image (must end with '.jpg').
        folderpath: Directory containing both images.

    Returns:
        Path to the combined image, or None on error.
    """
    # Validate inputs
    for fp in (annotated_fp, topdown_fp):
        assert fp.endswith('.jpg'), f"File {fp} must end with '.jpg'"
    assert os.path.isdir(folderpath), f"Folderpath {folderpath} does not exist."

    ann_path = os.path.join(folderpath, annotated_fp)
    top_path = os.path.join(folderpath, topdown_fp)
    assert os.path.exists(ann_path), f"Annotated image {ann_path} missing."
    assert os.path.exists(top_path), f"Top-down image {top_path} missing."

    try:
        ann = Image.open(ann_path)
        top = Image.open(top_path)

        # Resize top-down to 75% of annotated height, maintain aspect ratio
        new_height = int(ann.height * 0.75)
        new_width = int(top.width * (new_height / top.height))
        top_resized = top.resize((new_width, new_height))

        # Pad to match annotated width
        pad = (ann.width - new_width) // 2
        canvas = Image.new('RGB', (ann.width, ann.height), 'white')
        canvas.paste(top_resized, (pad, pad))

        # Combine vertically
        combined = Image.new('RGB', (ann.width, ann.height * 2))
        combined.paste(ann, (0, 0))
        combined.paste(canvas, (0, ann.height))

        output_fp = os.path.join(folderpath, 'combined_topdown.jpg')
        combined.save(output_fp)
        return output_fp

    except (FileNotFoundError, UnidentifiedImageError) as e:
        print(f"Error merging images: {e}")
        return None


def optimize_gif(
    input_path: str,
    output_path: str,
    max_width: Optional[int] = None,
    max_height: Optional[int] = None,
    frame_skip: int = 1
) -> None:
    """
    Optimize an animated GIF by resizing, skipping frames, and adding a pause at the end.

    Args:
        input_path: Path to the source GIF.
        output_path: Destination path for the optimized GIF.
        max_width: Maximum width for thumbnail resizing (optional).
        max_height: Maximum height for thumbnail resizing (optional).
        frame_skip: Skip every Nth frame to reduce size (default: 1).
    """
    try:
        with Image.open(input_path) as im:
            if im.format != 'GIF' or getattr(im, 'n_frames', 1) <= 1:
                print(f"File {input_path} is not a valid animated GIF.")
                return

            frames = []
            for i, frame in enumerate(ImageSequence.Iterator(im)):
                if i % frame_skip:
                    continue
                frm = frame.convert('RGBA')
                if max_width and max_height:
                    frm.thumbnail((max_width, max_height), Image.Resampling.LANCZOS)
                frames.append(frm)

            duration = im.info.get('duration', 100) * frame_skip
            pause_ms = 3000
            extra = pause_ms // im.info.get('duration', 100)
            frames.extend([frames[-1]] * extra)

            frames[0].save(
                output_path,
                save_all=True,
                append_images=frames[1:],
                optimize=True,
                duration=duration,
                loop=0
            )
            print(f"Optimized GIF saved to {output_path}")

    except FileNotFoundError:
        print(f"Input file {input_path} not found.")
    except UnidentifiedImageError:
        print(f"File {input_path} is not a valid image.")


def merge_gifs_top_down(
    annotated_gif: str,
    topdown_gif: str,
    folderpath: str,
    skip_existing: bool = True
) -> Optional[str]:
    """
    Optimize and merge two GIFs: annotated sequence and top-down sequence.

    Args:
        annotated_gif: Filename of the annotated GIF.
        topdown_gif: Filename of the top-down GIF.
        folderpath: Directory containing both GIFs.

    Returns:
        Path to the optimized, merged GIF, or None if merging fails.
    """
    ann_fp = os.path.join(folderpath, annotated_gif)
    top_fp = os.path.join(folderpath, topdown_gif)
    assert ann_fp.endswith('.gif') and top_fp.endswith('.gif'), "Both files must be GIFs."
    assert os.path.isdir(folderpath), f"Folderpath {folderpath} does not exist."

    merged_name = annotated_gif.replace('_annotated_seq.gif', '_optimized.gif')
    output_fp = os.path.join(folderpath, merged_name)

    if skip_existing and os.path.exists(output_fp):
        return output_fp

    optimize_gif(ann_fp, output_fp, max_width=1080, max_height=int(1080 * 9 / 16))
    return output_fp
