
from flask import session
from utils import preprocess_and_get_imagepaths_and_n_objects, postprocess_questions_and_fps


from flask import session
from typing import Dict, Any, List

from utils import (
    preprocess_and_get_imagepaths_and_n_objects,
    postprocess_questions_and_fps,
)


def _format_instructions(raw_text: str) -> str:
    """
    Convert a multiline raw string into HTML by replacing newlines with <br/> tags.

    Args:
        raw_text: The raw instruction text with line breaks.

    Returns:
        HTML-formatted instructions as a single string.
    """
    return raw_text.replace("\n", "<br/>")


def generate_landing_page(
    folderpath: str,
    main_folder: str,
    reset_session: bool = True
) -> Dict[str, Any]:
    """
    Create the context for the survey landing page, including session reset,
    image file paths, and participant instructions.

    Args:
        folderpath: Path to the image sequence folder.
        main_folder: Base project folder for path normalization.
        reset_session: If True, clear any existing 'answers' in the session.

    Returns:
        A dict with keys:
            - image_filepaths: List of relative image paths.
            - header: Page header text.
            - instructions: HTML-formatted participant instructions.
            - questions: List of question groups (initially empty).
            - num_frames: Number of frames in the sequence (default: 10).
    """
    if reset_session:
        session.pop('answers', None)
        
    print(folderpath)

    image_filepaths, n_objects = preprocess_and_get_imagepaths_and_n_objects(
        folderpath, main_folder
    )

    header = 'Navigating Through Crowds Survey Instructions'
    
    raw_instructions = '''
        <b>Title:</b> Navigating Through Crowds Survey
        <b>Primary Investigators:</b> [Redacted for anonymity]
        
        You are invited to be part of a research study. Your participation in this research survey is voluntary. You may choose not to participate. If you decide to participate, you may withdraw from the survey at any time without penalty. 
        
        This is a research survey about walking through crowds. You will be shown sequences of images of a robot navigating around people in both first-person and birds-eye views. The survey consists of multiple choice questions about the behavior of people and robots in a shared space.

        In order to participate, you must be 18 years or older, proficient in English, and able to complete the study on a desktop computer. Participants should not have any visual or cognitive impairments that would affect their ability to complete the questionnaire.
        
        If you have any questions, [Redacted for anonymity].
        
        <b>Survey Description & Instructions:</b>
        You will be shown sequences of images of a robot navigating around people in both first-person and  birds-eye views. 
        
        The slider, located below the views, controls where you are in the sequence. You can use the slider to move back and forth in the sequence. Please feel free to try moving this slider to see how it works.

        The first-person view (located above) is from the robot's perspective. Some visible people in the video have a unique circled number. The robot is generally moving forward but may turn or change its speed.

        You can think of the birds-eye view as a perspective from above, similar to a satellite view. The colored circled numbers correspond to the people in the image above it. The black label R corresponds to where the robot is, and the black label G corresponds to the goal point where the robot plans to go. As you move the slider, you may notice the bird's-eye view is not perfect, but is what the robot would see from above.
    '''
    
    instructions = _format_instructions(raw_instructions)

    # No questions on landing page; questions will be populated later
    question_groups: List[List[Dict[str, Any]]] = []

    # Normalize any question IDs and image paths
    question_groups, image_filepaths = postprocess_questions_and_fps(
        question_groups, folderpath, image_filepaths, main_folder
    )

    return {
        "image_filepaths": image_filepaths,
        "header": header,
        "instructions": instructions,
        "questions": question_groups,
        "num_frames": 10,
    }


def _build_question_groups(
    n_objects: int,
    definitions: List[Dict[str, Any]]
) -> List[List[Dict[str, Any]]]:
    """
    Helper to assemble question groups based on object count and field definitions.

    Args:
        n_objects: Number of people in the scene.
        definitions: A list of question templates for each person.

    Returns:
        A list of question groups, each group is a list of question dicts.
    """
    groups: List[List[Dict[str, Any]]] = []
    for template in definitions:
        group: List[Dict[str, Any]] = []
        for i in range(n_objects):
            question = template.copy()
            question['row'] = f"person {i+1}"
            question['id'] = f"{template['id']}_p{i+1}"
            group.append(question)
        groups.append(group)
    return groups


def generate_question_set_1(
    folderpath: str,
    main_folder: str
) -> Dict[str, Any]:
    """
    Questions about robot and people movements in the sequence.

    Returns a dict with page context for set 1.
    """
    image_filepaths, n_objects = preprocess_and_get_imagepaths_and_n_objects(
        folderpath, main_folder
    )

    header = 'Movements'
    raw_instructions = '''
    <i>Instructions:</i> This series of questions... (full text omitted)
    '''
    instructions = _format_instructions(raw_instructions)

    # Define question templates
    templates = [
        {
            'question': 'The robot is _____. (Select all that apply)',
            'type': 'multiple_select',
            'choices': ['moving ahead', 'turning left', 'turning right'],
            'id': 'q_robot_moving_direction',
        }
    ] + [
        {
            'question': 'In the beginning, person X is _____ the robot.',
            'type': 'multiple_choice_matrix',
            'choices': ['ahead of', 'to the left of', 'to the right of', 'behind'],
            'id': 'q_person_spatial_position_begin',
        },
        {
            'question': 'At the end, person X is _____ the robot.',
            'type': 'multiple_choice_matrix',
            'choices': ['ahead of', 'to the left of', 'to the right of', 'behind'],
            'id': 'q_person_spatial_position_end',
        },
        {
            'question': 'Person X ends up _____ compared to the beginning.',
            'type': 'multiple_choice_matrix',
            'choices': ['closer to', 'further away from', 'about the same distance to'],
            'id': 'q_person_distance_change',
        },
    ]

    question_groups = _build_question_groups(n_objects, templates)

    question_groups, image_filepaths = postprocess_questions_and_fps(
        question_groups, folderpath, image_filepaths, main_folder
    )

    return {
        "image_filepaths": image_filepaths,
        "header": header,
        "instructions": instructions,
        "questions": question_groups,
        "num_frames": 10,
    }

def get_all_questions(
    folderpath: str,
    main_folder: str
) -> List[Dict[str, Any]]:
    """
    Aggregate all four question sets into a single list.
    """
    return [
        generate_question_set_1(folderpath, main_folder),
        generate_question_set_2(folderpath, main_folder),
        generate_question_set_3(folderpath, main_folder),
        generate_question_set_4(folderpath, main_folder),
    ]


def generate_question_set_1(folderpath, main_folder):
    
    image_filepaths, n_objects = preprocess_and_get_imagepaths_and_n_objects(folderpath, main_folder)

    header = 'Movements'
    instructions = '''
    <i>Instructions:</i> This series of questions will ask you about the locations and movements of the robot and people shown in the video.
    
    If you can't see the person at the start of the video, please estimate their starting location based on the first seen location. For the end location, please do the same based on the last seen location.
    
    '''
    
    instructions = instructions.replace('\n', '<br/>')

    question_groups = []
    beginning_pos_questions = []
    end_pos_questions = []
    distance_change_questions = []
    
    robot_direction_questions = [{"description": "",
                    "question": "The robot is _____. (Select all that apply)",
                    "type": "multiple_select",
                    "choices": ["moving ahead", "turning left", "turning right"],
                    "id": "q_robot_moving_direction"}]
    
    for i in range(n_objects):
        beginning_pos_questions.append({"description": "",
                        "question": "In the beginning, person X" + " is _____ the robot.",
                        "type": "multiple_choice_matrix",
                        "row": "person " + str(i + 1),
                        "choices": ["ahead of", "to the left of", "to the right of", "behind"],
                          "id": "q_person_spatial_position_begin_p" + str(i + 1)})
        end_pos_questions.append({"description": "",
                          "question": "At the end, person X " + " is _____ the robot.",
                          "type": "multiple_choice_matrix",
                          "row": "person " + str(i + 1),
                          "choices": ["ahead of", "to the left of", "to the right of", "behind"],
                          "id": "q_person_spatial_position_end_p" + str(i + 1)})
        distance_change_questions.append({"description": "",
                          "question": "At the end, person X " + " ends up _____ the robot compared to the beginning.",
                          "type": "multiple_choice_matrix",
                          "row": "person " + str(i + 1),
                          "choices": ["closer to", "further away from", "about the same distance to"],
                          "id": "q_person_distance_change_p" + str(i + 1)})

    question_groups.append(robot_direction_questions)
    question_groups.append(beginning_pos_questions)
    question_groups.append(end_pos_questions)
    question_groups.append(distance_change_questions)

    question_groups, image_filepaths = postprocess_questions_and_fps(question_groups, folderpath, image_filepaths, main_folder)

    return {"image_filepaths": image_filepaths, "header": header, "instructions": instructions, "questions": question_groups, "num_frames": 10}


def generate_question_set_2(
    folderpath: str,
    main_folder: str
) -> Dict[str, Any]:
    """
    Questions about the goal location and path obstructions.
    """
    image_filepaths, n_objects = preprocess_and_get_imagepaths_and_n_objects(
        folderpath, main_folder
    )

    header = 'Goal Location'
    raw_instructions = '''
    <i>Instructions:</i> The goal location is the place where the robot would like to make progress towards, denoted by G in the bird's-eye view.
    (full text omitted)
    '''
    instructions = _format_instructions(raw_instructions)

    # Single-instance question groups
    beginning_goal = [
        {
            'question': 'In the beginning, the goal is ___ of the robot.',
            'type': 'multiple_choice',
            'choices': ['ahead', 'to the left', 'to the right'],
            'id': 'q_goal_position_begin',
        }
    ]
    end_goal = [
        {
            'question': 'At the end, the goal is ___ of the robot.',
            'type': 'multiple_choice',
            'choices': ['ahead', 'to the left', 'to the right'],
            'id': 'q_goal_position_end',
        }
    ]

    # Per-person obstruction templates
    obstruct_tmpl = {
        'question': "Is person X's path in the way of the robot's path to the goal?",
        'type': 'multiple_choice_matrix',
        'choices': ['yes', 'no'],
        'id': 'q_obstructing_path',
    }
    obstruct_end_tmpl = {
        'question': "At the end, is person X's position in the way of the robot's path to the goal?",
        'type': 'multiple_choice_matrix',
        'choices': ['yes', 'no'],
        'id': 'q_obstructing_end_position',
    }

    question_groups = [
        beginning_goal,
        end_goal,
        *_build_question_groups(n_objects, [obstruct_tmpl]),
        *_build_question_groups(n_objects, [obstruct_end_tmpl]),
    ]
    question_groups, image_filepaths = postprocess_questions_and_fps(
        question_groups, folderpath, image_filepaths, main_folder
    )

    return {
        "image_filepaths": image_filepaths,
        "header": header,
        "instructions": instructions,
        "questions": question_groups,
        "num_frames": 10,
    }


def generate_question_set_3(
    folderpath: str,
    main_folder: str
) -> Dict[str, Any]:
    """
    Questions about navigation actions between robot and people.
    """
    image_filepaths, n_objects = preprocess_and_get_imagepaths_and_n_objects(
        folderpath, main_folder
    )

    header = 'Navigation Actions'
    raw_instructions = '''
    <i>Instructions:</i> The following questions will ask about determining the most likely navigation actions taken by the people and robot.
    (full text omitted)
    '''
    instructions = _format_instructions(raw_instructions)

    action_tmpls = [
        {
            'question': "Is the robot's movement affected by person X?",
            'type': 'multiple_choice_matrix',
            'choices': ['yes', 'no'],
            'id': 'q_robot_affected',
        },
        {
            'question': 'The robot is most likely ____ person X.',
            'type': 'multiple_choice_matrix',
            'choices': ['avoiding', 'overtaking', 'not considering', 'following', 'yielding to'],
            'id': 'q_robot_action',
        },
        {
            'question': "Is person X's movement affected by the robot?",
            'type': 'multiple_choice_matrix',
            'choices': ['yes', 'no'],
            'id': 'q_person_affected',
        },
        {
            'question': 'Person X is most likely ___ the robot.',
            'type': 'multiple_choice_matrix',
            'choices': ['avoiding', 'overtaking', 'not considering', 'following', 'yielding to'],
            'id': 'q_person_action',
        },
    ]
    question_groups = _build_question_groups(n_objects, action_tmpls)
    question_groups, image_filepaths = postprocess_questions_and_fps(
        question_groups, folderpath, image_filepaths, main_folder
    )

    return {
        "image_filepaths": image_filepaths,
        "header": header,
        "instructions": instructions,
        "questions": question_groups,
        "num_frames": 10,
    }


def generate_question_set_4(
    folderpath: str,
    main_folder: str
) -> Dict[str, Any]:
    """
    Questions about suggested future navigation actions.
    """
    image_filepaths, n_objects = preprocess_and_get_imagepaths_and_n_objects(
        folderpath, main_folder
    )

    header = 'Suggested Future Navigation Actions'
    raw_instructions = '''
    <i>Instructions:</i> The following questions will ask about what you think the robot should do next in the scene.
    (full text omitted)
    '''
    instructions = _format_instructions(raw_instructions)

    future_tmpls = [
        {
            'question': "In the future, should the robot's movement towards the goal be affected by person X?",
            'type': 'multiple_choice_matrix',
            'choices': ['yes', 'no'],
            'id': 'q_robot_suggested_affected',
        },
        {
            'question': 'In the future, the robot should ____ person X as it makes progress towards the goal.',
            'type': 'multiple_choice_matrix',
            'choices': ['avoid', 'overtake', 'not consider', 'follow', 'yield to'],
            'id': 'q_robot_suggested_action',
        },
        {
            'question': 'In the future, person X will most likely ____ the robot as it attempts to make progress.',
            'type': 'multiple_choice_matrix',
            'choices': ['avoid', 'overtake', 'not consider', 'follow', 'yield to'],
            'id': 'q_human_future_action_prediction',
        },
    ]
    question_groups = _build_question_groups(n_objects, future_tmpls)
    question_groups, image_filepaths = postprocess_questions_and_fps(
        question_groups, folderpath, image_filepaths, main_folder
    )

    return {
        "image_filepaths": image_filepaths,
        "header": header,
        "instructions": instructions,
        "questions": question_groups,
        "num_frames": 10,
    }


def get_all_questions(
    folderpath: str,
    main_folder: str
) -> List[Dict[str, Any]]:
    """
    Aggregate all four question sets into a single list.
    """
    return [
        generate_question_set_1(folderpath, main_folder),
        generate_question_set_2(folderpath, main_folder),
        generate_question_set_3(folderpath, main_folder),
        generate_question_set_4(folderpath, main_folder),
    ]
