import torch
import torch.nn as nn
import torch.nn.functional as F
from layers.Transformer_EncDec import Encoder, EncoderLayer
from layers.SelfAttention_Family_mask import FullAttention, AttentionLayer
from layers.Embed import DataEmbedding_inverted
import numpy as np

def transform_corr(corr,s,b):
    corr = torch.FloatTensor(corr-corr.mean()).to(s.device)
    scale = torch.exp(s)
    M = 1/(1+torch.exp(-scale*corr+b))
    return M

class Model(nn.Module):
    def __init__(self, configs):
        super(Model, self).__init__()
        self.seq_len = configs.seq_len
        self.pred_len = configs.pred_len
        self.output_attention = configs.output_attention
        self.use_norm = configs.use_norm

        self.enc_embedding = DataEmbedding_inverted(configs.seq_len, configs.d_model, configs.embed, configs.freq,
                                                    configs.dropout)
        self.class_strategy = configs.class_strategy

        self.encoder = Encoder(
            [
                EncoderLayer(
                    AttentionLayer(
                        FullAttention(False, configs.factor, attention_dropout=configs.dropout,
                                      output_attention=configs.output_attention), configs.d_model, configs.n_heads),
                    configs.d_model,
                    configs.d_ff,
                    dropout=configs.dropout,
                    activation=configs.activation
                ) for l in range(configs.e_layers)
            ],
            norm_layer=torch.nn.LayerNorm(configs.d_model)
        )
        self.projector = nn.Linear(configs.d_model, configs.pred_len, bias=True)
        
        self.alpha_ds = nn.Parameter(torch.Tensor([1.0])*configs.init_alpha)
        self.beta_ds = nn.Parameter(torch.Tensor([0.0]))
        self.channel_mask = np.abs(np.load(f'corr/{configs.dataset}_corr.npy'))

    def forecast(self, x_enc, x_mark_enc, x_dec, x_mark_dec, attn_mask):
        if self.use_norm:
            means = x_enc.mean(1, keepdim=True).detach()
            x_enc = x_enc - means
            stdev = torch.sqrt(torch.var(x_enc, dim=1, keepdim=True, unbiased=False) + 1e-5)
            x_enc /= stdev

        _, _, N = x_enc.shape # B L N
        enc_out = self.enc_embedding(x_enc, x_mark_enc) 
        enc_out, attns = self.encoder(enc_out, attn_mask=attn_mask)
        dec_out = self.projector(enc_out).permute(0, 2, 1)[:, :, :N]

        if self.use_norm:
            dec_out = dec_out * (stdev[:, 0, :].unsqueeze(1).repeat(1, self.pred_len, 1))
            dec_out = dec_out + (means[:, 0, :].unsqueeze(1).repeat(1, self.pred_len, 1))

        return dec_out


    def forward(self, x_enc, x_mark_enc, x_dec, x_mark_dec):
        attn_mask = transform_corr(self.channel_mask,self.alpha_ds,self.beta_ds)
        dec_out = self.forecast(x_enc, x_mark_enc, x_dec, x_mark_dec, attn_mask)
        return dec_out[:, -self.pred_len:, :]  