"""Defines the Abstract Base class for an "Environment".

NOTE (@<anonymous>): This 'Environment' abstraction isn't super useful at the moment
because there's only the `ActiveDataLoader` that fits this interface (since we
can't send anything to the usual DataLoader).
"""
from abc import ABC, ABCMeta, abstractmethod
from dataclasses import dataclass
from typing import Generic, Iterable, Optional, Type, TypeVar

import gym
import numpy as np
from gym import spaces
from sequoia.common.batch import Batch
from sequoia.utils.logging_utils import get_logger
from torch import Tensor
from torch.utils.data import DataLoader

from .objects import (
    Actions,
    ActionType,
    Observations,
    ObservationType,
    Rewards,
    RewardType,
)

logger = get_logger(__file__)


class EnvironmentMeta(ABCMeta, Type["Environment"]):
    # TODO: Use something like this so that we can do:
    # isinstance(<EnvironmentProxy[env_type]>, <env_type>) -> True
    def __instancecheck__(self, instance):
        if hasattr(instance, "_environment_type"):
            # If the env is a proxy, then we check if it is a proxy to an env of this
            # type.
            return issubclass(instance._environment_type, self)
        return super().__instancecheck__(instance)


class Environment(
    gym.Env,
    Generic[ObservationType, ActionType, RewardType],
    ABC,
    metaclass=EnvironmentMeta,
):
    """ABC for a learning 'environment' in Reinforcement or Supervised Learning.

    Different settings can implement this interface however they want.
    """

    reward_space: gym.Space
    # TODO: This is currently changing. We don't really need to force the envs
    # on the RL branch to also be iterables/dataloaders, only those on the
    # supervised learning branch. Rather than force RL/active setups to adopt
    # something like __iter__ and send, we can adapt 'active' datasets to be gym
    # Envs instead.
    # The reason why I was considering doing it that way would have been so we
    # could use pytorch lightning Trainers and the other "facilities" meant for
    # supervised learning in RL.

    # @abstractmethod
    # def __iter__(self) -> Iterable[ObservationType]:
    #     """ Returns a generator yielding observations and accepting actions. """

    # @abstractmethod
    # def send(self, action: ActionType) -> RewardType:
    #     """ Send an action to the environment, and returns the corresponding reward. """

    # TODO: (@<anonymous>): Not sure if we really want/need an abstract __next__.
    # @abstractmethod
    # def __next__(self) -> ObservationType:
    #     """ Generate the next observation. """
