from .utils import EFATLayerBuilder as LayerBuilder
from .layers import *
import torch
import torch.nn as nn
import torch.nn.functional as F
from torchvision.models import ResNet50_Weights, ResNet18_Weights


def conv3x3(
    in_planes,
    out_planes,
    stride=1,
    groups=1,
    dilation=1,
    graphly_connected=False,
    **EFAT_kwargs,
):
    """3x3 convolution with padding"""
    if graphly_connected:
        return EFATConv(
            in_planes,
            out_planes,
            nn.Conv2d,
            {
                'kernel_size':3,
                'stride':stride,
                'padding':dilation,
                'groups':groups,
                'bias':False,
                'dilation':dilation,
            },
            **EFAT_kwargs,
        )
    else:
        return nn.Conv2d(
            in_planes, 
            out_planes,
            kernel_size=3,
            stride=stride,
            padding=dilation,
            groups=groups,
            bias=False,
            dilation=dilation
        )


def conv1x1(in_planes, out_planes, stride=1, graphly_connected=False, **EFAT_kwargs):
    """1x1 convolution"""
    if graphly_connected:
        return EFATConv(
            in_planes,
            out_planes,
            nn.Conv2d, 
            {
                'kernel_size':1,
                'stride':stride,
                'bias':False
            },
            **EFAT_kwargs
        )
    else:
        return nn.Conv2d(
            in_planes,
            out_planes,
            kernel_size=1,
            stride=stride,
            bias=False,
        )


class BasicBlock(nn.Module):
    expansion = 1

    def __init__(
        self,
        inplanes,
        planes,
        stride = 1,
        downsample = None,
        groups = 1,
        base_width = 64,
        dilation = 1,
        norm_layer = None,
        graphly_connected = False,
    ):
        super().__init__()
        self.gc = graphly_connected

        if norm_layer is None:
            norm_layer = nn.BatchNorm2d
        if groups != 1 or base_width != 64:
            raise ValueError("BasicBlock only supports groups=1 and base_width=64")
        if dilation > 1:
            raise NotImplementedError("Dilation > 1 not supported in BasicBlock")
        
        self.conv1 = conv3x3(inplanes,
                             planes,
                             stride,
                             heads=8,
                             dropout=0.2,
                             graphly_connected=self.gc)
        self.bn1 = norm_layer(planes)
        self.relu = nn.ReLU(inplace=True)
        self.conv2 = conv3x3(planes,
                             planes,
                             heads=8,
                             dropout=0.2,
                             graphly_connected=self.gc)
        self.bn2 = norm_layer(planes)
        self.downsample = downsample
        self.stride = stride

    def forward(self, batch):
        x, edges = batch
        identity = x

        if self.gc:
            out = self.conv1(batch)
        else:
            out = self.conv1(x)
        
        out = self.bn1(out)
        out = self.relu(out)

        if self.gc:
            out = self.conv2((out, edges))
        else:
            out = self.conv2(out)
        
        out = self.bn2(out)

        if self.downsample is not None:
            if self.gc:
                identity = self.downsample(batch)
            else:
                identity = self.downsample(x)

        out += identity
        out = self.relu(out)

        return out, edges


class Bottleneck(nn.Module):
    expansion = 4

    def __init__(
        self,
        inplanes,
        planes,
        stride = 1,
        downsample = None,
        groups = 1,
        base_width = 64,
        dilation = 1,
        norm_layer = None,
        graphly_connected = False,
    ):
        super().__init__()
        self.gc = graphly_connected

        if norm_layer is None:
            norm_layer = nn.BatchNorm2d
        
        width = int(planes * (base_width / 64.0)) * groups

        self.conv1 = conv1x1(inplanes, width, graphly_connected=self.gc)
        self.bn1 = norm_layer(width)
        self.conv2 = conv3x3(width, width, stride, groups, dilation, graphly_connected=self.gc)
        self.bn2 = norm_layer(width)
        self.conv3 = conv1x1(width, planes * self.expansion, graphly_connected=self.gc)
        self.bn3 = norm_layer(planes * self.expansion)
        self.relu = nn.ReLU(inplace=True)
        self.downsample = downsample
        self.stride = stride

    def forward(self, batch):
        x, edges = batch
        identity = x

        if self.gc:
            out = self.conv1(batch)
        else:
            out = self.conv1(x)
        
        out = self.bn1(out)
        out = self.relu(out)

        if self.gc:
            out = self.conv2((out, edges))
        else:
            out = self.conv2(out)
        
        out = self.bn2(out)
        out = self.relu(out)

        if self.gc:
            out = self.conv3((out, edges))
        else:
            out = self.conv3(out)
        
        out = self.bn3(out)

        if self.downsample is not None:
            if self.gc:
                identity = self.downsample((x, edges))
            else:
                identity = self.downsample(x)

        out += identity
        out = self.relu(out)

        return out, edges


class ResNetEFG(nn.Module):
    def __init__(
        self,
        block,
        layers,
        connections,
        num_classes = 7,
        zero_init_residual = False,
        groups = 1,
        width_per_group = 64,
        replace_stride_with_dilation = None,
        norm_layer = None,
    ):
        super().__init__()
        if norm_layer is None:
            norm_layer = nn.BatchNorm2d
        self._norm_layer = norm_layer

        self.inplanes = 64
        self.dilation = 1
        if replace_stride_with_dilation is None:
            replace_stride_with_dilation = [False, False, False]
        if len(replace_stride_with_dilation) != 3:
            raise ValueError(
                "replace_stride_with_dilation should be None "
                f"or a 3-element tuple, got {replace_stride_with_dilation}"
            )
        self.groups = groups
        self.base_width = width_per_group

#         self.conv1 = EFATConv(
#             3,
#             self.inplanes,
#             nn.Conv2d,
#             {
#                 'kernel_size':7,
#                 'stride':2,
#                 'padding':3,
#                 'bias':False,
#             },
#             heads=8,
#             dropout=0.2,
#         )
        self.conv1 = nn.Conv2d(
            3, self.inplanes, kernel_size=7, stride=2, padding=3, bias=False
        )

        self.bn1 = norm_layer(self.inplanes)
        self.relu = nn.ReLU(inplace=True)
        
#         self.maxpool = EFATPool(
#             self.inplanes,
#             nn.MaxPool2d,
#             {
#                 'kernel_size':3,
#                 'stride':2,
#                 'padding':1,
#             },
#             heads=8,
#             dropout=0.2,
#         )
        self.maxpool = nn.MaxPool2d(kernel_size=3, stride=2, padding=1)

        self.layer1 = self._make_layer(block, 64, layers[0], connections[0])
        self.layer2 = self._make_layer(
            block, 128, layers[1], connections[1], stride=2, dilate=replace_stride_with_dilation[0]
        )
        self.layer3 = self._make_layer(
            block, 256, layers[2], connections[2], stride=2, dilate=replace_stride_with_dilation[1]
        )
        self.layer4 = self._make_layer(
            block, 512, layers[3], connections[3], stride=2, dilate=replace_stride_with_dilation[2]
        )
      
        self.avgpool = EFATPool(
            512 * block.expansion,
            nn.AdaptiveAvgPool2d, 
            {'output_size':(1, 1)},
            heads=8, 
            dropout=0.2
        )
        self.fc = EFATDense(
            512 * block.expansion,
            num_classes,
            nn.Linear,
            {},
            heads=1,
            dropout=0.2,
        )

        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight, mode="fan_out", nonlinearity="relu")
            elif isinstance(m, (nn.BatchNorm2d, nn.GroupNorm)):
                nn.init.constant_(m.weight, 1)
                nn.init.constant_(m.bias, 0)

        if zero_init_residual:
            for m in self.modules():
                if isinstance(m, Bottleneck) and m.bn3.weight is not None:
                    nn.init.constant_(m.bn3.weight, 0)  # type: ignore[arg-type]
                elif isinstance(m, BasicBlock) and m.bn2.weight is not None:
                    nn.init.constant_(m.bn2.weight, 0)  # type: ignore[arg-type]

    def _make_layer(
        self,
        block,
        planes,
        blocks,
        connected,
        stride=1,
        dilate=False,
    ):
        self.gc = connected

        norm_layer = self._norm_layer
        downsample = None
        previous_dilation = self.dilation
        if dilate:
            self.dilation *= stride
            stride = 1
        if stride != 1 or self.inplanes != planes * block.expansion:
            downsample = nn.Sequential(
                conv1x1(
                    self.inplanes,
                    planes * block.expansion,
                    stride,
                    heads=8,
                    dropout=0.2,
                    graphly_connected=self.gc,
                ),
                norm_layer(planes * block.expansion),
            )

        layers = []
        layers.append(
            block(
                self.inplanes,
                planes,
                stride,
                downsample,
                self.groups,
                self.base_width,
                previous_dilation,
                norm_layer,
                graphly_connected=self.gc,
            )
        )
        self.inplanes = planes * block.expansion
        for _ in range(1, blocks):
            layers.append(
                block(
                    self.inplanes,
                    planes,
                    groups=self.groups,
                    base_width=self.base_width,
                    dilation=self.dilation,
                    norm_layer=norm_layer,
                    graphly_connected=self.gc,
                )
            )

        return nn.Sequential(*layers)

    def _forward_impl(self, batch):
        x, edges = batch

#         breakpoint()
        x = self.conv1(x)
        x = self.bn1(x)
        x = self.relu(x)
        x = self.maxpool(x)
        
        x = self.layer1((x, edges))
        x = self.layer2(x)
        x = self.layer3(x)
        x = self.layer4(x)

        x = self.avgpool(x)
        x = torch.flatten(x, 1)

        x = self.fc((x, edges))

        return x

    def forward(self, batch):
        x, edges = batch.x, batch.edge_index
        return self._forward_impl((x, edges))


def _adjust_state_dict(weight_state_dict, model_state_dict):
    new_state_dict = {}
    connected_layers = []

    # Adding weights not present in weight_state
    for name, weight in model_state_dict.items():
        if name not in weight_state_dict:
            new_state_dict[name] = torch.clone(weight)

            # If a layer requires attention then the layer is connected
            # If a layer is connected special copying is required
            parts = name.split('.')
            print(name, ":", parts)
            if "att" in parts:
                connected_layer = '.'.join(parts[:-1]) + ".weight"
                connected_layers.append(connected_layer)

    print(connected_layers)

    # Adding pretrained weights
    for name, weight in weight_state_dict.items():
        if "fc" not in name:
            if name in connected_layers:
                parts = name.split('.')
                # Duplicating weights
                fun = '.'.join(parts[:-1]) + ".fun.weight"
    #             att = '.'.join(parts[:-1]) + ".att.weight"
    #             bias = '.'.join(parts[:-1]) + ".att.bias"
                
                new_state_dict[fun] = torch.clone(weight)

                # Retaining normal pre initialised weights for attention
    #             new_state_dict[att] = torch.clone(model_state_dict[att])
    #             new_state_dict[bias] = torch.clone(model_state_dict[bias])
            else:
                new_state_dict[name] = torch.clone(weight)

    return new_state_dict


def _resnet(
    block,
    layers,
    connections,
    weights,
    progress,
    pretrained=None,
    **kwargs,
):
    model = ResNetEFG(block, layers, connections, **kwargs)

    if pretrained is not None:
        model.load_state_dict(pretrained)
    elif weights is not None:
        model.load_state_dict(
            _adjust_state_dict(weights.get_state_dict(progress=progress), model.state_dict())
        )

        for name, param in model.named_parameters():
            param.requires_grad = False
            parts = name.split('.')
            if "att" in parts or "fc" in parts:
                param.requires_grad = True

    for name, param in model.named_parameters():
        print(name, ":", param.requires_grad)

    return model


def resnet18EFG(config, num_classes, **kwargs):
    weights = None
    if config['pretrained']:
        weights = ResNet18_Weights.DEFAULT

    pretrained = None
    if 'weights' in config:
        pretrained = torch.load(config['weights'])

    return _resnet(BasicBlock,
                   [2, 2, 2, 2],
                   config['connections'],
                   weights,
                   config['progress'],
                   pretrained=pretrained,
                   **kwargs)

def resnet50EFG(config, num_classes, **kwargs):
    weights = None
    if config['pretrained']:
        weights = ResNet50_Weights.DEFAULT

    pretrained = None
    if 'weights' in config:
        pretrained = torch.load(config['weights'])

    return _resnet(Bottleneck,
                   [3, 4, 6, 3],
                   config['connections'],
                   weights,
                   config['progress'],
                   pretrained=pretrained,
                   **kwargs)

