using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using System.IO;

public class CameraCapture : MonoBehaviour
{
    public Camera targetCamera;   // 绑定要捕获的相机
    public string baseSavePath = "capture";  // 基础保存路径（不含序号和后缀）
    public bool matchGameResolution = false; // 是否匹配游戏窗口分辨率

    private int captureCount = 0;
    private RenderTexture renderTexture;
    private Texture2D captureTexture;

    [Header("固定分辨率设置（matchGameResolution为false时生效）")]
    public int width = 1920;
    public int height = 800;

    void Start()
    {
        // 确保目标相机不为空（默认使用当前物体的相机）
        if (targetCamera == null)
            targetCamera = GetComponent<Camera>();

        // 初始化渲染纹理和捕获纹理
        InitializeTextures();
    }

    // 初始化纹理资源（修复sRGB只读问题）
    private void InitializeTextures()
    {
        // 根据设置确定分辨率
        int finalWidth = width;
        int finalHeight = height;
        if (matchGameResolution)
        {
            finalWidth = Screen.width;
            finalHeight = Screen.height;
        }

        // 创建RenderTexture（移除sRGB赋值，避免编译错误）
        renderTexture = new RenderTexture(finalWidth, finalHeight, 24, RenderTextureFormat.ARGB32)
        {
            wrapMode = TextureWrapMode.Clamp,
            filterMode = FilterMode.Point, // 禁用插值，确保像素精准
            antiAliasing = 1, // 禁用抗锯齿
            useMipMap = false // 禁用Mipmap
        };
        renderTexture.Create();

        // 绑定到目标相机
        targetCamera.targetTexture = renderTexture;

        // 创建Texture2D（与RenderTexture同格式，避免通道转换损失）
        captureTexture = new Texture2D(finalWidth, finalHeight, TextureFormat.ARGB32, false)
        {
            wrapMode = TextureWrapMode.Clamp,
            filterMode = FilterMode.Point
        };
    }

    void Update()
    {
        // 按空格键触发截图（保留原有逻辑）
        if (Input.GetKeyDown(KeyCode.Space))
        {
            SaveFrame();
        }
    }

    // 保存当前帧画面
    void SaveFrame()
    {
        // 保存当前激活的RenderTexture，用于后续恢复
        RenderTexture previousActiveRT = RenderTexture.active;

        try
        {
            // 激活目标渲染纹理并触发相机渲染
            RenderTexture.active = renderTexture;
            targetCamera.Render();

            // 读取像素
            captureTexture.ReadPixels(new Rect(0, 0, renderTexture.width, renderTexture.height), 0, 0);
            captureTexture.Apply();

            // 编码为PNG并保存
            byte[] pngData = captureTexture.EncodeToPNG();
            string savePath = $"{baseSavePath}_{captureCount}.png";
            File.WriteAllBytes(savePath, pngData);

            Debug.Log($"图像已保存到: {Path.GetFullPath(savePath)}");
            captureCount++;
        }
        finally
        {
            // 恢复原有激活的RenderTexture
            RenderTexture.active = previousActiveRT;
        }
    }

    // 清理资源
    void OnDestroy()
    {
        if (renderTexture != null)
        {
            if (targetCamera != null)
                targetCamera.targetTexture = null;

            RenderTexture.active = null;
            Destroy(renderTexture);
        }

        if (captureTexture != null)
        {
            Destroy(captureTexture);
        }
    }

    // 分辨率变更时重新初始化纹理
    void OnRectTransformDimensionsChange()
    {
        if (matchGameResolution && Application.isPlaying)
        {
            OnDestroy();
            InitializeTextures();
        }
    }
}
