using UnityEngine;
using System.IO;
using System.Text;

[System.Serializable]
public class CameraIntrinsics
{
    public int width;
    public int height;
    public float fx;
    public float fy;
    public float cx;
    public float cy;
    public float nearClip;
    public float farClip;
    public float fov;
}

public class CaptureRGBD : MonoBehaviour
{
    public Camera cam;
    public Shader depthShader;  // 指定 DepthShader.shader
    public int width = 640;
    public int height = 480;
    public string saveDir = "CapturedData";

    void Start()
    {
        if (!Directory.Exists(saveDir)) Directory.CreateDirectory(saveDir);
        Capture();
    }

    void Capture()
    {
        // --- RGB ---
        RenderTexture rgbRT = new RenderTexture(width, height, 24, RenderTextureFormat.ARGB32);
        cam.targetTexture = rgbRT;
        Texture2D rgbTex = new Texture2D(width, height, TextureFormat.RGB24, false);
        cam.Render();
        RenderTexture.active = rgbRT;
        rgbTex.ReadPixels(new Rect(0, 0, width, height), 0, 0);
        rgbTex.Apply();
        File.WriteAllBytes(Path.Combine(saveDir, "rgb.png"), rgbTex.EncodeToPNG());

        // --- Depth ---
        RenderTexture depthRT = new RenderTexture(width, height, 24, RenderTextureFormat.RFloat);
        cam.targetTexture = depthRT;
        cam.SetReplacementShader(depthShader, "");  // 使用自定义深度shader
        Texture2D depthTex = new Texture2D(width, height, TextureFormat.RFloat, false);
        cam.Render();
        RenderTexture.active = depthRT;
        depthTex.ReadPixels(new Rect(0, 0, width, height), 0, 0);
        depthTex.Apply();

        float[] depthArray = new float[width * height];
        for (int y = 0; y < height; y++)
        {
            for (int x = 0; x < width; x++)
            {
                float d = depthTex.GetPixel(x, y).r;
                depthArray[y * width + x] = d;
            }
        }

        using (var fs = new FileStream(Path.Combine(saveDir, "depth.npy"), FileMode.Create))
        using (var bw = new BinaryWriter(fs))
        {
            foreach (float val in depthArray)
                bw.Write(val);
        }

        // --- Intrinsics ---
        float fov = cam.fieldOfView * Mathf.Deg2Rad;
        float fx = (width / 2f) / Mathf.Tan(fov / 2f);
        float fy = fx; // 假设像素方形
        float cx = width / 2f;
        float cy = height / 2f;

        CameraIntrinsics intrinsics = new CameraIntrinsics()
        {
            width = width,
            height = height,
            fx = fx,
            fy = fy,
            cx = cx,
            cy = cy,
            nearClip = cam.nearClipPlane,
            farClip = cam.farClipPlane,
            fov = cam.fieldOfView
        };

        string json = JsonUtility.ToJson(intrinsics, true);
        File.WriteAllText(Path.Combine(saveDir, "intrinsics.json"), json, Encoding.UTF8);

        // 清理
        cam.ResetReplacementShader();
        cam.targetTexture = null;
        RenderTexture.active = null;

        Debug.Log("Saved rgb.png, depth.npy, intrinsics.json to " + saveDir);
    }
}