using System.Collections;
using System.Collections.Generic;
using UnityEngine;

public class CircularMotion : MonoBehaviour
{
    [Header("运动物体（被控制者）")]
    public Transform targetObject;              // 👉 控制的目标物体

    [Header("参考平面（决定方向和坐标系）")]
    public Transform referencePlane;

    [Header("圆周运动参数")]
    public Vector3 localCenter = Vector3.zero;  // 在平面局部坐标下的圆心
    public float radius = 1f;
    public float angularSpeed = 90f;
    public float startAngleDeg = 0f;
    private bool scrollMode = false;

    float currentAngle;
    Vector3 worldCenter;
    bool isGrabbed = false;

    private float objectRadius;  // 添加物体半径属性

    public float GetModelHeight(GameObject obj)
    {
        Collider col = obj.GetComponent<Collider>();
        if (col != null)
        {
            // Debug.Log($"得到操作物体高度为{col.bounds.size.y}");
            return col.bounds.size.y;
        }
        else
        {
            Debug.LogWarning("未找到 Renderer，无法估算高度");
            return 0f;
        }
    }

    public float GetCurrentAngle(){
        return currentAngle;
    }

    public Vector3 GetWorldCenter(){
        return worldCenter;
    }

    public void setIsGrabbed(bool flag)
    {
        isGrabbed = flag;
    }

    public void SetMotion(string objectName, Vector3 localCenter, float radius, float angular_speed, float startAngleDeg)
    {
        this.localCenter = localCenter;
        this.radius = radius;
        this.angularSpeed = angular_speed;
        this.startAngleDeg = startAngleDeg;

        isGrabbed = false;
        targetObject = GameObject.Find(objectName).transform;

        // 再执行一次start
        currentAngle = startAngleDeg;
        if (targetObject != null && referencePlane != null)
        {
            worldCenter = referencePlane.TransformPoint(localCenter);
            float heightOffset = GetModelHeight(targetObject.gameObject) * 0.5f;
            worldCenter += referencePlane.up * heightOffset;
        }
        else Debug.LogError("CircularMotion 组件初始化失败：目标物体或参考平面未设置");

        scrollMode = targetObject.gameObject.layer == LayerMask.NameToLayer("canScroll");

        // 新增一个将targetobject放在初始位置的函数
        targetObject.position = worldCenter + referencePlane.TransformVector(new Vector3(Mathf.Cos(startAngleDeg * Mathf.Deg2Rad), 0f, Mathf.Sin(startAngleDeg * Mathf.Deg2Rad)) * radius);
        
        // 确保物体完全静止，为同步开始做准备
        Rigidbody rb = targetObject.GetComponent<Rigidbody>();
        if (rb != null)
        {
            rb.velocity = Vector3.zero;
            rb.angularVelocity = Vector3.zero;
        }
    }

    void Start()
    {
        currentAngle = startAngleDeg;
        if (targetObject != null && referencePlane != null){
            worldCenter = referencePlane.TransformPoint(localCenter);
            float heightOffset = GetModelHeight(targetObject.gameObject) * 0.5f;
            worldCenter += referencePlane.up * heightOffset;

            objectRadius = heightOffset;
        }
        else Debug.LogError("CircularMotion 组件初始化失败：目标物体或参考平面未设置");

        scrollMode = targetObject.gameObject.layer == LayerMask.NameToLayer("canScroll");
    }

    private Vector3 EstimateTangentialVelocity()
    {
        // 圆心 → 物体方向的半径向量
        Vector3 objectPos = targetObject.position;
        Vector3 radiusDir = (objectPos - worldCenter).normalized;

        // 切线方向 = 半径方向在 XZ 平面上的逆时针垂直向量
        // 即：radiusDir = (x, 0, z) -> tangential = (-z, 0, x)
        Vector3 tangentDir = new Vector3(-radiusDir.z, 0f, radiusDir.x);

        // 根据角速度正负决定切线方向方向（顺时针 or 逆时针）
        if (angularSpeed < 0)
        {
            tangentDir = -tangentDir;
        }

        // 切线速度 = ω * r
        float tangentialSpeed = Mathf.Abs(angularSpeed * Mathf.Deg2Rad * radius);
        return tangentDir * tangentialSpeed;
    }

    void FixedUpdate()
    {
        if (referencePlane == null || targetObject == null)
            return;

        if (!isGrabbed){
            currentAngle += angularSpeed * Time.fixedDeltaTime;
            if (currentAngle > 360f){
                currentAngle -= 360f;
            }
            float angleRad = currentAngle * Mathf.Deg2Rad;

            // // 构造圆周偏移（在局部平面上）
            Vector3 localOffset = new Vector3(Mathf.Cos(angleRad), 0f, Mathf.Sin(angleRad)) * radius;

            Vector3 worldOffset = referencePlane.TransformVector(localOffset);
            worldOffset = localOffset;
            targetObject.position = worldCenter + worldOffset;

            // 让 targetObject 的 local Y 轴对齐平面法向
            Quaternion targetRotation = Quaternion.LookRotation(referencePlane.forward, referencePlane.up);
            targetObject.rotation = targetRotation;

            if(scrollMode){
                float currentSpeed = angularSpeed * radius;
                float angularSpeed_scroll = currentSpeed / objectRadius;  // 弧度/秒

                Vector3 worldDirection = EstimateTangentialVelocity().normalized;
                // 计算旋转轴：垂直于运动方向和向上方向的轴
                Vector3 rotationAxis = Vector3.Cross(referencePlane.up, worldDirection).normalized;

                // 应用旋转
                targetObject.Rotate(rotationAxis, Mathf.Abs(angularSpeed_scroll) * Mathf.Rad2Deg * Time.fixedDeltaTime, Space.World);
            }
        }
        else{
            return;
        }
    }

    void OnDrawGizmos()
    {
        if (referencePlane == null) return;
        if (!enabled) return;

        Vector3 worldCenter = referencePlane.TransformPoint(localCenter);

        // 圆心点
        Gizmos.color = Color.red;
        Gizmos.DrawSphere(worldCenter, 0.1f);

        // 圆轨迹
        Gizmos.color = Color.green;
        int segments = 100;
        Vector3 prev = worldCenter + referencePlane.TransformVector(new Vector3(radius, 0f, 0f));
        for (int i = 1; i <= segments; i++)
        {
            float angle = i * Mathf.PI * 2f / segments;
            Vector3 localOffset = new Vector3(Mathf.Cos(angle), 0f, Mathf.Sin(angle)) * radius;
            Vector3 next = worldCenter + referencePlane.TransformVector(localOffset);
            Gizmos.DrawLine(prev, next);
            prev = next;
        }
        // 起始位置
        Gizmos.color = Color.blue;
        Gizmos.DrawSphere(worldCenter + referencePlane.TransformVector(new Vector3(Mathf.Cos(startAngleDeg * Mathf.Deg2Rad), 0f, Mathf.Sin(startAngleDeg * Mathf.Deg2Rad)) * radius), 0.1f);

        // 切线方向
        if (targetObject != null){
            Vector3 worldDirection = EstimateTangentialVelocity();
            Gizmos.color = Color.blue;
            Gizmos.DrawLine(targetObject.position, targetObject.position + worldDirection * 10f);
        }
    }
}
