using System.Collections;
using System.Collections.Generic;
using UnityEngine;

using WebSocketSharp; // 使用 websocket-sharp 的命名空间

// using System.Linq;
// using System.Text;
// using Unity.Barracuda;
using System;
using System.IO;

using EpisodeControl; // 自己写的采样参数管理器

using UnityEditor; // 绘制透明面，方便调试


public class Dynamic : MonoBehaviour
{
    public Transform finger_1_1;
    public Transform finger_1_2;
    public Transform finger_1_3;
    public Transform finger_2_1;
    public Transform finger_2_2;
    public Transform finger_2_3;
    public Transform finger_3_1;
    public Transform finger_3_2;
    public Transform finger_3_3;
    public Transform finger_4_1;
    public Transform finger_4_2;
    public Transform finger_4_3;
    public Transform finger_5_1;
    public Transform finger_5_2;
    public Transform finger_5_3;
    public Transform finger_nail_1;
    public Transform finger_nail_2;
    public Transform finger_nail_3;
    public Transform finger_nail_4;
    public Transform finger_nail_5;

    public Transform palmCenter; // 四指集合位置标记，用于精确定位抓取位置，**四指的集合**
    public Transform handsRoot; // 手腕关节
    public Transform armRoot; // 手臂根节点

    List<Transform> transforms = new List<Transform>(); // 存储所有手指关节
    List<Transform> detectjoints; // 存储所有带碰撞球的关节
    List<Transform> storeList; // 存储所有需要记录6d位姿的物体


    // 动态抓握系统
    bool isGrabbingObject = false; // 标记是否已经抓到物体
    public enum HandState { Idle, MovingToTarget, WaitingTarget, Grabbing, MovingToHome, MovingToRelease, Releasing, ReturningHome }
    public HandState currentState = HandState.Idle; // 当前手部状态
    Transform targetObject; // 目标抓取物体
    GameObject grabbedObject; // 当前抓取的物体
    float grabbedObjectHeight; // 新增：抓取物体的高度
    float release_offset = 0.6f; // 新增：放置高度偏移（用于非悬空/小幅度悬空放置）
    Vector3 targetPosition; // 目标移动位置
    Vector3 initialPosition; // 初始位置
    float initialHeight; // 初始高度
    float moveSpeed = 4.0f; // 手部移动速度
    float fingerRotateSpeed = 10.0f; // 手部移动速度
    float heightOffset = 0.1f; // 抓取时的高度偏移
    float heightOffset_d = 0.2f; // 动态物体抓取时候的等待高度
    List<float> initialRotations = new List<float>(); // 存储所有要旋转关节的初始旋转
    List<Vector3> initialRotations2 = new List<Vector3>(); // 存储所有关节的初始XYZ旋转角
    float maxGrabRotation = 90f; // 最大抓取旋转角度（度）
    Dictionary<int, bool> fingerContactFlags = new Dictionary<int, bool>()
    {
        {1, false}, {2, false}, {3, false}, {4, false}, {5, false}
    };
    List<Transform> overRotatedJoints = new List<Transform>(); // 超过旋转角度限制的关节
    float handRotationSpeed = 360f; // 手部旋转速度（度/秒）

    [Header("时间参数设置")]
    public float leadTime = 0.05f; // 手提前到达的时间

    [Tooltip("下落开始几秒后接到物体，注意必须大于leadTime")]
    public float interceptTime = 2f; // 预测拦截时间
    public float observationTime = 0.1f; // 等待时间
    private float waitTime = 0f; // 等待时间计数器



    // 相机拍摄每帧图像
    [Header("相机设置")]
    public Camera handCamera;  // 绑定的摄像机
    public RenderTexture renderTexture;  // 用于绑定相机输出
    public int width;
    public int height;
    public float sendInterval = 0.05f; // 发送间隔时间
    private float lastSendTime = 0f; // 上次发送时间
    private WebSocket ws; // WebSocket 实例
    Texture2D screenTexture;
    // 相机随手动逻辑
    public float maxZDistance;
    public float minZDistance;
    Vector3 initCameraPos;

    // 运动脚本来源
    [Header("运动脚本来源")]
    public Transform scriptLoader; // 新增：挂载运动脚本的物体
    public CircularMotion cm;
    public LinearMotion lm;
    public HarmonicMotion hm;

    // 采样参数管理
    private EpisodeManager episodeManager;

    // 超时重试逻辑
    float handMovingElapsedTime = 0f;
    float handWaitingElapsedTime = 0f;
    int retryTimes = 0;
    private float timeOutBar = 0;

    // 保存图像逻辑
    private string saveDir;
    private Dictionary<int, int> episodeFrameCounters = new Dictionary<int, int>();
    private HashSet<int> savedMetaDataEpisodes = new HashSet<int>(); // 新增：记录已保存meta_data的episode


    // Start is called before the first frame update
    // 初始化所有手指关节列表和指尖关节列表
    void Start()
    {
        // 添加所有手指关节到总列表，共15个关节
        transforms.Add(finger_1_1);
        transforms.Add(finger_1_2);
        transforms.Add(finger_1_3);
        transforms.Add(finger_2_1);
        transforms.Add(finger_2_2);
        transforms.Add(finger_2_3);
        transforms.Add(finger_3_1);
        transforms.Add(finger_3_2);
        transforms.Add(finger_3_3);
        transforms.Add(finger_4_1);
        transforms.Add(finger_4_2);
        transforms.Add(finger_4_3);
        transforms.Add(finger_5_1);
        transforms.Add(finger_5_2);
        transforms.Add(finger_5_3);

        // 改动：检测碰撞的关节，用detectjoints储存所有关节用于碰撞检测，需要5个指尖关节的同时去掉5个手指根关节，共15个关节
        detectjoints = new List<Transform>(transforms){
            finger_nail_1, finger_nail_2, finger_nail_3, finger_nail_4, finger_nail_5
        };
        detectjoints.Remove(finger_1_1);
        detectjoints.Remove(finger_2_1);
        detectjoints.Remove(finger_3_1);
        detectjoints.Remove(finger_4_1);
        detectjoints.Remove(finger_5_1);

        // 改动：记录所有需要记录6d位姿的物体，共21个关节（20个手指关节+掌根）再加 相机、掌心
        storeList = new List<Transform>(transforms){
            finger_nail_1, finger_nail_2, finger_nail_3, finger_nail_4, finger_nail_5, handsRoot
        };
        storeList.Add(palmCenter);
        storeList.Add(handCamera.transform);

        

        // 记录手部初始位置和高度
        initialPosition = transform.position;
        initialHeight = transform.position.y;
        // 记录相机初始位置
        initCameraPos = handCamera.transform.position;
        // 记录handsRoot的初始旋转
        // if (handsRoot != null)
        // {
        //     initialHandsRootRotation = handsRoot.localRotation.eulerAngles;
        // }
        // 记录所有关节的初始z轴旋转角度
        foreach (var joint in transforms)
        {
            if (joint != null)
            {
                initialRotations.Add(joint.localRotation.eulerAngles.z);
                initialRotations2.Add(joint.localRotation.eulerAngles);
            }
            else
            {
                initialRotations.Add(0f);
            }
        }

        // socket
        ws = new WebSocket("ws://localhost:8766");
        ws.Connect();
        Debug.Log("已连接到 Python WebSocket 服务器");


        // ego摄像机初始化贴图。【重要注意‼️】调试的时候注释掉保证摄像机画面可见
        // screenTexture = new Texture2D(renderTexture.width, renderTexture.height, TextureFormat.RGB24, false);
        // handCamera.targetTexture = renderTexture;
        
        renderTexture = new RenderTexture(width, height, 24, RenderTextureFormat.ARGB32);
        screenTexture = new Texture2D(width, height, TextureFormat.RGB24, false);  // 不压缩、不 mipmap
        handCamera.targetTexture = renderTexture;


        // 得到要用的运动控制脚本和运动参数JSON
        scriptLoader = GameObject.Find("ScriptLoader").transform;
        cm = scriptLoader.GetComponent<CircularMotion>();
        lm = scriptLoader.GetComponent<LinearMotion>();
        hm = scriptLoader.GetComponent<HarmonicMotion>();

        ConfigType configType = ConfigType.Linear;
        string path = "ParaConfigs/level2_linear_configs.json";
        if (cm!=null && cm.enabled){configType = ConfigType.Circular; path = "ParaConfigs/level2_circular_configs.json";}
        else if (hm!=null && hm.enabled){configType = ConfigType.Harmonic; path = "ParaConfigs/level2_harmonic_configs.json";}

        // 保证数据根目录存在
        saveDir = Path.Combine(Application.dataPath, "..", $"received_data_{path.Split('_')[1]}");
        Directory.CreateDirectory(saveDir);

        // 告诉Python端现在在做哪个任务
        if (ws!=null && ws.ReadyState == WebSocketState.Open) ws.Send($"{{ \"task_name\": \"{path.Split('_')[1]}\" }}");

        // 加载采样参数配置
        episodeManager = new EpisodeManager(path, configType, this, 44280);
        episodeManager.TryAdvanceEpisode(); // 运行第一个
    }

    private void RotateFingerGroup(Transform joint1, Transform joint2, Transform joint3, float finggerMaxRotation)
    {

        Transform[] joints = { joint1, joint2, joint3 };

        foreach (var joint in joints)
        {
            if (joint != null)
            {
                // 获取当前的本地旋转
                Vector3 currentRotation = joint.localRotation.eulerAngles;

                // 在z轴上增加旋转
                currentRotation.z += Time.deltaTime * fingerRotateSpeed;

                // 限制最大旋转角度
                if (currentRotation.z > finggerMaxRotation)
                {
                    currentRotation.z = finggerMaxRotation;
                }

                // 应用新的旋转值
                joint.localRotation = Quaternion.Euler(currentRotation);
            }
        }
    }


    void Update()
    {
        // 更新手部状态机
        UpdateHandStateMachine();

        // 持续检测指尖关节是否碰到物体
        CheckDetectJointsCollision2();
    }

    private void ClearAllFilesInFolder(string folderPath)
    {
        if (!Directory.Exists(folderPath))
        {
            Console.WriteLine($"❌ 文件夹不存在：{folderPath}");
            return;
        }

        try
        {
            // 删除根目录下的文件
            foreach (string file in Directory.GetFiles(folderPath))
            {
                File.Delete(file);
                // Console.WriteLine($"🧹 删除文件：{file}");
            }

            // Console.WriteLine($"✅ 所有文件已清除，保留文件夹结构：{folderPath}");
        }
        catch (Exception ex)
        {
            Console.WriteLine($"⚠️ 删除文件时出错：{ex.Message}");
        }
    }

    /// <summary>
    /// 保存当前episode的元数据信息到meta_data.json
    /// 包含任务类型、运动参数、预测抓取点位置和手部移动速度
    /// </summary>
    private void SaveMetaData(string episodePath, int episode)
    {
        try
        {
            var metaData = new EpisodeControl.MetaData
            {
                task_type = episodeManager.GetConfigTypeString(),
                predicted_intercept_position = targetPosition,
                hand_move_speed = moveSpeed,
                episode = episode
            };

            // 根据任务类型设置对应的运动参数
            var currentConfig = episodeManager.GetCurrentConfig();
            if (currentConfig != null)
            {
                switch (currentConfig)
                {
                    case EpisodeControl.LinearConfig linearConfig:
                        metaData.linear_params = linearConfig;
                        break;
                    case EpisodeControl.CircularConfig circularConfig:
                        metaData.circular_params = circularConfig;
                        break;
                    case EpisodeControl.HarmonicConfig harmonicConfig:
                        metaData.harmonic_params = harmonicConfig;
                        break;
                }
            }

            string json = JsonUtility.ToJson(metaData, true);
            string metaPath = Path.Combine(episodePath, "meta_data.json");
            File.WriteAllText(metaPath, json);
            
            Debug.Log($"已保存episode {episode}的元数据: 任务类型={metaData.task_type}, 预测位置={metaData.predicted_intercept_position}, 移动速度={metaData.hand_move_speed}");
        }
        catch (Exception ex)
        {
            Debug.LogError($"保存元数据失败: {ex.Message}");
        }
    }


    public void CheckForNextEpisode()
    {
        if(!isGrabbingObject){
            if (retryTimes > 3) // 失败次数超过3次，则认为模拟失败
            {
                retryTimes = 0;
                Debug.Log($"[Episode {episodeManager.currentConfig.index}] 模拟失败，准备下一轮");
                ws.Send($"{{ \"episode_failed\": {episodeManager.currentConfig.index} }}");
            }
            else{ // 失败次数不超过3次，则认为模拟失败，重试
                retryTimes++;
                Debug.Log($"[Episode {episodeManager.currentConfig.index}] 模拟失败，正在重试第{retryTimes}次");
                // ws.Send($"{{ \"episode_failed\": {episodeManager.currentEpisode} }}");
                episodeFrameCounters[episodeManager.currentConfig.index] =0;
                // 清空相应文件夹中的图像和json
                ClearAllFilesInFolder(Path.Combine(saveDir, $"episode_{episodeManager.currentConfig.index:00000}"));
                Debug.Log($"已清除episode_{episodeManager.currentConfig.index:00000}先前发送的错误文件");

                // 重置meta_data保存标志，允许重试时重新保存meta_data
                savedMetaDataEpisodes.Remove(episodeManager.currentConfig.index);

                episodeManager.currentEpisode--;
                
            }
            
        }
        else
        {
            retryTimes = 0;
            Debug.Log($"[Episode {episodeManager.currentConfig.index}] 模拟结束，准备下一轮");
            ws.Send($"{{ \"episode_done\": {episodeManager.currentConfig.index} }}");
        }

        // 重置为初始状态
        // 1、恢复物体物理属性
        targetObject.GetComponent<Rigidbody>().isKinematic = false;
        targetObject.GetComponent<Rigidbody>().useGravity = true;

        // 2、恢复手部关节的初始旋转和位置
        for (int i = 0; i < transforms.Count; i++)
        {
            if (transforms[i] != null)
            {
                transforms[i].localRotation = Quaternion.Euler(initialRotations2[i]);
            }
        }
        transform.position = initialPosition;

        // 3、恢复flags
        isGrabbingObject = false;
        fingerContactFlags = new Dictionary<int, bool>()
        {
            {1, false}, {2, false}, {3, false}, {4, false}, {5, false}
        };

        // 4、恢复重力
        EnableGravityAndRelease();

        // 5、恢复相机位置
        handCamera.transform.position = initCameraPos;

        // 6、加载下一轮
        episodeManager.TryAdvanceEpisode();
        currentState = Dynamic.HandState.Idle;

    }

    void LateUpdate()
    {
        if (currentState == HandState.MovingToHome)
        {
            CheckForNextEpisode();
            return;
        }

        // 实现相机跟随逻辑
        if (palmCenter == null || targetPosition == null) return;
        if (currentState != HandState.MovingToTarget && currentState != HandState.MovingToHome) return;
        Vector3 cameraPos = handCamera.transform.position;
        Vector3 handPos = palmCenter.position;

        float zDistance = handPos.z - cameraPos.z;

        // 计算 Z 方向上的手部速度分量（用于控制相机移动速度）
        Vector3 handMoveDirection = (targetPosition - palmCenter.position).normalized;
        float zVelocity = handMoveDirection.z * moveSpeed;
        float deltaZ = 0f;

        if (zDistance > maxZDistance)
        {
            deltaZ = Mathf.Sign(zVelocity) * Mathf.Abs(zVelocity) * Time.deltaTime;
        }
        else if (zDistance < minZDistance)
        {
            // Debug.Log($"相机太近了，当前距离：{zDistance}，阈值：{minZDistance}");
            deltaZ = Mathf.Sign(zVelocity) * Mathf.Abs(zVelocity) * Time.deltaTime;
        }

        // 更新相机位置，只改变 Z 坐标
        handCamera.transform.position += new Vector3(0f, 0f, deltaZ);

    }


    void FixedUpdate()
    {
        // 每 fixedUpdate 保存一次图像与关节数据
        if (Time.time - lastSendTime >= sendInterval)
        {
            // 获取当前 episode
            int ep = episodeManager.currentConfig.index;
            if (!episodeFrameCounters.ContainsKey(ep))
            {
                episodeFrameCounters[ep] = 0;
            }

            // 获取保存路径
            string episodePath = Path.Combine(saveDir, $"episode_{ep:00000}");
            Directory.CreateDirectory(episodePath); // 若不存在则创建

            int frameIdx = episodeFrameCounters[ep];

            // === 图像采集 ===
            RenderTexture.active = renderTexture;
            handCamera.Render(); // 渲染当前帧

            screenTexture.ReadPixels(new Rect(0, 0, renderTexture.width, renderTexture.height), 0, 0);
            screenTexture.Apply();
            RenderTexture.active = null;
  

            // 保存为 JPG
            byte[] imageBytes = screenTexture.EncodeToJPG(85);
            Directory.CreateDirectory(episodePath); // 若不存在则创建
            string imagePath = Path.Combine(episodePath, $"img_{frameIdx:0000}.jpg");
            File.WriteAllBytes(imagePath, imageBytes);

            // === joints 采集 ===
            HandData handData = new HandData
            {
                episode = ep
            };

            foreach (Transform joint in storeList)
            {
                if (joint != null)
                {
                    JointData jd = new JointData
                    {
                        name = joint.name,
                        position = new float[]
                        {
                        joint.position.x,
                        joint.position.y,
                        joint.position.z
                        },
                        rotation = new float[]
                        {
                        joint.eulerAngles.x,
                        joint.eulerAngles.y,
                        joint.eulerAngles.z
                        }
                    };
                    handData.joints.Add(jd);
                }
            }

            string json = JsonUtility.ToJson(handData, true); // pretty-print
            string jsonPath = Path.Combine(episodePath, $"joints_{frameIdx:0000}.json");
            File.WriteAllText(jsonPath, json);

            // 更新帧编号
            episodeFrameCounters[ep]++;
            lastSendTime = Time.time;
        }
    }
    
    /// <summary>
    /// 对所有手指关节进行z轴旋转
    /// 只旋转还没有接触到物体的手指，已接触的手指停止旋转
    /// <param name="rotateSpeed">旋转速度</param>
    /// </summary>
    private void RotateAllFingerJointsZ()
    {
        // 新版本：直接使用detectjoints列表进行旋转
        if (currentState != HandState.Grabbing || targetObject == null)
        {
            Debug.Log("当前状态不是Grabbing，或者目标物体为空，不进行旋转");
            return;
        }

        // 使用碰撞检测得到的状态判断是否旋转
        // RotateFingerGroup(finger_1_1, finger_1_2, !fingerContactFlags[1]); // 拇指
        RotateThumbGroup(finger_1_1, finger_1_2, finger_1_3, !fingerContactFlags[1]); // 拇指
        RotateFingerGroup(finger_2_1, finger_2_2, finger_2_3, !fingerContactFlags[2]); // 食指
        RotateFingerGroup(finger_3_1, finger_3_2, finger_3_3, !fingerContactFlags[3]); // 中指
        RotateFingerGroup(finger_4_1, finger_4_2, finger_4_3, !fingerContactFlags[4]); // 无名指
        RotateFingerGroup(finger_5_1, finger_5_2, finger_5_3, !fingerContactFlags[5]); // 小指
    }

    /// <summary>
    /// 旋转手指组（拇指版本 - 2个关节）
    /// </summary>
    /// <param name="joint1">第一个关节</param>
    /// <param name="joint2">第二个关节</param>
    /// <param name="canRotate">是否可以旋转</param>
    private void RotateFingerGroup(Transform joint1, Transform joint2, bool canRotate)
    {
        if (!canRotate) return;

        Transform[] joints = { joint1, joint2 };

        foreach (var joint in joints)
        {
            if (joint != null)
            {
                // 获取当前的本地旋转
                Vector3 currentRotation = joint.localRotation.eulerAngles;

                // 在z轴上增加旋转
                currentRotation.z += Time.deltaTime * fingerRotateSpeed;

                // 应用新的旋转值
                joint.localRotation = Quaternion.Euler(currentRotation);
            }
        }
    }

    /// <summary>
    /// 旋转手指组（拇指版本 - 3个关节）
    /// 旋转时： joint1 Z变大； joint2 Y变小； joint3 Y变小
    /// </summary>
    /// <param name="joint1">第一个关节</param>
    /// <param name="joint2">第二个关节</param>
    /// <param name="joint3">第三个关节</param>
    /// <param name="canRotate">是否可以旋转</param>
    private void RotateThumbGroup(Transform joint1, Transform joint2, Transform joint3, bool canRotate)
    {
        if (!canRotate) return;

        if(joint1 != null)
        {
            Vector3 currentRotation = joint1.localRotation.eulerAngles;
            currentRotation.z += Time.deltaTime * fingerRotateSpeed;
            joint1.localRotation = Quaternion.Euler(currentRotation);
        }
        
        if(joint2 != null && joint3 != null)
        {
            Vector3 currentRotation = joint2.localRotation.eulerAngles;
            currentRotation.y -= Time.deltaTime * fingerRotateSpeed;
            joint2.localRotation = Quaternion.Euler(currentRotation);
        }
    }

    /// <summary>
    /// 旋转手指组（四指版本 - 3个关节）
    /// </summary>
    /// <param name="joint1">第一个关节</param>
    /// <param name="joint2">第二个关节</param>
    /// <param name="joint3">第三个关节</param>
    /// <param name="canRotate">是否可以旋转</param>
    private void RotateFingerGroup(Transform joint1, Transform joint2, Transform joint3, bool canRotate)
    {
        if (!canRotate)
        {
            // Debug.Log($"手指已接触物体，停止旋转: {joint3?.name}");
            return;
        }

        Transform[] joints = { joint1, joint2, joint3 };

        foreach (var joint in joints)
        {
            if (joint != null)
            {
                // 获取当前的本地旋转
                Vector3 currentRotation = joint.localRotation.eulerAngles;

                // 在z轴上增加旋转
                currentRotation.z += Time.deltaTime * fingerRotateSpeed;

                // 应用新的旋转值
                joint.localRotation = Quaternion.Euler(currentRotation);
            }
        }
    }

    /// <summary>
    /// 检测 **所有指关节** 是否与物体发生碰撞
    /// 使用球形射线检测，半径为0.04单位
    /// 只有当所有手指都接触到目标物体时才算抓取成功
    /// 如果旋转角度超过90度还没抓到，也算抓取完成
    /// </summary>
    private void CheckDetectJointsCollision2()
    {
        if (currentState != HandState.Grabbing || isGrabbingObject || targetObject == null)
            return;


        GameObject detectedObject = null;

        foreach (var joint in detectjoints)
        {
            if (joint == null) continue;



            SphereCollider jointCollider = joint.GetComponent<SphereCollider>();
            if (jointCollider == null) continue;

            // 计算世界空间下的球心位置
            Vector3 worldCenter = jointCollider.transform.TransformPoint(jointCollider.center);
            float worldRadius = jointCollider.radius * Mathf.Max(
                joint.lossyScale.x,
                joint.lossyScale.y,
                joint.lossyScale.z
            );

            // Debug.Log($"worldCenter: {worldCenter}, worldRadius: {worldRadius}"); // Debug之后发现即使是每个joint上面的SphereCollider没有启用，上面的代码还是可以正常获得碰撞球参数
            Collider[] hitColliders = Physics.OverlapSphere(worldCenter, worldRadius);



            foreach (var collider in hitColliders)
            {
                if (!IsHandComponent(collider.transform) &&
                    collider.gameObject.layer != LayerMask.NameToLayer("Ground") &&
                    (collider.transform == targetObject || collider.transform.IsChildOf(targetObject) || targetObject.IsChildOf(collider.transform)))
                {
                    detectedObject = collider.gameObject;

                    // 判定当前关节属于哪一根手指
                    int fingerIndex = GetFingerIndex(joint);
                    if (fingerIndex > 0)
                        fingerContactFlags[fingerIndex] = true;
                }
            }
        }

        int fingersInContact = 0;
        foreach (var entry in fingerContactFlags)
        {
            if (entry.Value) fingersInContact++;
        }

        bool maxRotationReached = CheckMaxRotationReached();

        // Debug.Log($"接触检测：当前已有{fingersInContact}/5 根手指接触到目标物体");
        if (maxRotationReached)
        {
            // Debug.Log("手指旋转已达到90度限制");
        }

        if ((fingersInContact >= 5 && detectedObject != null) || maxRotationReached)
        {
            isGrabbingObject = true;
            if (detectedObject != null)
            {
                GrabObject(detectedObject);
                Debug.Log($"抓取成功！{fingersInContact}/5 根手指接触到了物体：{detectedObject.name}");
            }
            else if (maxRotationReached)
            {
                GrabObject(targetObject.gameObject);
                Debug.Log($"达到最大旋转角度，强制完成抓取：{targetObject.name}");
            }
        }
        else if (fingersInContact > 0)
        {
            // Debug.Log($"部分接触：{fingersInContact}/5 根手指接触到物体，继续抓握...");
        }
    }

    /// <summary>
    /// 判定当前关节属于哪一根手指，根据关节Transform返回手指索引
    /// </summary>
    private int GetFingerIndex(Transform joint)
    {
        if (joint == finger_1_1 || joint == finger_1_2 || joint == finger_1_3 || joint == finger_nail_1) return 1;
        if (joint == finger_2_1 || joint == finger_2_2 || joint == finger_2_3 || joint == finger_nail_2) return 2;
        if (joint == finger_3_1 || joint == finger_3_2 || joint == finger_3_3 || joint == finger_nail_3) return 3;
        if (joint == finger_4_1 || joint == finger_4_2 || joint == finger_4_3 || joint == finger_nail_4) return 4;
        if (joint == finger_5_1 || joint == finger_5_2 || joint == finger_5_3 || joint == finger_nail_5) return 5;
        return 0;
    }



    /// <summary>
    /// 检查是否达到最大旋转角度
    /// 检查任意一根手指的旋转角度是否超过90度
    /// </summary>
    /// <returns>如果达到最大旋转角度返回true</returns>
    private bool CheckMaxRotationReached()
    {
        bool flag = false;
        overRotatedJoints.Clear(); // 每次检测前清空列表
        for (int i = 0; i < transforms.Count && i < initialRotations.Count; i++)
        {
            if (transforms[i] != null)
            {
                float currentRotation = transforms[i].localRotation.eulerAngles.z;
                float initialRotation = initialRotations[i];

                // 计算旋转差值（考虑角度环绕问题）
                float rotationDifference = Mathf.Abs(Mathf.DeltaAngle(initialRotation, currentRotation));

                if (rotationDifference >= maxGrabRotation)
                {
                    // Debug.Log($"关节 {transforms[i].name} 旋转角度达到限制: {rotationDifference:F1}度");
                    // 记录该关节
                    overRotatedJoints.Add(transforms[i]);
                    flag = true;
                }
            }
        }

        return flag;
    }

    /// <summary>
    /// 判断给定的Transform是否属于手部组件
    /// 用于在碰撞检测中排除手部自身的碰撞
    /// </summary>
    /// <param name="targetTransform">要检查的Transform</param>
    /// <returns>如果是手部组件返回true，否则返回false</returns>
    /// 
    private bool IsHandComponent(Transform targetTransform)
    {
        // 检查是否是手部的任何一个关节
        foreach (var joint in transforms)
        {
            if (joint != null && (targetTransform == joint || targetTransform.IsChildOf(joint) || joint.IsChildOf(targetTransform)))
            {
                return true;
            }
        }

        // 检查是否是手部根对象的子对象
        if (targetTransform.IsChildOf(this.transform) || targetTransform == this.transform)
        {
            return true;
        }

        return false;
    }

    
    /// <summary>
    /// 尝试选择抓取目标
    /// 检查点击的物体是否可以被抓取（排除地板等）
    /// </summary>
    private void TrySelectGrabTarget()
    {
        Vector3 p0 = palmCenter.position;
        Vector3 interceptPoint;
        float best_t = interceptTime;

        if (cm != null && cm.enabled)
        {
            targetObject = cm.targetObject;
            Debug.Log($"选择抓取目标：{targetObject.name}，手部正在移动到目标位置（圆周运动物体）");
            // 圆运动参数
            float angularSpeedDeg = cm != null ? cm.angularSpeed : 0f;  // 角速度（度/秒）
            float radius = cm != null ? cm.radius : 0f;

            Transform referencePlane = cm.referencePlane;
            Vector3 center = cm.GetWorldCenter();

            float angularSpeedRad = angularSpeedDeg * Mathf.Deg2Rad;   // 转换为弧度
            float currentAngleDeg = cm.GetCurrentAngle();
            float currentAngleRad = currentAngleDeg * Mathf.Deg2Rad;

            // 再次计算目标拦截位置
            float interceptTheta = currentAngleRad + angularSpeedRad * best_t;
            Vector3 interceptOffset = new Vector3(Mathf.Cos(interceptTheta), 0f, Mathf.Sin(interceptTheta)) * radius;
            interceptPoint = center + interceptOffset;

            float T = Mathf.Abs(2 * Mathf.PI / angularSpeedRad);
            timeOutBar = Mathf.Min(0.1f * T, 0.5f);

            moveSpeed = Vector3.Distance(interceptPoint, p0) / (best_t - leadTime);

        }
        else if (lm != null && lm.enabled)
        {
            targetObject = lm.targetObject;
            Debug.Log($"选择抓取目标：{targetObject.name}，手部正在移动到目标位置（直线运动物体）");
            Vector3 q0 = targetObject.transform.position;

            float v0 = lm != null ? lm.getCurrentSpeed() : 0f;   // 初速度
            float a = lm != null ? lm.acceleration : 0f;         // 加速度
            Vector3 moveDirection = lm != null ? lm.getWorldDirection().normalized : Vector3.zero;
            // Debug.Log($"加速度：{a}，初速度：{v0}，方向：{moveDirection}");
            
            interceptPoint = q0 + moveDirection * (v0 * best_t + 0.5f * a * best_t * best_t);

            timeOutBar = 0.5f;

            moveSpeed = Vector3.Distance(interceptPoint, p0) / (best_t - leadTime);
            // Debug.Log($"手部运动速度：{moveSpeed}");

        }
        else if (hm != null && hm.enabled)
        {
            targetObject = hm.targetObject;
            Debug.Log($"选择抓取目标：{targetObject.name}，手部正在移动到目标位置（简谐运动物体）");

            // interceptPoint = hm.PredictDisplacementAfter(best_t);
            interceptPoint = hm.PredictDisplacementAt(best_t + observationTime);

            float interceptPhase = hm.GetPhaseAt(best_t + observationTime) % (2 * Mathf.PI);
            if (interceptPhase < 0.5f * Mathf.PI) timeOutBar = Mathf.Min(0.5f * Mathf.PI - interceptPhase,0.5f);
            else if(interceptPhase > Mathf.PI && interceptPhase < 1.5f * Mathf.PI) timeOutBar = Mathf.Min(1.5f * Mathf.PI - interceptPhase, 0.5f);
            else timeOutBar = 0.5f;



            moveSpeed = Vector3.Distance(interceptPoint, p0) / (best_t - leadTime);
        }
        else
        {
            Debug.LogError("没有找到对应的圆周运动或直线运动脚本");
            return;
        }

        Bounds bounds = GetObjectBounds(targetObject.gameObject);
        float objectTopY = bounds.max.y;
        interceptPoint.y = objectTopY + heightOffset_d;

        targetPosition = interceptPoint;
        grabbedObjectHeight = interceptPoint.y;

        Debug.Log($"[预测] {targetObject.name} 将在 {best_t:F2}s 后拦截，目标位置：{targetPosition}，手部速度：{moveSpeed}");
        currentState = HandState.MovingToTarget;
        fingerRotateSpeed = 60.0f;

        // 保存当前episode的元数据信息到meta_data.json（每个episode只保存一次）
        // int currentEp = episodeManager.currentEpisode;
        int currentEp = episodeManager.currentConfig.index;
        if (!savedMetaDataEpisodes.Contains(currentEp))
        {
            string episodePath = Path.Combine(saveDir, $"episode_{currentEp:00000}");
            Directory.CreateDirectory(episodePath); // 确保目录存在
            SaveMetaData(episodePath, currentEp);
            savedMetaDataEpisodes.Add(currentEp);
        }
    }


    /// <summary>
    /// 计算抓取位置
    /// 根据物体的实际大小计算正确的抓取高度
    /// </summary>
    /// <param name="targetObject">目标物体</param>
    /// <returns>计算出的抓取位置</returns>
    private Vector3 CalculateGrabPosition(GameObject targetObject)
    {
        Vector3 grabPosition = targetObject.transform.position;

        // 获取物体的边界框
        Bounds objectBounds = GetObjectBounds(targetObject);

        if (objectBounds.size != Vector3.zero)
        {
            // 使用物体顶部 + 偏移值作为抓取高度
            float objectTopY = objectBounds.max.y;
            grabPosition.y = objectTopY + heightOffset;

            // Debug.Log($"物体 {targetObject.name} 的边界信息:");
            // Debug.Log($"  中心位置: {objectBounds.center}");
            // Debug.Log($"  大小: {objectBounds.size}");
            // Debug.Log($"  顶部Y坐标: {objectTopY}");
            // Debug.Log($"  计算的抓取高度: {grabPosition.y}");
        }
        else
        {
            // 如果无法获取边界框，使用默认偏移
            grabPosition.y = targetObject.transform.position.y + heightOffset;
            // Debug.LogWarning($"无法获取物体 {targetObject.name} 的边界框，使用默认偏移");
        }

        return grabPosition;
    }
    /// <summary>
    /// 获取物体的边界框
    /// 尝试从Collider或Renderer获取边界信息
    /// </summary>
    /// <param name="obj">目标物体</param>
    /// <returns>物体的边界框</returns>
    private Bounds GetObjectBounds(GameObject obj)
    {
        Bounds bounds = new Bounds();

        // 方法1: 尝试从Collider获取边界
        Collider collider = obj.GetComponent<Collider>();
        if (collider != null)
        {
            bounds = collider.bounds; // 【世界空间中的 AABB 包围盒
            Debug.Log($"从Collider获取边界: {bounds}");
            return bounds;
        }

        // 方法2: 尝试从Renderer获取边界
        Renderer renderer = obj.GetComponent<Renderer>();
        if (renderer != null)
        {
            bounds = renderer.bounds;
            // Debug.Log($"从Renderer获取边界: {bounds}");
            return bounds;
        }

        // 方法3: 如果物体有子对象，尝试获取所有子对象的边界
        Renderer[] childRenderers = obj.GetComponentsInChildren<Renderer>();
        if (childRenderers.Length > 0)
        {
            bounds = childRenderers[0].bounds;
            for (int i = 1; i < childRenderers.Length; i++)
            {
                bounds.Encapsulate(childRenderers[i].bounds);
            }
            // Debug.Log($"从子对象Renderer获取合并边界: {bounds}");
            return bounds;
        }

        // 方法4: 如果都没有，尝试获取所有子对象的Collider边界
        Collider[] childColliders = obj.GetComponentsInChildren<Collider>();
        if (childColliders.Length > 0)
        {
            bounds = childColliders[0].bounds;
            for (int i = 1; i < childColliders.Length; i++)
            {
                bounds.Encapsulate(childColliders[i].bounds);
            }
            // Debug.Log($"从子对象Collider获取合并边界: {bounds}");
            return bounds;
        }

        // Debug.LogWarning($"物体 {obj.name} 没有Collider或Renderer组件，无法获取边界");
        return new Bounds(obj.transform.position, Vector3.zero);
    }

    /// <summary>
    /// 选择物体放置位置
    /// 使用初始高度而不是加上偏移
    /// </summary>
    /// <param name="position">放置位置坐标</param>
    private void SelectReleasePosition(Vector3 position)
    {
        if (grabbedObject != null)
        {
            // 使用初始高度，而不是position.y + heightOffset
            targetPosition = new Vector3(position.x, initialHeight, position.z); // 这里直接用initialHeight当作放置点的高，会使得物体自由下落
            targetPosition = new Vector3(position.x, position.y + release_offset, position.z);
            currentState = HandState.MovingToRelease; new Vector3(position.x, initialHeight, position.z);
            // Debug.Log($"选择放置位置：{position}，手部正在移动到放置位置（使用初始高度）");
        }
    }

    /// <summary>
    /// 更新手部状态机
    /// 处理不同状态下的逻辑
    /// </summary>
    private void UpdateHandStateMachine()
    {
        switch (currentState)
        {
            case HandState.Idle:
                UpdateIdle();
                break;

            case HandState.MovingToTarget:
                UpdateMovingToTarget();
                break;

            case HandState.WaitingTarget: // 抓握动态物体才需要
                UpdateWaitingTarget();
                break;

            case HandState.Grabbing:
                UpdateGrabbingState();
                break;

            case HandState.MovingToHome:
                UpdateMovingToHome();
                break;

            case HandState.MovingToRelease:
                UpdateMovingToRelease();
                break;

            case HandState.Releasing:
                UpdateReleasingState();
                break;

            case HandState.ReturningHome:
                UpdateReturningHome();
                break;
        }
    }

    private void UpdateIdle()
    {
        waitTime += Time.deltaTime;
        if (waitTime >= observationTime)
        {
            TrySelectGrabTarget();
            currentState = HandState.MovingToTarget;
            waitTime = 0f;
            handMovingElapsedTime = 0f;
        }
    }


    /// <summary>
    /// 更新移动到抓取目标的状态
    /// 使用手掌心位置进行精确定位
    /// </summary>
    private void UpdateMovingToTarget()
    {
        if (targetObject == null || palmCenter == null)
        {
            if (palmCenter == null)
                Debug.LogWarning("手掌心位置未设置！请在Inspector中指定palmCenter");
            currentState = HandState.Idle;
            return;
        }

        if (handMovingElapsedTime > 2.0f) // 这里定死超时阈值就行，这个逻辑应该不常用
        {
            currentState = HandState.MovingToHome;
            handMovingElapsedTime = 0f;
            return;
        }
        
        // 使用手掌心位置计算距离和移动
        Vector3 palmPosition = palmCenter.position;
        float distance = Vector3.Distance(palmPosition, targetPosition);

        if (distance > 0.05f)
        {
            if (lm != null && lm.enabled)
            {
                Vector3 q0 = targetObject.transform.position;
                float v0 = lm.getCurrentSpeed();        // 动态读取当前速度
                float a = lm.acceleration;
                Vector3 direction = lm.getWorldDirection().normalized;

                float t = interceptTime - handMovingElapsedTime - leadTime;
                t = Mathf.Max(t, 0.01f); // 防止为负

                Vector3 dynamicIntercept = q0 + direction * (v0 * t + 0.5f * a * t * t);
                
                dynamicIntercept.y = targetPosition.y;

                targetPosition = dynamicIntercept;

                float distToTarget = Vector3.Distance(palmCenter.position, targetPosition);
                moveSpeed = distToTarget / t;
                // Debug.Log($" t: {t}, dynamicIntercept: {dynamicIntercept}, moveSpeed: {moveSpeed}");
            }
            else if (cm != null && cm.enabled)
            {
                float angularSpeedDeg = cm != null ? cm.angularSpeed : 0f;  // 角速度（度/秒）
                float radius = cm != null ? cm.radius : 0f;

                Transform referencePlane = cm.referencePlane;
                Vector3 center = cm.GetWorldCenter();

                float angularSpeedRad = angularSpeedDeg * Mathf.Deg2Rad;   // 转换为弧度
                float currentAngleDeg = cm.GetCurrentAngle();
                float currentAngleRad = currentAngleDeg * Mathf.Deg2Rad;

                float t = interceptTime - handMovingElapsedTime - leadTime;
                t = Mathf.Max(t, 0.01f); // 防止为负

                float interceptTheta = currentAngleRad + angularSpeedRad * t;
                Vector3 interceptOffset = new Vector3(Mathf.Cos(interceptTheta), 0f, Mathf.Sin(interceptTheta)) * radius;
                Vector3 dynamicIntercept = center + interceptOffset;
                dynamicIntercept.y = targetPosition.y;

                targetPosition = dynamicIntercept;

                float distToTarget = Vector3.Distance(palmCenter.position, targetPosition);
                moveSpeed = distToTarget / t;
                // Debug.Log($" t: {t}, dynamicIntercept: {dynamicIntercept}, moveSpeed: {moveSpeed}");
            }
            else if (hm != null && hm.enabled)
            {
                // float t = interceptTime - handMovingElapsedTime - leadTime;
                // t = Mathf.Max(t, 0.01f); // 防止为负
                
                // // Vector3 dynamicIntercept = hm.PredictDisplacementAfter(t);

                // dynamicIntercept.y = targetPosition.y;

                // targetPosition = dynamicIntercept;

                // float distToTarget = Vector3.Distance(palmCenter.position, targetPosition);
                // moveSpeed = distToTarget / t;
                // Debug.Log($" t: {t}, dynamicIntercept: {dynamicIntercept}, moveSpeed: {moveSpeed}");
              
            }

            // 计算移动方向和距离
            Vector3 moveDirection = (targetPosition - palmPosition).normalized;
            Vector3 moveAmount = moveDirection * moveSpeed * Time.deltaTime;

            // 移动整个手部（通过根Transform）
            transform.position += moveAmount;

            handMovingElapsedTime += Time.deltaTime;
        }
        else
        {   // 手掌心到达目标位置，开始自动抓取
            // 动态物体需要等四指集合xz和物体中心xz容差很小的时候进入抓握
            currentState = HandState.WaitingTarget;
            handWaitingElapsedTime = 0f;
            Debug.Log("手掌心到达抓取位置，等待物体进入，手动停止物体运动");
            // if (lm != null && lm.enabled)lm.StopMove();
            // if (cm != null && cm.enabled)cm.StopMove();
            // if (hm != null && hm.enabled)hm.StopMove();
        }
    }

    private void UpdateWaitingTarget()
    {
        if (targetObject == null || palmCenter == null)
        {
            if (palmCenter == null)
                Debug.LogWarning("手掌心位置未设置！请在Inspector中指定palmCenter");
            currentState = HandState.Idle;
            return;
        }

        if (handWaitingElapsedTime > timeOutBar)
        {
            currentState = HandState.MovingToHome;
            handWaitingElapsedTime = 0f;
            return;
        }

        // 动态物体需要等四指集合xz和物体中心xz容差很小的时候进入抓握
        Vector3 palmPosition = palmCenter.position;
        Vector3 palmXZ = new Vector3(palmPosition.x, 0, palmPosition.z);
        Vector3 targetXZ = new Vector3(targetObject.position.x, 0, targetObject.position.z);

        float distanceXZ = Vector3.Distance(palmXZ, targetXZ);

        if (distanceXZ < 0.06f) // 容差值可以根据需求调整
        {
            float targetY = grabbedObjectHeight + heightOffset - heightOffset_d + transform.position.y - palmPosition.y;
            Vector3 currentPos = transform.position;
            transform.position = new Vector3(currentPos.x, targetY, currentPos.z);
            currentState = HandState.Grabbing;
            Debug.Log("手掌心与目标物体位置接近，开始抓取");

            CircularMotion cm = scriptLoader.GetComponent<CircularMotion>();
            LinearMotion lm = scriptLoader.GetComponent<LinearMotion>();
            HarmonicMotion hm = scriptLoader.GetComponent<HarmonicMotion>();

            if (cm != null && cm.enabled)
                cm.setIsGrabbed(true); // 设置物体为被抓取状态  
            else if (lm != null && lm.enabled)
                lm.setIsGrabbedAndStopMoving(true); // 设置物体为被抓取状态
            else if (hm != null && hm.enabled)
                hm.setIsGrabbedAndStopMoving(true); // 设置物体为被抓取状态
            else
                Debug.LogError("没有找到对应的圆周运动或直线运动脚本");
        }
        handWaitingElapsedTime += Time.deltaTime;
    }

    /// <summary>
    /// 更新抓握状态
    /// 持续执行抓握动作直到检测到物体
    /// </summary>
    private void UpdateGrabbingState()
    {
        // 如果还没有抓到物体，继续执行抓握动作
        if (!isGrabbingObject)
        {
            // 持续执行抓握动作，让手指不断弯曲
            RotateAllFingerJointsZ();
        }

        // 注意：一旦球形检测到物体，CheckDetectJointsCollision()会自动调用GrabObject()
        // 然后isGrabbingObject会变为true，抓握动作会自动停止
        // 抓取完成后自动回到初始位置
        if (isGrabbingObject)
        {
            if (overRotatedJoints.Count > 0)
            {
                // 新增：如果有超过旋转角度限制的关节，恢复之
                for (int i = 0; i < transforms.Count && i < initialRotations.Count; i++)
                {
                    Transform joint = transforms[i];
                    if (joint != null && overRotatedJoints.Contains(joint))
                    {
                        Vector3 currentRotation = joint.localRotation.eulerAngles; // 必须完整构造并赋值，直接joint.localRotation.eulerAngles.z = initialRotations[i]无效
                        currentRotation.z = initialRotations[i];
                        joint.localRotation = Quaternion.Euler(currentRotation); // 重新生成一个四元数，赋值给 localRotation。
                        Debug.Log($"还原关节 {joint.name} 的Z轴旋转到初始角度 {initialRotations[i]}");
                    }
                }

                // 清空已处理列表
                overRotatedJoints.Clear();
            }
            targetPosition = initialPosition;
            currentState = HandState.MovingToHome;
            Debug.Log("抓取完成，正在返回初始位置");
        }
    }


    /// <summary>
    /// 更新移动到初始位置的状态
    /// 抓取物体后回到初始位置等待放置指令
    /// </summary>
    private void UpdateMovingToHome()
    {
        if (palmCenter == null)
        {
            Debug.LogWarning("手掌心位置未设置！请在Inspector中指定palmCenter");
            return;
        }

        // 使用手掌心位置计算距离和移动
        Vector3 palmPosition = palmCenter.position;
        float distance = Vector3.Distance(palmPosition, targetPosition);

        if (distance > 0.01f) // 这个阈值最好还是调试下看什么最好
        {
            // 计算移动方向和距离
            Vector3 moveDirection = (targetPosition - palmPosition).normalized;
            Vector3 moveAmount = moveDirection * moveSpeed * Time.deltaTime;

            // 移动整个手部（通过根Transform）
            transform.position += moveAmount;
        }
        else
        {
            // 到达初始位置，等待用户点击地板进行放置
            Debug.Log("已回到初始位置，请点击地板选择放置位置");
        }
    }

    /// <summary>
    /// 更新移动到放置位置的状态
    /// 使用手掌心位置进行精确定位
    /// </summary>
    private void UpdateMovingToRelease()
    {
        if (palmCenter == null)
        {
            Debug.LogWarning("手掌心位置未设置！请在Inspector中指定palmCenter");
            return;
        }

        // 使用手掌心位置计算距离和移动
        Vector3 palmPosition = palmCenter.position;
        float distance = Vector3.Distance(palmPosition, targetPosition);

        if (distance > 0.05f)
        {
            // 计算移动方向和距离
            Vector3 moveDirection = (targetPosition - palmPosition).normalized;
            Vector3 moveAmount = moveDirection * moveSpeed * Time.deltaTime;

            // 移动整个手部（通过根Transform）
            transform.position += moveAmount;
        }
        else
        {
            // 手掌心到达放置位置，开始反向抓握（松开手指）
            currentState = HandState.Releasing;
            Debug.Log("手掌心到达放置位置，开始松开手指");
        }
    }

    /// <summary>
    /// 更新松开手指状态
    /// 执行反向抓握，将手指恢复到初始位置
    /// </summary>
    private void UpdateReleasingState()
    {
        // 检查是否所有手指都已经回到初始状态
        bool allFingersReset = true;

        for (int i = 0; i < transforms.Count && i < initialRotations.Count; i++)
        {
            if (transforms[i] != null)
            {
                Vector3 currentRotation = transforms[i].localRotation.eulerAngles;
                float targetRotation = initialRotations[i];

                // 检查当前旋转是否接近初始旋转
                float rotationDifference = Mathf.Abs(Mathf.DeltaAngle(currentRotation.z, targetRotation));

                if (rotationDifference > 1f) // 容差1度
                {
                    allFingersReset = false;

                    // 逐渐回到初始旋转
                    float newRotation = Mathf.MoveTowardsAngle(currentRotation.z, targetRotation, Time.deltaTime * 50f);
                    currentRotation.z = newRotation;
                    transforms[i].localRotation = Quaternion.Euler(currentRotation);
                }
            }
        }

        // 如果所有手指都回到初始状态，立即开启物体重力并放下
        if (allFingersReset)
        {
            EnableGravityAndRelease();
            currentState = HandState.ReturningHome;
            targetPosition = initialPosition;
            Debug.Log("手指已松开，物体重力已开启，正在返回初始位置");
        }
    }

    /// <summary>
    /// 更新返回初始位置状态
    /// 放置完成后回到初始位置
    /// </summary>
    private void UpdateReturningHome()
    {
        if (palmCenter == null)
        {
            Debug.LogWarning("手掌心位置未设置！请在Inspector中指定palmCenter");
            return;
        }

        // 使用手掌心位置计算距离和移动
        Vector3 palmPosition = palmCenter.position;
        float distance = Vector3.Distance(palmPosition, targetPosition);

        if (distance > 0.05f)
        {
            // 计算移动方向和距离
            Vector3 moveDirection = (targetPosition - palmPosition).normalized;
            Vector3 moveAmount = moveDirection * moveSpeed * Time.deltaTime;

            // 移动整个手部（通过根Transform）
            transform.position += moveAmount;
        }
        else
        {
            // 回到初始位置，完成整个循环
            currentState = HandState.Idle;
            targetObject = null;
            grabbedObject = null; // 清空抓取物体
            isGrabbingObject = false; // 重置抓取状态

            // 清空手指接触标志
            fingerContactFlags[1] = false;
            fingerContactFlags[2] = false;
            fingerContactFlags[3] = false;
            fingerContactFlags[4] = false;
            fingerContactFlags[5] = false;

            // 清空超过旋转角度限制的关节列表
            overRotatedJoints.Clear();

            // 恢复所有关节的逻辑在releasing状态下完成了


            Debug.Log("已返回初始位置，准备下一次抓取");
        }
    }

    /// <summary>
    /// 开启物体重力并解除父子关系
    /// 在手指松开的瞬间调用
    /// </summary>
    private void EnableGravityAndRelease()
    {
        if (grabbedObject != null)
        {
            // 取消父子关系
            grabbedObject.transform.SetParent(null);

            // 立即恢复物体的重力
            Rigidbody rb = grabbedObject.GetComponent<Rigidbody>();
            if (rb != null)
            {
                rb.useGravity = true;
                rb.isKinematic = false;
            }

            Debug.Log($"物体重力已开启，开始下落：{grabbedObject.name}");
            grabbedObject = null;
        }

        // 重置抓取状态
        isGrabbingObject = false;
    }

    /// <summary>
    /// 抓取物体
    /// 将物体设置为手部的子对象并禁用重力
    /// </summary>
    /// <param name="obj">要抓取的物体</param>
    private void GrabObject(GameObject obj)
    {
        if (obj != null && grabbedObject == null)
        {
            grabbedObject = obj;

            // 设置物体为手部的子对象
            obj.transform.SetParent(this.transform);

            // 禁用物体的重力
            Rigidbody rb = obj.GetComponent<Rigidbody>();
            if (rb != null)
            {
                rb.useGravity = false;
                rb.isKinematic = true; // 设置为运动学模式，防止物理干扰
            }

            Debug.Log($"成功抓取物体：{obj.name}");
        }
    }

    void DrawTransparentPlane(Vector3 center, Vector3 normal, float size, Color outlineColor)
    {
        Vector3 up = Vector3.up;
        if (Vector3.Dot(normal, up) > 0.99f)
            up = Vector3.forward;

        Vector3 right = Vector3.Cross(normal, up).normalized;
        Vector3 tangent = Vector3.Cross(right, normal).normalized;

        Vector3 p1 = center + (right + tangent) * size;
        Vector3 p2 = center + (right - tangent) * size;
        Vector3 p3 = center + (-right - tangent) * size;
        Vector3 p4 = center + (-right + tangent) * size;

        Vector3[] verts = new Vector3[] { p1, p2, p3, p4 };

        Handles.DrawSolidRectangleWithOutline(verts, Handles.color, outlineColor);
    }

    void OnDrawGizmos()
    {
        // 在参考平面上绘制手掌心点
        Vector3 palmPosition = palmCenter.position;
        if(lm!=null){
            Transform referencePlane = lm.referencePlane;
            Vector3 palmXZ_plane = new Vector3(palmPosition.x, referencePlane.position.y, palmPosition.z);
            Gizmos.color = Color.red;
            Gizmos.DrawSphere(palmXZ_plane, 0.05f);

            if (targetObject != null)
            {
                Gizmos.color = Color.green;
                Vector3 targetObject_plane = new Vector3(targetObject.position.x, referencePlane.position.y, targetObject.position.z);
                Gizmos.DrawSphere(targetObject_plane, 0.05f);
            }
        }

        // 绘制掌心坐标和targetPosition空间状态
        if (targetPosition!=null){
            Gizmos.color = Color.blue;
            Gizmos.DrawSphere(palmCenter.position, 0.05f);
            Gizmos.DrawLine(palmCenter.position, targetPosition);
        }

        // 绘制maxZDistance和minZDistance对应的两个平面
        // 获取当前相机的位置
        Vector3 camPos = handCamera.transform.position;
        // Vector3 forward = handCamera.transform.forward.normalized;
        Vector3 forward = Vector3.forward;

        // 计算两个平面的位置
        Vector3 minPlanePos = camPos + forward * minZDistance;
        Vector3 maxPlanePos = camPos + forward * maxZDistance;

        // 绘制 minZDistance 平面（绿色透明 + 黑色边框）
        Handles.color = new Color(0f, 1f, 0f, 0.2f); // 绿色透明
        DrawTransparentPlane(minPlanePos, forward, 3f, Color.black);

        // 绘制 maxZDistance 平面（红色透明 + 黑色边框）
        Handles.color = new Color(1f, 0f, 0f, 0.2f); // 红色透明
        DrawTransparentPlane(maxPlanePos, forward, 3f, Color.black);

    }
}
