using UnityEngine;
using System.IO;
using System.Collections.Generic;
using WebSocketSharp;

namespace EpisodeControl
{
    // 数据传输格式
    [System.Serializable]
    public class JointData
    {
        public string name;
        public float[] position;
        public float[] rotation;
    }

    [System.Serializable]
    public class HandData
    {
        public int episode;
        public List<JointData> joints = new List<JointData>();
    }

    // 基础配置类（不用于实际解析）
    public abstract class BaseConfig
    {
        public int index;

        public float intercept_time;
        public float early_arrival_time;
        public float observe_time;

        public string objectName;
    }

    // 直线运动配置及其包装类
    [System.Serializable]
    public class LinearConfig : BaseConfig
    {
        public Vector3 initial_position;
        public Vector3 direction;
        public float initial_speed;
        public float acceleration;
    }

    [System.Serializable]
    public class LinearConfigWrapper
    {
        public List<LinearConfig> configs;
    }

    
    // 圆周运动配置及其包装类
    [System.Serializable]
    public class CircularConfig : BaseConfig
    {
        public Vector3 localCenter;
        public float radius;
        public float angularSpeed;
        public float startAngleDeg;
    }

    [System.Serializable]
    public class CircularConfigWrapper
    {
        public List<CircularConfig> configs;
    }

    
    // 谐波运动配置及其包装类
    [System.Serializable]
    public class HarmonicConfig : BaseConfig
    {
        public Vector3 localCenter;
        public Vector3 localDirection;
        public float amplitude;
        public float angularFrequency;
        public float initialPhase;
    }

    [System.Serializable]
    public class HarmonicConfigWrapper
    {
        public List<HarmonicConfig> configs;
    }

    // 配置类型枚举
    public enum ConfigType
    {
        Linear,
        Circular,
        Harmonic
    }

    // 新增：元数据类，用于保存每个episode的详细信息
    [System.Serializable]
    public class MetaData
    {
        public string task_type;
        public Vector3 predicted_intercept_position;
        public float hand_move_speed;
        public int episode;
        
        // 不同类型的运动参数，只有对应类型的会被赋值
        public LinearConfig linear_params;
        public CircularConfig circular_params; 
        public HarmonicConfig harmonic_params;
    }

    public class EpisodeManager
    {
        private string configPath;
        private ConfigType configType;
        private List<LinearConfig> linearConfigs;
        private List<CircularConfig> circularConfigs;
        private List<HarmonicConfig> harmonicConfigs;
        public int currentEpisode = 0;
        private Dynamic controller;
        public BaseConfig currentConfig; // 新增：保存当前使用的配置

        public EpisodeManager(string configPath, ConfigType type, Dynamic dynamicController, int startEpisode=0)
        {
            this.configPath = configPath;
            this.configType = type;
            controller = dynamicController;
            currentEpisode = startEpisode;
            LoadConfig();
        }

        public void LoadConfig()
        {
            string fullPath = Path.Combine(Application.dataPath, "..", configPath);
            if (!File.Exists(fullPath))
            {
                Debug.LogError($"配置文件不存在: {fullPath}");
                return;
            }

            string json = File.ReadAllText(fullPath);
            string wrappedJson = "{\"configs\":" + json + "}";

            try
            {
                switch (configType)
                {
                    case ConfigType.Linear:
                        var linearWrapper = JsonUtility.FromJson<LinearConfigWrapper>(wrappedJson);
                        linearConfigs = linearWrapper.configs;
                        Debug.Log($"成功加载 {linearConfigs.Count} 个直线运动配置，将从第{currentEpisode}组参数开始");
                        break;

                    case ConfigType.Circular:
                        var circularWrapper = JsonUtility.FromJson<CircularConfigWrapper>(wrappedJson);
                        circularConfigs = circularWrapper.configs;
                        Debug.Log($"成功加载 {circularConfigs.Count} 个圆周运动配置，将从第{currentEpisode}组参数开始");
                        break;

                    case ConfigType.Harmonic:
                        var harmonicWrapper = JsonUtility.FromJson<HarmonicConfigWrapper>(wrappedJson);
                        harmonicConfigs = harmonicWrapper.configs;
                        Debug.Log($"成功加载 {harmonicConfigs.Count} 个谐波运动配置，将从第{currentEpisode}组参数开始");
                        break;
                }
            }
            catch (System.Exception e)
            {
                Debug.LogError($"配置加载错误: {e.Message}");
            }
        }

        public void InitFromConfig()
        {
            int configCount = 0;

            switch (configType)
            {
                case ConfigType.Linear when linearConfigs != null:
                    if (currentEpisode < linearConfigs.Count)
                    {
                        var cfg = linearConfigs[currentEpisode];

                        Debug.Log($@"[Episode {currentEpisode}/{linearConfigs.Count}]
                        参数设置:
                        - objectName: {cfg.objectName}, initial_position: {cfg.initial_position}, direction: {cfg.direction}, initial_speed: {cfg.initial_speed}, acceleration: {cfg.acceleration}
                        - intercept_time: {cfg.intercept_time}, observe_time: {cfg.observe_time}, early_arrival_time: {cfg.early_arrival_time}
                        ");

                        controller.interceptTime = cfg.intercept_time;
                        controller.observationTime = cfg.observe_time;
                        controller.leadTime = cfg.early_arrival_time;

                        if (controller.lm != null)
                        {
                            controller.lm.SetMotion(
                                cfg.objectName,
                                cfg.initial_position,
                                cfg.direction.normalized,
                                cfg.initial_speed,
                                cfg.acceleration
                            );
                        }
                        currentConfig = cfg;
                        configCount = linearConfigs.Count;
                    }
                    break;

                case ConfigType.Circular when circularConfigs != null:
                    if (currentEpisode < circularConfigs.Count)
                    {
                        var cfg = circularConfigs[currentEpisode];

                        Debug.Log($@"[Episode {currentEpisode}/{circularConfigs.Count}]
                        参数设置:
                        - objectName: {cfg.objectName}, localCenter: {cfg.localCenter}, radius: {cfg.radius}, angularSpeed: {cfg.angularSpeed}, startAngleDeg: {cfg.startAngleDeg}
                        - intercept_time: {cfg.intercept_time}, observe_time: {cfg.observe_time}, early_arrival_time: {cfg.early_arrival_time}
                        ");

                        controller.interceptTime = cfg.intercept_time;
                        controller.observationTime = cfg.observe_time;
                        controller.leadTime = cfg.early_arrival_time;

                        if (controller.cm != null)
                        {
                            controller.cm.SetMotion(
                                cfg.objectName,
                                cfg.localCenter,
                                cfg.radius,
                                cfg.angularSpeed,
                                cfg.startAngleDeg
                            );
                        }
                        currentConfig = cfg;
                        configCount = circularConfigs.Count;
                    }
                    break;

                case ConfigType.Harmonic when harmonicConfigs != null:
                    if (currentEpisode < harmonicConfigs.Count)
                    {
                        var cfg = harmonicConfigs[currentEpisode];

                        Debug.Log($@"[Episode {currentEpisode}/{harmonicConfigs.Count}]
                        参数设置:
                        - objectName: {cfg.objectName}, localCenter: {cfg.localCenter}, localDirection: {cfg.localDirection}, amplitude: {cfg.amplitude}, angularFrequency: {cfg.angularFrequency}, initialPhase: {cfg.initialPhase}
                        - intercept_time: {cfg.intercept_time}, observe_time: {cfg.observe_time}, early_arrival_time: {cfg.early_arrival_time}
                        ");

                        controller.interceptTime = cfg.intercept_time;
                        controller.observationTime = cfg.observe_time;
                        controller.leadTime = cfg.early_arrival_time;

                        if (controller.hm != null)
                        {
                            controller.hm.SetMotion(
                                cfg.objectName,
                                cfg.localCenter,
                                cfg.localDirection,
                                cfg.amplitude,
                                cfg.angularFrequency,
                                cfg.initialPhase
                            );
                        }
                        currentConfig = cfg;
                        configCount = harmonicConfigs.Count;
                    }
                    break;
            }

            if (currentConfig == null)
            {
                Debug.LogError($"无法初始化配置 {currentEpisode}/{configCount}");
            }
        }

        public void TryAdvanceEpisode()
        {
            int maxCount = configType switch
            {
                ConfigType.Linear => linearConfigs?.Count ?? 0,
                ConfigType.Circular => circularConfigs?.Count ?? 0,
                ConfigType.Harmonic => harmonicConfigs?.Count ?? 0,
                _ => 0
            };

            if (currentEpisode < maxCount - 1)
            {
                InitFromConfig();
                currentEpisode++;
            }
            else
            {
                Debug.Log("✅ 所有配置采样完毕");
            }
        }

        // 新增：获取当前episode的配置信息，用于生成meta_data
        public BaseConfig GetCurrentConfig()
        {
            switch (configType)
            {
                case ConfigType.Linear when linearConfigs != null:
                    if (currentEpisode < linearConfigs.Count)
                        return linearConfigs[currentEpisode];
                    break;
                    
                case ConfigType.Circular when circularConfigs != null:
                    if (currentEpisode < circularConfigs.Count)
                        return circularConfigs[currentEpisode];
                    break;
                    
                case ConfigType.Harmonic when harmonicConfigs != null:
                    if (currentEpisode < harmonicConfigs.Count)
                        return harmonicConfigs[currentEpisode];
                    break;
            }
            return null;
        }

        // 新增：获取配置类型字符串
        public string GetConfigTypeString()
        {
            return configType switch
            {
                ConfigType.Linear => "linear",
                ConfigType.Circular => "circular", 
                ConfigType.Harmonic => "harmonic",
                _ => "unknown"
            };
        }
    }
}
