using System.Collections;
using System.Collections.Generic;
using UnityEngine;

using WebSocketSharp; // 使用 websocket-sharp 的命名空间

using System.Linq;
using System.Text;

using UnityEngine.Splines; // 使用Spline样条
using Unity.Mathematics; // 需要使用 float4x4 类型




public class HandDraw : MonoBehaviour
{
    public Transform finger_1;
    public Transform finger_1_1;
    public Transform finger_2_1;
    public Transform finger_2_2;
    public Transform finger_2_3;
    public Transform finger_3_1;
    public Transform finger_3_2;
    public Transform finger_3_3;
    public Transform finger_4_1;
    public Transform finger_4_2;
    public Transform finger_4_3;
    public Transform finger_5_1;
    public Transform finger_5_2;
    public Transform finger_5_3;
    public Transform finger_nail_1;
    public Transform finger_nail_2;
    public Transform finger_nail_3;
    public Transform finger_nail_4;
    public Transform finger_nail_5;
    public Transform palmCenter; // 四指集合位置标记，用于精确定位抓取位置，**四指的集合**
    private List<Transform> transforms = new List<Transform>(); // 存储所有手指关节
    private List<Transform> tipJoints = new List<Transform>(); // 存储所有带碰撞球的关节
    private bool isGrabbingObject = false; // 标记是否已经抓到物体
    private float sphereRadius = 0.05f; // 球形射线检测半径

    // 抓取系统相关变量
    private enum HandState { Idle, MovingToTarget, Grabbing, PullingOrPush, Releasing, MovingToHome } // 手部状态机
    private HandState currentState = HandState.Idle; // 当前手部状态
    private Transform targetObject; // 目标抓取物体
    private GameObject grabbedObject; // 当前抓取的物体
    //private float grabbedObjectHeight; // 新增：抓取物体的高度
    // private float release_offset = 0.6f; // 新增：放置高度偏移（用于非悬空/小幅度悬空放置）
    private Vector3 targetPosition; // 目标移动位置
    private Vector3 initialPosition; // 初始位置
    private float initialHeight; // 初始高度
    private float moveSpeed = 2.0f; // 手部移动速度
    private float heightOffset = -0.1f; // 抓取时的高度偏移，因为是从下面拉开抽屉，所以需要向下偏移一点
    private List<float> initialRotations = new List<float>(); // 存储所有关节的初始z轴旋转
    private float maxGrabRotation = 90f; // 最大抓取旋转角度（度）

    public Transform handsRoot; // 手腕关节

    public Transform armRoot; // 手臂根节点


    // 改动新增，为了“所有关节碰撞球都检测”，操作抽屉的时候大拇指不转动
    private Dictionary<int, bool> fingerContactFlags = new Dictionary<int, bool>()
    {
        {1, true}, {2, false}, {3, false}, {4, false}, {5, false}
    };
    private List<Transform> overRotatedJoints = new List<Transform>(); // 超过旋转角度限制的关节




    // 推拉抽屉使用
    // private Transform drawerTransform; // 抽屉的Transform
    // private Transform handlerTransform; // 抽屉把手的Transform
    // private float translate_obj_speed = 0.1f; // 推入或拉出的速度，每秒0.1单位
    // private float closedPosition = 0.17f; // 抽屉关闭时的位置
    // private float openPosition = 0.38f; // 抽屉打开时的位置
    // private string operation; // 执行什么操作，"open" 或 "close"
    // private float rotate_arm_speed = 90f; // 手臂旋转速度，每秒90度

    // 手物交互相关变量
    private Transform operationTransform; // 被操控的子物体的Transform
    private Transform contactTransform; // 手需要接触物体的Transform
    private string operationOjectName; // 当前操作的什么物体，“BesideTable” 或 “Box” 或 “Closet”
    private Transform handMoveReference; // 手移动参考点，通常是手掌中心位置

    private string operation; // 操作，"open" 或 "close"
    private string operationAxis; // 操作轴，"x", "y", "z"
    private string operationType; // 操作类型，"translate" 或 "rotate"

    private float closedPosition; // 被操作物体关闭时候的位置（旋转角or位置）
    private float openPosition; // 被操作物体打开时候的位置（旋转角or位置）

    private float rotate_arm_speed = 90f; // 靠近过程中如果需要arm旋转，定义旋转速度
    private float rotate_obj_speed; // 被操作物体在旋转时速度
    private float translate_obj_speed; // 被操作物体在旋转时速度

    public SplineContainer splineContainer;  // 制定靠近目标物的轨迹，拖拽 Spline GameObject
    private float traveledDistance = 0f;
    private float totalLength;
    private Spline spline;

    // socket传输数据使用
    // private WebSocket ws;
    // private int counter = 0;
    // private float sendInterval = 1.0f;  // 每 1 秒发一次
    // private float lastSendTime = 0f;

    // 相机拍摄每帧图像
    public Camera handCamera;  // 绑定的摄像机
    public RenderTexture renderTexture;  // 用于绑定相机输出
    private Texture2D screenTexture;

    // Start is called before the first frame update
    // 初始化所有手指关节列表和指尖关节列表
    void Start()
    {
        // 添加所有手指关节到总列表
        transforms.Add(finger_1);
        transforms.Add(finger_1_1);
        transforms.Add(finger_2_1);
        transforms.Add(finger_2_2);
        transforms.Add(finger_2_3);
        transforms.Add(finger_3_1);
        transforms.Add(finger_3_2);
        transforms.Add(finger_3_3);
        transforms.Add(finger_4_1);
        transforms.Add(finger_4_2);
        transforms.Add(finger_4_3);
        transforms.Add(finger_5_1);
        transforms.Add(finger_5_2);
        transforms.Add(finger_5_3);

        // 改动：检测所有的关节，用tipJoints储存所有关节用于碰撞检测
        tipJoints.Add(finger_1);
        tipJoints.Add(finger_1_1);
        tipJoints.Add(finger_2_1);
        tipJoints.Add(finger_2_2);
        tipJoints.Add(finger_2_3);
        tipJoints.Add(finger_3_1);
        tipJoints.Add(finger_3_2);
        tipJoints.Add(finger_3_3);
        tipJoints.Add(finger_4_1);
        tipJoints.Add(finger_4_2);
        tipJoints.Add(finger_4_3);
        tipJoints.Add(finger_5_1);
        tipJoints.Add(finger_5_2);
        tipJoints.Add(finger_5_3);


        // 记录初始位置和高度
        initialPosition = transform.position;
        initialHeight = transform.position.y;


        // 记录所有关节的初始z轴旋转角度
        foreach (var joint in transforms)
        {
            if (joint != null)
            {
                initialRotations.Add(joint.localRotation.eulerAngles.z);
            }
            else
            {
                initialRotations.Add(0f);
            }
        }

        // socket
        // ws = new WebSocket("ws://localhost:8765");
        // ws.Connect();
        // Debug.Log("已连接到 Python WebSocket 服务器");

        // ego摄像机初始化贴图
        screenTexture = new Texture2D(renderTexture.width, renderTexture.height, TextureFormat.RGB24, false);
        handCamera.targetTexture = renderTexture;

        // Spline 样条
        spline = splineContainer.Spline;
        totalLength = SplineUtility.CalculateLength(spline, splineContainer.transform.localToWorldMatrix);
        traveledDistance = 0f;

    }
    void FixedUpdate()
    {
        // if (ws != null && ws.ReadyState == WebSocketState.Open)
        // {
        //     if (Time.time - lastSendTime >= sendInterval)
        //     {
        //         HandData handData = new HandData();

        //         // 读取 RenderTexture 内容到 Texture2D
        //         RenderTexture.active = renderTexture;
        //         screenTexture.ReadPixels(new Rect(0, 0, renderTexture.width, renderTexture.height), 0, 0);
        //         screenTexture.Apply();
        //         RenderTexture.active = null;

        //         // 编码为 JPG
        //         byte[] imageBytes = screenTexture.EncodeToJPG();
        //         string imageBase64 = System.Convert.ToBase64String(imageBytes);

        //         handData.image = imageBase64;

        //         foreach (Transform joint in transforms)
        //         {
        //             if (joint != null)
        //             {
        //                 JointData jd = new JointData
        //                 {
        //                     name = joint.name,
        //                     position = new float[] {
        //             joint.position.x,
        //             joint.position.y,
        //             joint.position.z
        //         },
        //                     rotation = new float[] {
        //             joint.eulerAngles.x,
        //             joint.eulerAngles.y,
        //             joint.eulerAngles.z
        //         }
        //                 };
        //                 handData.joints.Add(jd);
        //             }
        //         }
        //         string json = JsonUtility.ToJson(handData);
        //         ws.Send(json);

        //         lastSendTime = Time.time;
        //     }
        // }
    }

    // Update is called once per frame
    // 检测用户输入并执行相应的旋转操作，同时持续进行碰撞检测和抓取系统
    void Update()
    {
        // 处理鼠标点击输入
        HandleMouseInput();

        // 更新手部状态机
        UpdateHandStateMachine();

        // 持续检测指尖关节是否碰到物体
        CheckTipJointsCollision2();

    }

    /// <summary>
    /// 根据鼠标点击的物体
    /// 设置要执行的具体动作规划参数【（1）open or close；（2）手移动参考点；（3）平移or旋转，速度如何，轴如何
    /// 返回根据 被点击的物体 有没有成功设置动作规划参数
    /// </summary>
    private bool SelectActionDetails(GameObject clickedObject)
    {
        string clickedObjectName = clickedObject.name;
        if (clickedObjectName.Contains("BedsideTable"))
        {
            Debug.Log($"查找路径：{clickedObject.name}/SM__BesideTable/Body01/Body02/Drawer/handler");
            operationTransform = clickedObject.transform.Find("SM__BesideTable/Body01/Body02/Drawer");
            contactTransform = clickedObject.transform.Find("SM__BesideTable/Body01/Body02/Drawer/handler");

            if (operationTransform != null && contactTransform != null)
            {

                // 设置操作轴、操作类型、开关状态、运动速度、操作物体名称
                operationAxis = "y"; // 抽屉是沿y轴移动
                operationType = "translate"; // 抽屉是平移操作
                closedPosition = 0.17f; // 抽屉关闭时的位置
                openPosition = 0.38f; // 抽屉打开时的位置
                rotate_arm_speed = 90f; // 手臂旋转速度，每秒90度
                translate_obj_speed = 0.1f; // 抽屉平移速度，每秒0.1单位
                operationOjectName = "BesideTable"; // 当前操作的物体名称
                Debug.Log($"设置操作参数：operation={operation}, operationAxis={operationAxis}, operationType={operationType}, closedPosition={closedPosition}, openPosition={openPosition}, rotate_arm_speed={rotate_arm_speed}, translate_obj_speed={translate_obj_speed}");

                // 根据被操作物体init状态设置operation
                Vector3 drawerInitPosition = operationTransform.localPosition;
                if (Mathf.Abs(drawerInitPosition.y - closedPosition) <= 0.03f) operation = "open";
                else operation = "close";
                Debug.Log($"初始抽屉局部位置: {drawerInitPosition.y}, 将执行操作: {operation}");

                // 设置抓取位置
                Vector3 targetPosition = contactTransform.transform.position;
                Bounds objectBounds = GetObjectBounds(contactTransform.gameObject); // 默认是世界坐标系（Global Space）下的坐标
                if (objectBounds.size != Vector3.zero)
                {
                    // 使用物体顶部 + 偏移值作为抓取高度
                    float objectTopY = objectBounds.min.y; // 手从下面拉开抽屉
                    targetPosition.y = objectTopY + heightOffset;

                    Debug.Log($"物体 {contactTransform.name} 的边界信息:");
                    Debug.Log($"  中心位置: {objectBounds.center}");
                    Debug.Log($"  大小: {objectBounds.size}");
                    Debug.Log($"  顶部Y坐标: {objectTopY}");
                    Debug.Log($"  计算的抓取高度: {targetPosition.y}");
                    return true;
                }
                else
                {
                    // 如果无法获取边界框，使用默认偏移
                    targetPosition.y = targetObject.transform.position.y + heightOffset;
                    Debug.LogWarning($"无法获取物体 {targetObject.name} 的边界框，使用默认偏移");
                    return false;
                }
            }
            else
            {
                Debug.LogWarning($"点击的床头柜没有抽屉或把手");
                return false;
            }

        }
        else if (clickedObjectName.Contains("Box"))
        {
            Debug.Log($"查找路径：{clickedObject.name}/SM_Box_Close/Box_body/Body_Hinge");
            operationTransform = clickedObject.transform.Find("SM_Box_Close/Box_body/Body_Hinge");
            contactTransform = clickedObject.transform.Find("SM_Box_Close/Box_body/Body_Hinge");

            if (operationTransform != null && contactTransform != null)
            {
                // 设置操作轴、操作类型、开关状态、运动速度
                operationAxis = "x"; // 抽屉是沿x轴旋转
                operationType = "rotate"; // 抽屉是平移操作
                closedPosition = 0f; // 盒子关闭时的位置
                openPosition = 135f; // 盒子打开时的位置
                rotate_arm_speed = 90f; // 手臂旋转速度，每秒90度
                translate_obj_speed = 5f; // 盒子盖子旋转速度，每秒5度
                operationOjectName = "Box"; // 当前操作的物体名称
                Debug.Log($"设置操作参数：operation={operation}, operationAxis={operationAxis}, operationType={operationType}, closedPosition={closedPosition}, openPosition={openPosition}, rotate_arm_speed={rotate_arm_speed}, translate_obj_speed={translate_obj_speed}");

                // 根据被操作物体init状态设置operation
                Vector3 InitRotation = operationTransform.eulerAngles;
                if (Mathf.Abs(InitRotation.x - closedPosition) <= 3f) operation = "open";
                else operation = "close";
                Debug.Log($"初始局部位置: {InitRotation.x}, 将执行操作: {operation}");

                // 设置抓取位置
                Vector3 targetPosition = contactTransform.transform.position;
                Bounds objectBounds = GetObjectBounds(contactTransform.gameObject); // 默认是世界坐标系（Global Space）下的坐标
                if (objectBounds.size != Vector3.zero)
                {
                    // 使用物体顶部 + 偏移值作为抓取高度
                    float objectTopY = objectBounds.min.y; // 手从下面拉开抽屉
                    targetPosition.y = objectTopY + heightOffset;

                    Debug.Log($"物体 {contactTransform.name} 的边界信息:");
                    Debug.Log($"  中心位置: {objectBounds.center}");
                    Debug.Log($"  大小: {objectBounds.size}");
                    Debug.Log($"  顶部Y坐标: {objectTopY}");
                    Debug.Log($"  计算的抓取高度: {targetPosition.y}");
                    return true;
                }
                else
                {
                    // 如果无法获取边界框，使用默认偏移
                    targetPosition.y = targetObject.transform.position.y + heightOffset;
                    Debug.LogWarning($"无法获取物体 {targetObject.name} 的边界框，使用默认偏移");
                    return false;
                }
            }
            else
            {
                Debug.LogWarning($"点击的盒子没有盖子");
                return false;
            }

        }
        else if (clickedObjectName.Contains("Closet"))
        {
            Debug.Log($"查找路径：{clickedObject.name}/SK_Closet_close/joint1/joint2/joint3");
            operationTransform = clickedObject.transform.Find("SK_Closet_close/joint1/joint2/joint3");
            contactTransform = clickedObject.transform.Find("SK_Closet_close/joint1/joint2/joint3");

            if (operationTransform != null && contactTransform != null)
            {
                // 设置操作轴、操作类型、开关状态、运动速度
                operationAxis = "z"; // 衣柜门是沿z轴旋转
                operationType = "rotate"; // 衣柜门是旋转操作
                closedPosition = -90.0f; // 衣柜门关闭时的位置
                openPosition = 40.0f; // 衣柜门打开时的位置
                rotate_arm_speed = 90f; // 手臂旋转速度，每秒90度
                translate_obj_speed = 5f; // 衣柜门旋转速度，每秒5度
                operationOjectName = "Closet"; // 当前操作的物体名称
                Debug.Log($"设置操作参数：operation={operation}, operationAxis={operationAxis}, operationType={operationType}, closedPosition={closedPosition}, openPosition={openPosition}, rotate_arm_speed={rotate_arm_speed}, translate_obj_speed={translate_obj_speed}");

                // 根据被操作物体init状态设置operation
                Vector3 InitRotation = operationTransform.eulerAngles;
                if (Mathf.Abs(InitRotation.z - closedPosition) <= 3f) operation = "open";
                else operation = "close";
                Debug.Log($"初始局部位置: {InitRotation.z}, 将执行操作: {operation}");

                // 设置抓取位置
                Vector3 targetPosition = contactTransform.transform.position;
                Bounds objectBounds = GetObjectBounds(contactTransform.gameObject); // 默认是世界坐标系（Global Space）下的坐标
                if (objectBounds.size != Vector3.zero)
                {
                    // 使用物体顶部 + 偏移值作为抓取高度
                    float objectTopY = objectBounds.min.y; // 手从下面拉开抽屉
                    targetPosition.y = objectTopY + heightOffset;

                    Debug.Log($"物体 {contactTransform.name} 的边界信息:");
                    Debug.Log($"  中心位置: {objectBounds.center}");
                    Debug.Log($"  大小: {objectBounds.size}");
                    Debug.Log($"  顶部Y坐标: {objectTopY}");
                    Debug.Log($"  计算的抓取高度: {targetPosition.y}");
                    return true;
                }
                else
                {
                    // 如果无法获取边界框，使用默认偏移
                    targetPosition.y = targetObject.transform.position.y + heightOffset;
                    Debug.LogWarning($"无法获取物体 {targetObject.name} 的边界框，使用默认偏移");
                    return false;
                }
            }
            else
            {
                Debug.LogWarning($"点击的盒子没有盖子");
                return false;
            }
        }
        else
        {
            Debug.LogWarning($"点击的物体{clickedObjectName}不是床头柜 或 盒子 或 衣柜：");
            return false;
        }

    }

    /// <summary>
    /// 对所有手指关节进行z轴旋转
    /// 只旋转还没有接触到物体的手指，已接触的手指停止旋转
    /// </summary>
    private void RotateAllFingerJointsZ()
    {
        // 新版本：直接使用tipJoints列表进行旋转
        if (currentState != HandState.Grabbing || targetObject == null)
        {
            // 全部旋转（无碰撞限制）
            RotateFingerGroup(finger_2_1, finger_2_2, finger_2_3, true);
            RotateFingerGroup(finger_3_1, finger_3_2, finger_3_3, true);
            RotateFingerGroup(finger_4_1, finger_4_2, finger_4_3, true);
            RotateFingerGroup(finger_5_1, finger_5_2, finger_5_3, true);
            return;
        }

        // 使用碰撞检测得到的状态判断是否旋转
        RotateFingerGroup(finger_2_1, finger_2_2, finger_2_3, !fingerContactFlags[2]); // 食指
        RotateFingerGroup(finger_3_1, finger_3_2, finger_3_3, !fingerContactFlags[3]); // 中指
        RotateFingerGroup(finger_4_1, finger_4_2, finger_4_3, !fingerContactFlags[4]); // 无名指
        RotateFingerGroup(finger_5_1, finger_5_2, finger_5_3, !fingerContactFlags[5]); // 小指
    }


    /// <summary>
    /// 旋转手指组（拇指版本 - 2个关节）
    /// </summary>
    /// <param name="joint1">第一个关节</param>
    /// <param name="joint2">第二个关节</param>
    /// <param name="canRotate">是否可以旋转</param>
    private void RotateFingerGroup(Transform joint1, Transform joint2, bool canRotate)
    {
        if (!canRotate) return;

        Transform[] joints = { joint1, joint2 };

        foreach (var joint in joints)
        {
            if (joint != null)
            {
                // 获取当前的本地旋转
                Vector3 currentRotation = joint.localRotation.eulerAngles;

                // 在z轴上增加旋转
                currentRotation.z += Time.deltaTime * 10;

                // 应用新的旋转值
                joint.localRotation = Quaternion.Euler(currentRotation);
            }
        }
    }

    /// <summary>
    /// 旋转手指组（四指版本 - 3个关节）
    /// </summary>
    /// <param name="joint1">第一个关节</param>
    /// <param name="joint2">第二个关节</param>
    /// <param name="joint3">第三个关节</param>
    /// <param name="canRotate">是否可以旋转</param>
    private void RotateFingerGroup(Transform joint1, Transform joint2, Transform joint3, bool canRotate)
    {
        if (!canRotate)
        {
            Debug.Log($"手指已接触物体，停止旋转: {joint3?.name}");
            return;
        }

        Transform[] joints = { joint1, joint2, joint3 };

        foreach (var joint in joints)
        {
            if (joint != null)
            {
                // 获取当前的本地旋转
                Vector3 currentRotation = joint.localRotation.eulerAngles;

                // 在z轴上增加旋转
                currentRotation.z += Time.deltaTime * 10;

                // 应用新的旋转值
                joint.localRotation = Quaternion.Euler(currentRotation);
            }
        }
    }

    /// <summary>
    /// 检测所有3号关节（指尖关节）是否与物体发生碰撞
    /// 使用球形射线检测，半径为0.04单位
    /// 只有当所有手指都接触到目标物体时才算抓取成功
    /// 如果旋转角度超过90度还没抓到，也算抓取完成
    /// </summary>
    private void CheckTipJointsCollision()
    {
        if (currentState != HandState.Grabbing || isGrabbingObject || targetObject == null) // 状态机不是Grabbing 或 已经抓取物体 或 没有目标物体时不执行
            return;

        int fingersInContact = 0;
        GameObject detectedObject = null;

        // 遍历所有指尖关节进行碰撞检测
        foreach (var tipJoint in tipJoints)
        {
            if (tipJoint != null)
            {
                // 使用球形射线检测，以关节位置为中心
                Collider[] hitColliders = Physics.OverlapSphere(tipJoint.position, sphereRadius);

                bool fingerInContact = false;

                // 检查是否碰到了目标物体
                foreach (var collider in hitColliders)
                {
                    // 检查是否是目标物体（排除手部自身和地板）
                    if (!IsHandComponent(collider.transform) &&
                        collider.gameObject.layer != LayerMask.NameToLayer("Ground") &&
                        (collider.transform == targetObject || collider.transform.IsChildOf(targetObject) || targetObject.IsChildOf(collider.transform)))
                    {
                        fingerInContact = true;
                        detectedObject = collider.gameObject;
                        break;
                    }
                }

                if (fingerInContact)
                {
                    fingersInContact++;
                }
            }
        }

        // 检查是否达到最大旋转角度
        bool maxRotationReached = CheckMaxRotationReached();

        // 输出调试信息
        Debug.Log($"接触检测: {fingersInContact}/{tipJoints.Count} 根手指接触到目标物体");
        if (maxRotationReached)
        {
            Debug.Log("手指旋转已达到90度限制");
        }

        // 抓取成功条件：所有手指都接触到物体 OR 达到最大旋转角度
        if ((fingersInContact >= tipJoints.Count && detectedObject != null) || maxRotationReached)
        {
            isGrabbingObject = true;
            if (detectedObject != null)
            {
                GrabObject(detectedObject);
                Debug.Log($"抓取成功！{fingersInContact}/{tipJoints.Count} 根手指接触到了物体：{detectedObject.name}");
            }
            else if (maxRotationReached)
            {
                // 即使没有检测到物体，也尝试抓取（可能是小物体或检测问题）
                GrabObject(targetObject.gameObject);
                Debug.Log($"达到最大旋转角度，强制完成抓取：{targetObject.name}");
            }
        }
        else if (fingersInContact > 0)
        {
            Debug.Log($"部分接触：{fingersInContact}/{tipJoints.Count} 根手指接触到物体，继续抓握...");
        }
    }

    /// <summary>
    /// 检测 所有 关节是否与物体发生碰撞
    /// 使用球形射线检测，半径为0.04单位
    /// 只有当所有手指都接触到目标物体时才算抓取成功
    /// 如果旋转角度超过90度还没抓到，也算抓取完成
    /// </summary>
    /// <summary>
    /// 检测 **所有指关节** 是否与物体发生碰撞
    /// 使用球形射线检测，半径为0.04单位
    /// 只有当所有手指都接触到目标物体时才算抓取成功
    /// 如果旋转角度超过90度还没抓到，也算抓取完成
    /// </summary>
    private void CheckTipJointsCollision2()
    {
        if (currentState != HandState.Grabbing || isGrabbingObject || targetObject == null)
            return;


        GameObject detectedObject = null;

        foreach (var joint in tipJoints)
        {
            if (joint == null) continue;



            SphereCollider jointCollider = joint.GetComponent<SphereCollider>();
            if (jointCollider == null) continue;

            // 计算世界空间下的球心位置
            Vector3 worldCenter = jointCollider.transform.TransformPoint(jointCollider.center);
            float worldRadius = jointCollider.radius * Mathf.Max(
                joint.lossyScale.x,
                joint.lossyScale.y,
                joint.lossyScale.z
            );

            Collider[] hitColliders = Physics.OverlapSphere(worldCenter, worldRadius);



            foreach (var collider in hitColliders)
            {
                if (!IsHandComponent(collider.transform) &&
                    collider.gameObject.layer != LayerMask.NameToLayer("Ground") &&
                    (collider.transform == targetObject || collider.transform.IsChildOf(targetObject) || targetObject.IsChildOf(collider.transform)))
                {
                    detectedObject = collider.gameObject;

                    // 判定当前关节属于哪一根手指
                    int fingerIndex = GetFingerIndex(joint);
                    if (fingerIndex > 0)
                        fingerContactFlags[fingerIndex] = true;
                }
            }
        }

        int fingersInContact = 0;
        foreach (var entry in fingerContactFlags)
        {
            if (entry.Value) fingersInContact++;
        }

        bool maxRotationReached = CheckMaxRotationReached();

        Debug.Log($"接触检测：当前已有{fingersInContact}/4 根手指接触到目标物体");
        if (maxRotationReached)
        {
            Debug.Log("手指旋转已达到90度限制");
        }

        if ((fingersInContact >= 5 && detectedObject != null) || maxRotationReached)
        {
            isGrabbingObject = true;
            if (detectedObject != null)
            {
                //GrabObject(detectedObject);
                Debug.Log($"抓取成功！{fingersInContact}/4 根手指接触到了物体：{detectedObject.name}");
            }
            else if (maxRotationReached)
            {
                //GrabObject(targetObject.gameObject);
                Debug.Log($"达到最大旋转角度，强制完成抓取：{targetObject.name}");
            }
        }
        else if (fingersInContact > 0)
        {
            Debug.Log($"部分接触：{fingersInContact}/4 根手指接触到物体，继续抓握...");
        }
    }

    /// <summary>
    /// 判定当前关节属于哪一根手指，根据关节Transform返回手指索引
    /// </summary>
    private int GetFingerIndex(Transform joint)
    {
        if (joint == finger_1 || joint == finger_1_1) return 1;
        if (joint == finger_2_1 || joint == finger_2_2 || joint == finger_2_3) return 2;
        if (joint == finger_3_1 || joint == finger_3_2 || joint == finger_3_3) return 3;
        if (joint == finger_4_1 || joint == finger_4_2 || joint == finger_4_3) return 4;
        if (joint == finger_5_1 || joint == finger_5_2 || joint == finger_5_3) return 5;
        return 0;
    }



    /// <summary>
    /// 检查是否达到最大旋转角度
    /// 检查每个关节的旋转角度是否超过90度
    /// </summary>
    /// <returns>如果达到最大旋转角度返回true</returns>
    private bool CheckMaxRotationReached()
    {
        bool flag = false;
        overRotatedJoints.Clear(); // 每次检测前清空列表
        for (int i = 0; i < transforms.Count && i < initialRotations.Count; i++)
        {
            if (transforms[i] != null)
            {
                float currentRotation = transforms[i].localRotation.eulerAngles.z;
                float initialRotation = initialRotations[i];

                // 计算旋转差值（考虑角度环绕问题）
                float rotationDifference = Mathf.Abs(Mathf.DeltaAngle(initialRotation, currentRotation));

                if (rotationDifference >= maxGrabRotation)
                {
                    Debug.Log($"关节 {transforms[i].name} 旋转角度达到限制: {rotationDifference:F1}度");
                    // 记录该关节
                    overRotatedJoints.Add(transforms[i]);
                    flag = true;
                }
            }
        }

        return flag;
    }

    /// <summary>
    /// 判断给定的Transform是否属于手部组件
    /// 用于在碰撞检测中排除手部自身的碰撞
    /// </summary>
    /// <param name="targetTransform">要检查的Transform</param>
    /// <returns>如果是手部组件返回true，否则返回false</returns>
    private bool IsHandComponent(Transform targetTransform)
    {
        // 检查是否是手部的任何一个关节
        foreach (var joint in transforms)
        {
            if (joint != null && (targetTransform == joint || targetTransform.IsChildOf(joint) || joint.IsChildOf(targetTransform)))
            {
                return true;
            }
        }

        // 检查是否是手部根对象的子对象
        if (targetTransform.IsChildOf(this.transform) || targetTransform == this.transform)
        {
            return true;
        }

        return false;
    }

    /// <summary>
    /// 处理鼠标点击输入
    /// 左键：拉或推抽屉
    /// </summary>
    private void HandleMouseInput()
    {
        Ray ray = Camera.main.ScreenPointToRay(Input.mousePosition);
        RaycastHit hit;

        // 左键点击 - 拉抽屉
        if (Input.GetMouseButtonDown(0))
        {
            if (Physics.Raycast(ray, out hit))
            {
                // 根据当前状态处理左键点击
                switch (currentState)
                {
                    case HandState.Idle:
                        // 空闲状态：尝试选择抓取目标
                        TrySelectGrabTarget(hit);
                        break;

                    case HandState.MovingToHome:
                        // 返回初始位置
                        break;
                }
            }
        }
    }

    /// <summary>
    /// 尝试选择抓取目标
    /// 检查点击的物体是否可以被抓取（排除地板等）
    /// </summary>
    /// <param name="hit">射线检测命中信息</param>
    private void TrySelectGrabTarget(RaycastHit hit) // hit表示鼠标点击后由 Physics.Raycast 命中的碰撞体信息（点击的物体、位置、法线等）。
    {
        // ======================检查点击的物体是否可以抓取========================
        GameObject clickedObject = hit.collider.gameObject; // 获取用户点击的物体

        // 检查是否是地板或其他不可抓取的物体
        if (clickedObject.layer == LayerMask.NameToLayer("Ground")) // 判断物体是否属于 "Ground" 层
        {
            Debug.Log("不能抓取地板！");
            return;
        }

        // 检查物体是否有Collider和Rigidbody（可抓取的基本条件）
        if (clickedObject.GetComponent<Collider>() == null) // 检查物体是否有碰撞体组件
        {
            Debug.Log("目标物体没有碰撞体，无法抓取！");
            return;
        }




        // 设置抓取目标，找到抽屉子物体
        targetObject = clickedObject.transform;

        // 根据抓取目标设置动作规划的具体参数
        if (SelectActionDetails(clickedObject))
        {
            currentState = HandState.MovingToTarget;
            Debug.Log($"选择抓取目标：{targetObject.name}，手部正在移动到目标位置");
        }
    }

    /// <summary>
    /// 计算抓取位置
    /// 根据物体的实际大小计算正确的抓取高度
    /// </summary>
    /// <param name="targetObject">目标物体</param>
    /// <returns>计算出的抓取位置</returns>
    // private Vector3 CalculateGrabPosition(GameObject targetObject)
    // {
    //     // 尝试查找子物体路径，获得抽屉和把手的obj
    //     Debug.Log($"查找路径：{targetObject.name}/SM__BesideTable/Body01/Body02/Drawer/handler");
    //     drawerTransform = targetObject.transform.Find("SM__BesideTable/Body01/Body02/Drawer");
    //     handlerTransform = targetObject.transform.Find("SM__BesideTable/Body01/Body02/Drawer/handler");


    //     if (handlerTransform != null && drawerTransform != null) // 确保找到抽屉和把手
    //     {
    //         Vector3 drawerInitPosition = drawerTransform.localPosition;
    //         if(Mathf.Abs(drawerInitPosition.y - closedPosition) <= 0.03f) operation = "open";
    //         else operation = "close";
    //         Debug.Log($"初始抽屉局部位置: {drawerInitPosition.y}, 将执行操作: {operation}");

    //         Vector3 grabPosition = handlerTransform.transform.position;

    //         // 获取物体的边界框
    //         Bounds objectBounds = GetObjectBounds(handlerTransform.gameObject); // 默认是世界坐标系（Global Space）下的坐标

    //         if (objectBounds.size != Vector3.zero)
    //         {
    //             // 使用物体顶部 + 偏移值作为抓取高度
    //             float objectTopY = objectBounds.min.y; // 手从下面拉开抽屉
    //             grabPosition.y = objectTopY + heightOffset;

    //             Debug.Log($"物体 {handlerTransform.name} 的边界信息:");
    //             Debug.Log($"  中心位置: {objectBounds.center}");
    //             Debug.Log($"  大小: {objectBounds.size}");
    //             Debug.Log($"  顶部Y坐标: {objectTopY}");
    //             Debug.Log($"  计算的抓取高度: {grabPosition.y}");
    //         }
    //         else
    //         {
    //             // 如果无法获取边界框，使用默认偏移
    //             grabPosition.y = targetObject.transform.position.y + heightOffset;
    //             Debug.LogWarning($"无法获取物体 {targetObject.name} 的边界框，使用默认偏移");
    //         }
    //         return grabPosition;
    //     }
    //     else
    //     {
    //         Debug.LogWarning("未找到handler节点，使用默认中心点位置");
    //         return targetObject.transform.position + new Vector3(0, heightOffset, 0);
    //     } 
    // }

    /// <summary>
    /// 获取物体的边界框
    /// 尝试从Collider或Renderer获取边界信息
    /// </summary>
    /// <param name="obj">目标物体</param>
    /// <returns>物体的边界框</returns>
    private Bounds GetObjectBounds(GameObject obj)
    {
        Bounds bounds = new Bounds();

        // 方法1: 尝试从Collider获取边界
        Collider collider = obj.GetComponent<Collider>();
        if (collider != null)
        {
            bounds = collider.bounds; // 【世界空间中的 AABB 包围盒
            Debug.Log($"从Collider获取边界: {bounds}");
            return bounds;
        }

        // 方法2: 尝试从Renderer获取边界
        Renderer renderer = obj.GetComponent<Renderer>();
        if (renderer != null)
        {
            bounds = renderer.bounds;
            Debug.Log($"从Renderer获取边界: {bounds}");
            return bounds;
        }

        // 方法3: 如果物体有子对象，尝试获取所有子对象的边界
        Renderer[] childRenderers = obj.GetComponentsInChildren<Renderer>();
        if (childRenderers.Length > 0)
        {
            bounds = childRenderers[0].bounds;
            for (int i = 1; i < childRenderers.Length; i++)
            {
                bounds.Encapsulate(childRenderers[i].bounds);
            }
            Debug.Log($"从子对象Renderer获取合并边界: {bounds}");
            return bounds;
        }

        // 方法4: 如果都没有，尝试获取所有子对象的Collider边界
        Collider[] childColliders = obj.GetComponentsInChildren<Collider>();
        if (childColliders.Length > 0)
        {
            bounds = childColliders[0].bounds;
            for (int i = 1; i < childColliders.Length; i++)
            {
                bounds.Encapsulate(childColliders[i].bounds);
            }
            Debug.Log($"从子对象Collider获取合并边界: {bounds}");
            return bounds;
        }

        Debug.LogWarning($"物体 {obj.name} 没有Collider或Renderer组件，无法获取边界");
        return new Bounds(obj.transform.position, Vector3.zero);
    }



    /// <summary>
    /// 更新手部状态机
    /// 处理不同状态下的逻辑
    /// </summary>
    private void UpdateHandStateMachine()
    {
        switch (currentState)
        {
            case HandState.MovingToTarget:
                UpdateMovingToTarget();
                break;

            case HandState.Grabbing:
                UpdateGrabbingState();
                break;

            case HandState.PullingOrPush:
                UpdatePullingOrPush();
                break;
            case HandState.MovingToHome:
                UpdateMovingToHome();
                break;


        }
    }

    /// <summary>
    /// 更新移动到抓取目标的状态
    /// 使用四指集合位置进行精确定位
    /// </summary>
    private void UpdateMovingToTarget()
    {
        if (targetObject == null || palmCenter == null)
        {
            if (palmCenter == null)
                Debug.LogWarning("四指集合位置未设置！请在Inspector中指定palmCenter");
            currentState = HandState.Idle;
            return;
        }

        if (splineContainer == null) return;

        traveledDistance += moveSpeed * Time.deltaTime;
        if (traveledDistance >= totalLength)
        {
            traveledDistance = totalLength; // 确保不超过总长度
            Debug.Log("手已经到达样条终点！");
            currentState = HandState.Grabbing;
            return;
        }

        // 查找 traveledDistance 对应的 t 值
        float t;
        SplineUtility.GetPointAtLinearDistance(
            spline,
            0f,
            traveledDistance,
            out t
        );

        // 获取位置和方向
        Vector3 localPos = spline.EvaluatePosition(t);
        Vector3 localTangent = spline.EvaluateTangent(t);

        Vector3 worldPos = splineContainer.transform.TransformPoint(localPos);
        Vector3 worldTangent = splineContainer.transform.TransformDirection(localTangent);

        // 设置手的位姿
        transform.position = worldPos;
        // if (worldTangent != Vector3.zero)
        //     transform.rotation = Quaternion.LookRotation(worldTangent);
    }

    /// <summary>
    /// 更新抓握状态
    /// 持续执行抓握动作直到检测到物体
    /// </summary>
    private void UpdateGrabbingState()
    {
        // 如果还没有抓到物体，继续执行抓握动作
        if (!isGrabbingObject)
        {
            // 持续执行抓握动作，让手指不断弯曲
            RotateAllFingerJointsZ();
        }

        // 注意：一旦球形检测到物体，CheckTipJointsCollision()会自动调用GrabObject()
        // 然后isGrabbingObject会变为true，抓握动作会自动停止
        // 抓取完成后自动回到初始位置
        if (isGrabbingObject)
        {
            if (overRotatedJoints.Count > 0)
            {
                // 新增：如果有超过旋转角度限制的关节，恢复之
                for (int i = 0; i < transforms.Count && i < initialRotations.Count; i++)
                {
                    Transform joint = transforms[i];
                    if (joint != null && overRotatedJoints.Contains(joint))
                    {
                        Vector3 currentRotation = joint.localRotation.eulerAngles; // 必须完整构造并赋值，直接joint.localRotation.eulerAngles.z = initialRotations[i]无效
                        currentRotation.z = initialRotations[i];
                        joint.localRotation = Quaternion.Euler(currentRotation); // 重新生成一个四元数，赋值给 localRotation。
                        Debug.Log($"还原关节 {joint.name} 的Z轴旋转到初始角度 {initialRotations[i]}");
                    }
                }

                // 清空已处理列表
                overRotatedJoints.Clear();


            }
            currentState = HandState.PullingOrPush;
            Debug.Log("抓取完成，正在执行拉出或推入");
        }
    }

    /// <summary>
    /// 绑定手为目标物体的抽屉部分的子物体，并移动
    /// </summary>
    private void UpdatePullingOrPush()
    {
        if (contactTransform == null)
        {
            Debug.LogWarning("目标物体为空，无法执行拉出或推入操作");
            return;
        }

        if (operationTransform == null)
        {
            Debug.LogWarning("未找到 Drawer 子物体，无法执行拉出或推入操作");
            return;
        }

        // 设置为手的子对象
        this.transform.SetParent(operationTransform); // 手作为抽屉的子物体

        // // 获取 Rigidbody 并设置属性
        Rigidbody rb = operationTransform.GetComponent<Rigidbody>();
        if (rb != null)
        {
            rb.useGravity = false;
            rb.isKinematic = true;
        }

        // 读取当前位置
        Vector3 currentPos = operationTransform.localPosition;
        Vector3 targetPos = currentPos;

        if (operation == "open")
        {
            // 打开抽屉，目标位置为初始位置
            targetPos.y = openPosition;
            Debug.Log($"抽屉正在打开，目标位置: {targetPos.y}， 当前抽屉位置: {currentPos.y}");
        }
        else if (operation == "close")
        {
            // 关闭抽屉，目标位置为初始位置
            targetPos.y = closedPosition;
            Debug.Log($"抽屉正在关闭，目标位置: {targetPos.y}， 当前抽屉位置: {currentPos.y}");
        }
        else
        {
            Debug.LogWarning("未知操作类型，请检查操作类型是否正确");
            return;
        }


        // 按方向和速度平滑移动
        // Debug.Log($"计算moveDirection之前targetPos: {targetPos}, currentPos: {currentPos}");
        Vector3 moveDirection = (targetPos - currentPos).normalized;
        //Debug.Log($"计算moveDirection之后moveDirection: {moveDirection}，translate_obj_speed: {translate_obj_speed}，Time.deltaTime: {Time.deltaTime}");
        Vector3 moveAmount = moveDirection * translate_obj_speed * Time.deltaTime;
        //Debug.Log($"moveAmount:{moveAmount}");

        // 移动抽屉
        operationTransform.localPosition += moveAmount;
        Debug.Log(operationTransform.localPosition);


        if (Vector3.Distance(currentPos, targetPos) < 0.01f)
        {
            // 到达目标位置，重置状态
            Debug.Log($"抽屉已{operation}完成");
            this.transform.SetParent(null); // 取消手和抽屉的父子关系
            currentState = HandState.MovingToHome;
            targetPosition = initialPosition; // 为回到初始位置做准备

        }
    }

    /// <summary>
    /// 更新移动到初始位置的状态
    /// 抓取物体后回到初始位置等待放置指令
    /// </summary>
    private void UpdateMovingToHome()
    {
        if (palmCenter == null)
        {
            Debug.LogWarning("四指集合位置未设置！请在Inspector中指定palmCenter");
            return;
        }

        // 使用四指集合位置计算距离和移动
        Vector3 palmPosition = palmCenter.position;
        float distance = Vector3.Distance(palmPosition, targetPosition);

        if (distance > 0.05f)
        {
            // 计算移动方向和距离
            Vector3 moveDirection = (targetPosition - palmPosition).normalized;
            Vector3 moveAmount = moveDirection * moveSpeed * Time.deltaTime;

            // 移动整个手部（通过根Transform）
            transform.position += moveAmount;
        }
        else
        {
            // 到达初始位置，等待用户点击地板进行放置
            Debug.Log("已回到初始位置，抽屉操作闭环结束");
            currentState = HandState.Idle;
        }
    }







    /// <summary>
    /// 抓取物体
    /// 将物体设置为手部的子对象并禁用重力
    /// </summary>
    /// <param name="obj">要抓取的物体</param>
    private void GrabObject(GameObject obj)
    {
        if (obj != null && grabbedObject == null)
        {
            grabbedObject = obj;

            // 设置物体为手部的子对象
            obj.transform.SetParent(this.transform);

            // 禁用物体的重力
            Rigidbody rb = obj.GetComponent<Rigidbody>();
            if (rb != null)
            {
                rb.useGravity = false;
                rb.isKinematic = true; // 设置为运动学模式，防止物理干扰
            }

            Debug.Log($"成功抓取物体：{obj.name}");
        }
    }

    /// <summary>
    /// 释放物体
    /// 取消父子关系并恢复重力
    /// </summary>
    private void ReleaseObject()
    {
        if (grabbedObject != null)
        {
            // 取消父子关系
            grabbedObject.transform.SetParent(null);

            // 恢复物体的重力
            Rigidbody rb = grabbedObject.GetComponent<Rigidbody>();
            if (rb != null)
            {
                rb.useGravity = true;
                rb.isKinematic = false;
            }

            Debug.Log($"释放物体：{grabbedObject.name}");
            grabbedObject = null;
        }

        // 重置状态
        targetObject = null;
        currentState = HandState.Idle;
        isGrabbingObject = false;
    }

    /// <summary>
    /// 在Scene视图中绘制Gizmo
    /// 实时显示球形射线检测范围和四指集合位置
    /// </summary>
    private void OnDrawGizmos()
    {
        // 绘制四指集合位置
        if (palmCenter != null)
        {
            Gizmos.color = Color.blue;
            Gizmos.DrawWireSphere(palmCenter.position, 0.02f);
            Gizmos.DrawLine(palmCenter.position, palmCenter.position + palmCenter.forward * 0.05f);
        }

        // 绘制所有指尖关节的球形检测范围（根据SphereCollider实际半径）
        if (tipJoints != null && tipJoints.Count > 0)
        {
            foreach (var tipJoint in tipJoints)
            {
                if (tipJoint != null)
                {
                    // 检测是否接触到目标物体
                    bool contactingTarget = false;

                    if (currentState == HandState.Grabbing && targetObject != null)
                    {
                        SphereCollider jointCollider = tipJoint.GetComponent<SphereCollider>();
                        if (jointCollider != null)
                        {
                            Vector3 worldCenter = jointCollider.transform.TransformPoint(jointCollider.center);
                            float worldRadius = jointCollider.radius * Mathf.Max(
                                tipJoint.lossyScale.x,
                                tipJoint.lossyScale.y,
                                tipJoint.lossyScale.z
                            );
                            Collider[] hitColliders = Physics.OverlapSphere(worldCenter, worldRadius);
                            foreach (var collider in hitColliders)
                            {
                                if (!IsHandComponent(collider.transform) &&
                                    collider.gameObject.layer != LayerMask.NameToLayer("Ground") &&
                                    (collider.transform == targetObject || collider.transform.IsChildOf(targetObject) || targetObject.IsChildOf(collider.transform)))
                                {
                                    contactingTarget = true;
                                    break;
                                }
                            }
                        }
                    }

                    // 根据接触状态设置颜色
                    if (contactingTarget)
                    {
                        Gizmos.color = Color.red; // 红色表示接触到目标物体
                    }
                    else if (currentState == HandState.Grabbing)
                    {
                        Gizmos.color = Color.yellow; // 黄色表示抓取状态但未接触
                    }
                    else
                    {
                        Gizmos.color = Color.green; // 绿色表示正常状态
                    }

                    // 绘制球形检测范围（使用实际SphereCollider半径）
                    SphereCollider jointColliderDraw = tipJoint.GetComponent<SphereCollider>();
                    if (jointColliderDraw != null)
                    {
                        Vector3 worldCenter = jointColliderDraw.transform.TransformPoint(jointColliderDraw.center);
                        float worldRadius = jointColliderDraw.radius * Mathf.Max(
                            tipJoint.lossyScale.x,
                            tipJoint.lossyScale.y,
                            tipJoint.lossyScale.z
                        );
                        Gizmos.DrawWireSphere(worldCenter, worldRadius);
                    }
                }
            }
        }

        // 绘制目标位置
        if (currentState != HandState.Idle && targetPosition != Vector3.zero)
        {
            Gizmos.color = Color.yellow;
            Gizmos.DrawWireCube(targetPosition, Vector3.one * 0.1f);

            // 绘制四指集合到目标位置的连线
            if (palmCenter != null)
            {
                Gizmos.color = Color.magenta;
                Gizmos.DrawLine(palmCenter.position, targetPosition);
            }
        }


        // 绘制当前状态信息
        if (palmCenter != null)
        {
            // 在手上方显示当前状态
            UnityEditor.Handles.Label(palmCenter.position + Vector3.up * 0.1f, $"状态: {currentState}");
        }
    }
}
