using System.Collections;
using System.Collections.Generic;
using UnityEngine;

public class HarmonicMotion : MonoBehaviour
{
    [Header("运动物体（被控制者）")]
    public Transform targetObject;

    [Header("参考平面（决定方向和坐标系）")]
    public Transform referencePlane;

    [Header("简谐运动参数")]
    public Vector3 localCenter = Vector3.zero;           // 简谐运动的中心点（局部坐标）
    public Vector3 localDirection = Vector3.right;       // 简谐运动方向（局部方向）
    public float amplitude = 1f;                         // 运动幅度
    public float angularFrequency = 2 * Mathf.PI;        // 角频率 ω（rad/s），2π表示1秒来回一圈

    [Range(0, 2 * Mathf.PI)]
    public float initialPhase = 0f;  // 初始相位单位是弧度
    private bool scrollMode = false;
    private float objectRadius = 0.5f;


    private Vector3 worldCenter;
    private Vector3 worldDirection;
    private Rigidbody rb;
    private bool isGrabbed = false;

    private float timeOffset;
    private float elapsedTime = 0f;


    public void setIsGrabbedAndStopMoving(bool flag)
    {
        isGrabbed = flag;
        if (flag && rb != null)
        {
            Vector3 currentV = rb.velocity;
            rb.velocity = new Vector3(0, currentV.y, 0);
        }
    }

    public void SetMotion(string objectName, Vector3 localCenter, Vector3 localDirection, float amplitude, float angularFrequency, float initialPhase)
    {
        this.localCenter = localCenter;
        this.localDirection = localDirection;
        this.amplitude = amplitude;
        this.angularFrequency = angularFrequency;
        this.initialPhase = initialPhase;

        isGrabbed = false;
        elapsedTime = 0f;
        targetObject = GameObject.Find(objectName).transform;


        // 再执行一次start
        if (targetObject == null || referencePlane == null)
        {
            Debug.LogError("HarmonicMotion 初始化失败：targetObject 或 referencePlane 未设置");
            return;
        }

        worldCenter = referencePlane.TransformPoint(localCenter);
        float heightOffset = GetModelHeight(targetObject.gameObject) * 0.5f;
        worldCenter += referencePlane.up * heightOffset;
        worldDirection = referencePlane.TransformDirection(localDirection.normalized);

        rb = targetObject.GetComponent<Rigidbody>();
        if (rb == null)
        {
            Debug.LogError("HarmonicMotion 错误：目标物体未绑定 Rigidbody");
            return;
        }

        rb.freezeRotation = true;
        rb.drag = 0;
        rb.angularDrag = 0;

        // 随机初始相位，避免多个对象同步运动
        timeOffset = initialPhase / angularFrequency;
        scrollMode = targetObject.gameObject.layer == LayerMask.NameToLayer("canScroll");

        // 新增一个将targetobject放在初始位置的函数
        targetObject.position = worldCenter + worldDirection * amplitude * Mathf.Sin(initialPhase);
        
        // 确保物体完全静止，为同步开始做准备
        if (rb != null)
        {
            rb.velocity = Vector3.zero;
            rb.angularVelocity = Vector3.zero;
        }

    }

    public float GetModelHeight(GameObject obj)
    {
        Collider col = obj.GetComponent<Collider>();
        if (col != null)
        {
            // Debug.Log($"得到操作物体高度为{col.bounds.size.y}");
            return col.bounds.size.y;
        }
        else
        {
            Debug.LogWarning("未找到 Renderer，无法估算高度");
            return 0f;
        }
    }


    public Vector3 PredictDisplacementAfter(float deltaTime)
    {
        float t = elapsedTime + deltaTime;
        Debug.Log($"现在的elapsedTime:{elapsedTime},传入的deltaTime:{deltaTime},现在的initialPhase:{initialPhase}");
        float displacement = amplitude * Mathf.Sin(angularFrequency * t + initialPhase);
        return worldCenter + worldDirection * displacement;
    }

    public Vector3 PredictDisplacementAt(float t)
    {
        Debug.Log($"预测在{t}时刻的物体位置");
        float displacement = amplitude * Mathf.Sin(angularFrequency * t + initialPhase);
        return worldCenter + worldDirection * displacement;
    }

    // 得到t时刻的相位
    public float GetPhaseAt(float t)
    {
        return angularFrequency * t + initialPhase;
    }

    void Start()
    {
        if (targetObject == null || referencePlane == null)
        {
            Debug.LogError("HarmonicMotion 初始化失败：targetObject 或 referencePlane 未设置");
            return;
        }

        worldCenter = referencePlane.TransformPoint(localCenter);
        float heightOffset = GetModelHeight(targetObject.gameObject) * 0.5f;
        worldCenter += referencePlane.up * heightOffset;
        worldDirection = referencePlane.TransformDirection(localDirection.normalized);

        rb = targetObject.GetComponent<Rigidbody>();
        if (rb == null)
        {
            Debug.LogError("HarmonicMotion 错误：目标物体未绑定 Rigidbody");
            return;
        }

        rb.freezeRotation = true;
        rb.drag = 0;
        rb.angularDrag = 0;

        // 初始相位与物体的滚动模式
        timeOffset = initialPhase / angularFrequency;
        scrollMode = targetObject.gameObject.layer == LayerMask.NameToLayer("canScroll");
    }

    void FixedUpdate()
    {
        if (rb == null || isGrabbed) return;

        elapsedTime += Time.fixedDeltaTime;  // ⏱️ 局部计时器

        float t = elapsedTime + timeOffset;
        // 简谐运动位移 s(t) = A * sin(ωt)
        float displacement = amplitude * Mathf.Sin(angularFrequency * t);

        // 简谐速度 v(t) = A * ω * cos(ωt)
        float velocity = amplitude * angularFrequency * Mathf.Cos(angularFrequency * t);

        // 更新位置
        Vector3 pos = worldCenter + worldDirection * displacement;

        // 保留垂直方向的速度
        Vector3 currentVelocity = rb.velocity;
        Vector3 newVelocity = worldDirection * velocity;
        newVelocity.y = currentVelocity.y;

        // 更新速度和位置
        rb.MovePosition(pos);
        rb.velocity = newVelocity;

        if (scrollMode){
            // 计算滚动旋转
            // 对于球体，角速度 ω = v/r，其中 v 是线速度，r 是半径
            float angularSpeed = velocity / objectRadius;  // 弧度/秒
            
            // 计算旋转轴：垂直于运动方向和向上方向的轴
            Vector3 rotationAxis = Vector3.Cross(referencePlane.up, worldDirection).normalized;
            
            // 应用旋转
            targetObject.Rotate(rotationAxis, angularSpeed * Mathf.Rad2Deg * Time.fixedDeltaTime, Space.World);
        }
    }

    void OnDrawGizmos()
    {
        if (referencePlane == null || targetObject == null) return;
        if (!enabled) return;

        Vector3 centerWorld = referencePlane.TransformPoint(localCenter);
        float heightOffset = GetModelHeight(targetObject.gameObject) * 0.5f;
        centerWorld += referencePlane.up * heightOffset;
        Vector3 dirWorld = referencePlane.TransformDirection(localDirection.normalized);

        Gizmos.color = Color.yellow;
        // 绘制中心点
        Gizmos.DrawSphere(centerWorld, 0.05f);
        // 绘制运动方向
        Gizmos.DrawLine(centerWorld - dirWorld * amplitude, centerWorld + dirWorld * amplitude);
        // 绘制初始相位及此时运动方向
        Gizmos.color = Color.blue;
        Gizmos.DrawLine(centerWorld, centerWorld + dirWorld * amplitude * Mathf.Sin(initialPhase));
    }
}
