using System.Collections;
using System.Collections.Generic;
using UnityEngine;

public class LinearMotion : MonoBehaviour
{

    [Header("运动物体（被控制者）")]
    public Transform targetObject;

    [Header("参考平面（决定方向和坐标系）")]
    public Transform referencePlane;

    [Header("直线运动参数")]
    public Vector3 localStartPos = Vector3.zero;  // 在参考平面局部坐标下的起点
    public Vector3 localDirection = Vector3.forward; // 运动方向（参考平面坐标系下）
    public float moveSpeed = 1f;                   // 初速度
    public float acceleration = 0f;                // 加速度
    private bool scrollMode = false;

    private float currentSpeed;
    private Rigidbody rb;
    private bool isGrabbed = false;
    private Vector3 worldDirection;
    private Vector3 worldStartPos;
    private float objectRadius;  // 添加物体半径属性

    public void setIsGrabbedAndStopMoving(bool flag)
    {
        isGrabbed = flag;
        if (flag && rb != null)
        {
            Vector3 currentV = rb.velocity;
            rb.velocity = new Vector3(0, currentV.y, 0);
            // 冻结旋转
            rb.freezeRotation = true;
        }
    }

    public float getCurrentSpeedFromRb()
    {
        return rb != null ? rb.velocity.magnitude : currentSpeed;
    }

    public float getCurrentSpeed()
    {
        return currentSpeed;
    }

    public Vector3 getWorldDirection(){
        return worldDirection;
    }
    public void StopMove(){
        Debug.Log("外界调用StopMove");
        Vector3 currentVelocity = rb.velocity;
        Vector3 newVelocity = Vector3.zero;
        newVelocity.y = currentVelocity.y;
        rb.velocity = newVelocity;

        isGrabbed = true;
    }

    public float GetModelHeight(GameObject obj)
    {
        Collider col = obj.GetComponent<Collider>();
        if (col != null)
        {
            Debug.Log($"得到操作物体高度为{col.bounds.size.y}");
            return col.bounds.size.y;
        }
        else
        {
            Debug.LogWarning("未找到 Renderer，无法估算高度");
            return 0f;
        }
    }

    public void SetMotion(string objectName, Vector3 initial_position, Vector3 direction, float initial_speed, float acceleration){
        localStartPos = initial_position;
        localDirection = direction;
        moveSpeed = initial_speed;
        this.acceleration = acceleration;

        isGrabbed = false;
        targetObject = GameObject.Find(objectName).transform;

        // 再执行一次Start()
        // 计算世界起点和方向
        worldStartPos = referencePlane.TransformPoint(localStartPos);
        float heightOffset = GetModelHeight(targetObject.gameObject) * 0.5f;
        worldStartPos += referencePlane.up * heightOffset;

        worldDirection = referencePlane.TransformDirection(localDirection.normalized);

        targetObject.position = worldStartPos;

        rb = targetObject.GetComponent<Rigidbody>();
        if (rb == null)
        {
            Debug.LogError("LinearMotion 错误：目标物体未绑定 Rigidbody");
            return;
        }

        rb.freezeRotation = true;
        rb.drag = 0;
        rb.angularDrag = 0;

        objectRadius = heightOffset;
        currentSpeed = moveSpeed;
        scrollMode = targetObject.gameObject.layer == LayerMask.NameToLayer("canScroll");
        
        // 确保物体完全静止，为同步开始做准备
        if (rb != null)
        {
            rb.velocity = Vector3.zero;
            rb.angularVelocity = Vector3.zero;
        }
    }

    void Start()
    {
        if (targetObject == null || referencePlane == null)
        {
            Debug.LogError("LinearMotion 初始化失败：targetObject 或 referencePlane 未设置");
            return;
        }

        // 计算世界起点和方向
        worldStartPos = referencePlane.TransformPoint(localStartPos);
        float heightOffset = GetModelHeight(targetObject.gameObject) * 0.5f;
        worldStartPos += referencePlane.up * heightOffset;

        worldDirection = referencePlane.TransformDirection(localDirection.normalized);

        targetObject.position = worldStartPos;

        rb = targetObject.GetComponent<Rigidbody>();
        if (rb == null)
        {
            Debug.LogError("LinearMotion 错误：目标物体未绑定 Rigidbody");
            return;
        }

        rb.freezeRotation = !scrollMode;  // 只在非滚动模式下冻结旋转
        rb.drag = 0;
        rb.angularDrag = 0;

        // 计算物体半径（假设物体是球形或圆柱形）
        objectRadius = heightOffset;
        currentSpeed = moveSpeed;
        scrollMode = targetObject.gameObject.layer == LayerMask.NameToLayer("canScroll");

    }

    void FixedUpdate()
    {
        if (rb == null || isGrabbed) return;

        if (!isGrabbed)
        {
            currentSpeed += acceleration * Time.fixedDeltaTime;

            Vector3 currentVelocity = rb.velocity;
            Vector3 newVelocity = worldDirection * currentSpeed;

            // 保留竖直速度成分（受重力影响）
            newVelocity.y = currentVelocity.y;

            rb.velocity = newVelocity;
            
            if (scrollMode)
            {
                // 计算滚动旋转
                // 对于球体，角速度 ω = v/r，其中 v 是线速度，r 是半径
                float angularSpeed = currentSpeed / objectRadius;  // 弧度/秒
                
                // 计算旋转轴：垂直于运动方向和向上方向的轴
                Vector3 rotationAxis = Vector3.Cross(referencePlane.up, worldDirection).normalized;
                
                // 应用旋转
                targetObject.Rotate(rotationAxis, angularSpeed * Mathf.Rad2Deg * Time.fixedDeltaTime, Space.World);
            }
        }
        else
        {
            // 被抓取时：只保留竖直速度，不然会没有重力
            Vector3 currentVelocity = rb.velocity;
            Vector3 newVelocity = Vector3.zero;
            newVelocity.y = currentVelocity.y;
            rb.velocity = newVelocity;
        }
    }

    void OnDrawGizmos()
    {
        if (referencePlane == null) return;
        if (!enabled) return;

        // 起点和方向箭头
        Vector3 worldStart = referencePlane.TransformPoint(localStartPos);
        if (targetObject != null){
            float heightOffset = targetObject.localScale.y * 0.5f;
            worldStartPos += referencePlane.up * heightOffset;

            Vector3 worldDir = referencePlane.TransformDirection(localDirection.normalized) * 0.5f;

            Gizmos.color = Color.cyan;
            Gizmos.DrawSphere(worldStart, 0.05f);
            Gizmos.DrawLine(worldStart, worldStart + worldDir);
            Gizmos.DrawRay(worldStart, worldDir);
        }
    }
}
