// using System.Collections;
// using System.Collections.Generic;
// using UnityEngine;
// using System.IO;
// using System;
// using System.Linq;

// /// <summary>
// /// 轨迹分析器
// /// 用于分析手部动作轨迹的平滑性、直线性等质量指标
// /// </summary>
// public class TrajectoryAnalyzer : MonoBehaviour
// {
//     [Header("分析配置")]
//     public bool enableRealtimeAnalysis = true; // 是否启用实时分析
//     public float samplingInterval = 0.05f; // 采样间隔
//     public int smoothnessWindowSize = 5; // 平滑性分析窗口大小
    
//     [Header("质量阈值")]
//     [Range(0f, 1f)]
//     public float smoothnessThreshold = 0.8f; // 平滑性阈值
//     [Range(0f, 1f)] 
//     public float linearityThreshold = 0.7f; // 直线性阈值
//     public float maxVelocityChange = 2f; // 最大速度变化阈值
//     public float maxAccelerationChange = 5f; // 最大加速度变化阈值
    
//     // 轨迹数据记录
//     private List<TrajectoryPoint> currentTrajectory = new List<TrajectoryPoint>();
//     private List<EpisodeTrajectoryAnalysis> allAnalyses = new List<EpisodeTrajectoryAnalysis>();
    
//     // 实时分析
//     private float lastSampleTime = 0f;
//     private Vector3 lastPosition;
//     private Vector3 lastVelocity;
//     private bool isRecording = false;
    
//     // 分析结果
//     private TrajectoryQuality currentQuality;
    
//     void Update()
//     {
//         if (enableRealtimeAnalysis && isRecording)
//         {
//             UpdateRealtimeAnalysis();
//         }
//     }
    
//     /// <summary>
//     /// 开始记录轨迹
//     /// </summary>
//     public void StartTrajectoryRecording(int episodeId, int attempt, Transform target)
//     {
//         currentTrajectory.Clear();
//         isRecording = true;
//         lastSampleTime = Time.time;
//         lastPosition = target.position;
//         lastVelocity = Vector3.zero;
        
//         // 添加起始点
//         AddTrajectoryPoint(target.position, target.rotation);
        
//         Debug.Log($"开始记录Episode {episodeId} 尝试 {attempt} 的轨迹");
//     }
    
//     /// <summary>
//     /// 停止记录轨迹并分析
//     /// </summary>
//     public TrajectoryQuality StopTrajectoryRecording(int episodeId, int attempt, bool success)
//     {
//         isRecording = false;
        
//         if (currentTrajectory.Count > 1)
//         {
//             currentQuality = AnalyzeTrajectory(currentTrajectory);
            
//             // 保存分析结果
//             EpisodeTrajectoryAnalysis analysis = new EpisodeTrajectoryAnalysis
//             {
//                 episode_id = episodeId,
//                 attempt = attempt,
//                 success = success,
//                 trajectory_points = new List<TrajectoryPoint>(currentTrajectory),
//                 quality = currentQuality,
//                 analysis_time = DateTime.Now.ToString("yyyy-MM-dd HH:mm:ss")
//             };
            
//             allAnalyses.Add(analysis);
//             SaveTrajectoryAnalysis(analysis);
            
//             Debug.Log($"Episode {episodeId} 轨迹分析完成: 平滑性={currentQuality.smoothness:F2}, 直线性={currentQuality.linearity:F2}");
//         }
        
//         return currentQuality;
//     }
    
//     /// <summary>
//     /// 更新实时轨迹分析
//     /// </summary>
//     private void UpdateRealtimeAnalysis()
//     {
//         if (Time.time - lastSampleTime >= samplingInterval)
//         {
//             // 获取当前手部位置（这里需要根据你的手部控制器调整）
//             Transform handTransform = GetHandTransform();
//             if (handTransform != null)
//             {
//                 AddTrajectoryPoint(handTransform.position, handTransform.rotation);
//             }
            
//             lastSampleTime = Time.time;
//         }
//     }
    
//     /// <summary>
//     /// 添加轨迹点
//     /// </summary>
//     private void AddTrajectoryPoint(Vector3 position, Quaternion rotation)
//     {
//         float currentTime = Time.time;
        
//         // 计算速度和加速度
//         Vector3 velocity = Vector3.zero;
//         Vector3 acceleration = Vector3.zero;
        
//         if (currentTrajectory.Count > 0)
//         {
//             float deltaTime = currentTime - currentTrajectory[currentTrajectory.Count - 1].timestamp;
//             if (deltaTime > 0)
//             {
//                 velocity = (position - lastPosition) / deltaTime;
                
//                 if (currentTrajectory.Count > 1)
//                 {
//                     acceleration = (velocity - lastVelocity) / deltaTime;
//                 }
//             }
//         }
        
//         TrajectoryPoint point = new TrajectoryPoint
//         {
//             position = position,
//             rotation = rotation.eulerAngles,
//             velocity = velocity,
//             acceleration = acceleration,
//             timestamp = currentTime
//         };
        
//         currentTrajectory.Add(point);
        
//         // 更新记录
//         lastPosition = position;
//         lastVelocity = velocity;
//     }
    
//     /// <summary>
//     /// 分析轨迹质量
//     /// </summary>
//     private TrajectoryQuality AnalyzeTrajectory(List<TrajectoryPoint> trajectory)
//     {
//         if (trajectory.Count < 2)
//         {
//             return new TrajectoryQuality();
//         }
        
//         TrajectoryQuality quality = new TrajectoryQuality();
        
//         // 计算各项指标
//         quality.smoothness = CalculateSmoothness(trajectory);
//         quality.linearity = CalculateLinearity(trajectory);
//         quality.velocity_consistency = CalculateVelocityConsistency(trajectory);
//         quality.acceleration_smoothness = CalculateAccelerationSmoothness(trajectory);
//         quality.path_efficiency = CalculatePathEfficiency(trajectory);
//         quality.total_distance = CalculateTotalDistance(trajectory);
//         quality.average_velocity = CalculateAverageVelocity(trajectory);
//         quality.max_velocity = CalculateMaxVelocity(trajectory);
//         quality.duration = trajectory[trajectory.Count - 1].timestamp - trajectory[0].timestamp;
        
//         // 计算综合评分
//         quality.overall_score = CalculateOverallScore(quality);
        
//         return quality;
//     }
    
//     /// <summary>
//     /// 计算轨迹平滑性
//     /// </summary>
//     private float CalculateSmoothness(List<TrajectoryPoint> trajectory)
//     {
//         if (trajectory.Count < smoothnessWindowSize)
//             return 0f;
        
//         float totalVariance = 0f;
//         int windowCount = 0;
        
//         for (int i = smoothnessWindowSize; i < trajectory.Count; i++)
//         {
//             List<float> velocityMagnitudes = new List<float>();
            
//             for (int j = i - smoothnessWindowSize; j < i; j++)
//             {
//                 velocityMagnitudes.Add(trajectory[j].velocity.magnitude);
//             }
            
//             float mean = velocityMagnitudes.Average();
//             float variance = velocityMagnitudes.Sum(v => Mathf.Pow(v - mean, 2)) / velocityMagnitudes.Count;
            
//             totalVariance += variance;
//             windowCount++;
//         }
        
//         float averageVariance = totalVariance / windowCount;
        
//         // 将方差转换为0-1的平滑性评分（方差越小，平滑性越高）
//         return Mathf.Exp(-averageVariance);
//     }
    
//     /// <summary>
//     /// 计算轨迹直线性
//     /// </summary>
//     private float CalculateLinearity(List<TrajectoryPoint> trajectory)
//     {
//         if (trajectory.Count < 2)
//             return 0f;
        
//         Vector3 startPos = trajectory[0].position;
//         Vector3 endPos = trajectory[trajectory.Count - 1].position;
        
//         float straightLineDistance = Vector3.Distance(startPos, endPos);
//         float actualPathDistance = CalculateTotalDistance(trajectory);
        
//         if (actualPathDistance == 0f)
//             return 0f;
        
//         // 直线性 = 直线距离 / 实际路径长度
//         return Mathf.Clamp01(straightLineDistance / actualPathDistance);
//     }
    
//     /// <summary>
//     /// 计算速度一致性
//     /// </summary>
//     private float CalculateVelocityConsistency(List<TrajectoryPoint> trajectory)
//     {
//         if (trajectory.Count < 2)
//             return 0f;
        
//         List<float> velocityMagnitudes = trajectory.Skip(1).Select(p => p.velocity.magnitude).ToList();
        
//         float mean = velocityMagnitudes.Average();
//         float variance = velocityMagnitudes.Sum(v => Mathf.Pow(v - mean, 2)) / velocityMagnitudes.Count;
//         float stdDev = Mathf.Sqrt(variance);
        
//         // 一致性评分：标准差越小，一致性越高
//         return Mathf.Exp(-stdDev);
//     }
    
//     /// <summary>
//     /// 计算加速度平滑性
//     /// </summary>
//     private float CalculateAccelerationSmoothness(List<TrajectoryPoint> trajectory)
//     {
//         if (trajectory.Count < 3)
//             return 0f;
        
//         List<float> accelerationChanges = new List<float>();
        
//         for (int i = 2; i < trajectory.Count; i++)
//         {
//             float change = Vector3.Distance(trajectory[i].acceleration, trajectory[i-1].acceleration);
//             accelerationChanges.Add(change);
//         }
        
//         float averageChange = accelerationChanges.Average();
        
//         // 加速度变化越小，平滑性越高
//         return Mathf.Exp(-averageChange / maxAccelerationChange);
//     }
    
//     /// <summary>
//     /// 计算路径效率
//     /// </summary>
//     private float CalculatePathEfficiency(List<TrajectoryPoint> trajectory)
//     {
//         return CalculateLinearity(trajectory); // 路径效率等同于直线性
//     }
    
//     /// <summary>
//     /// 计算总距离
//     /// </summary>
//     private float CalculateTotalDistance(List<TrajectoryPoint> trajectory)
//     {
//         float totalDistance = 0f;
        
//         for (int i = 1; i < trajectory.Count; i++)
//         {
//             totalDistance += Vector3.Distance(trajectory[i-1].position, trajectory[i].position);
//         }
        
//         return totalDistance;
//     }
    
//     /// <summary>
//     /// 计算平均速度
//     /// </summary>
//     private float CalculateAverageVelocity(List<TrajectoryPoint> trajectory)
//     {
//         if (trajectory.Count < 2)
//             return 0f;
        
//         float totalDistance = CalculateTotalDistance(trajectory);
//         float duration = trajectory[trajectory.Count - 1].timestamp - trajectory[0].timestamp;
        
//         return duration > 0 ? totalDistance / duration : 0f;
//     }
    
//     /// <summary>
//     /// 计算最大速度
//     /// </summary>
//     private float CalculateMaxVelocity(List<TrajectoryPoint> trajectory)
//     {
//         return trajectory.Skip(1).Max(p => p.velocity.magnitude);
//     }
    
//     /// <summary>
//     /// 计算综合评分
//     /// </summary>
//     private float CalculateOverallScore(TrajectoryQuality quality)
//     {
//         // 权重分配
//         float smoothnessWeight = 0.3f;
//         float linearityWeight = 0.25f;
//         float velocityWeight = 0.2f;
//         float accelerationWeight = 0.15f;
//         float efficiencyWeight = 0.1f;
        
//         return quality.smoothness * smoothnessWeight +
//                quality.linearity * linearityWeight +
//                quality.velocity_consistency * velocityWeight +
//                quality.acceleration_smoothness * accelerationWeight +
//                quality.path_efficiency * efficiencyWeight;
//     }
    
//     /// <summary>
//     /// 获取手部Transform（需要根据你的场景调整）
//     /// </summary>
//     private Transform GetHandTransform()
//     {
//         // 这里需要根据你的手部控制器结构来获取手部位置
//         VLATestManager testManager = FindObjectOfType<VLATestManager>();
//         if (testManager != null && testManager.handController != null)
//         {
//             return testManager.handController.transform;
//         }
//         return null;
//     }
    
//     /// <summary>
//     /// 保存轨迹分析结果
//     /// </summary>
//     private void SaveTrajectoryAnalysis(EpisodeTrajectoryAnalysis analysis)
//     {
//         string resultsDir = Path.Combine(Application.dataPath, "..", "vla_test_results", "trajectories");
//         Directory.CreateDirectory(resultsDir);
        
//         string fileName = $"trajectory_episode_{analysis.episode_id:00000}_attempt_{analysis.attempt}.json";
//         string filePath = Path.Combine(resultsDir, fileName);
        
//         string json = JsonUtility.ToJson(analysis, true);
//         File.WriteAllText(filePath, json);
//     }
    
//     /// <summary>
//     /// 保存所有轨迹分析结果汇总
//     /// </summary>
//     public void SaveAllTrajectoryAnalyses()
//     {
//         TrajectoryAnalysisSummary summary = new TrajectoryAnalysisSummary
//         {
//             total_episodes = allAnalyses.Count,
//             analyses = allAnalyses,
//             average_smoothness = allAnalyses.Average(a => a.quality.smoothness),
//             average_linearity = allAnalyses.Average(a => a.quality.linearity),
//             average_overall_score = allAnalyses.Average(a => a.quality.overall_score),
//             success_rate = allAnalyses.Count(a => a.success) / (float)allAnalyses.Count,
//             analysis_completed_time = DateTime.Now.ToString("yyyy-MM-dd HH:mm:ss")
//         };
        
//         string resultsDir = Path.Combine(Application.dataPath, "..", "vla_test_results");
//         string summaryPath = Path.Combine(resultsDir, "trajectory_analysis_summary.json");
        
//         string json = JsonUtility.ToJson(summary, true);
//         File.WriteAllText(summaryPath, json);
        
//         Debug.Log($"轨迹分析汇总已保存: {summaryPath}");
//     }
    
//     /// <summary>
//     /// 获取当前轨迹质量
//     /// </summary>
//     public TrajectoryQuality GetCurrentQuality()
//     {
//         return currentQuality;
//     }
    
//     /// <summary>
//     /// 清除所有分析数据
//     /// </summary>
//     [ContextMenu("清除分析数据")]
//     public void ClearAnalysisData()
//     {
//         allAnalyses.Clear();
//         currentTrajectory.Clear();
//         Debug.Log("轨迹分析数据已清除");
//     }
// }

// // 轨迹分析相关数据结构
// [System.Serializable]
// public class TrajectoryPoint
// {
//     public Vector3 position;
//     public Vector3 rotation;
//     public Vector3 velocity;
//     public Vector3 acceleration;
//     public float timestamp;
// }

// [System.Serializable]
// public class TrajectoryQuality
// {
//     [Range(0f, 1f)]
//     public float smoothness = 0f; // 平滑性
//     [Range(0f, 1f)]
//     public float linearity = 0f; // 直线性
//     [Range(0f, 1f)]
//     public float velocity_consistency = 0f; // 速度一致性
//     [Range(0f, 1f)]
//     public float acceleration_smoothness = 0f; // 加速度平滑性
//     [Range(0f, 1f)]
//     public float path_efficiency = 0f; // 路径效率
//     [Range(0f, 1f)]
//     public float overall_score = 0f; // 综合评分
    
//     // 统计信息
//     public float total_distance = 0f;
//     public float average_velocity = 0f;
//     public float max_velocity = 0f;
//     public float duration = 0f;
// }

// [System.Serializable]
// public class EpisodeTrajectoryAnalysis
// {
//     public int episode_id;
//     public int attempt;
//     public bool success;
//     public List<TrajectoryPoint> trajectory_points = new List<TrajectoryPoint>();
//     public TrajectoryQuality quality;
//     public string analysis_time;
// }

// [System.Serializable]
// public class TrajectoryAnalysisSummary
// {
//     public int total_episodes;
//     public List<EpisodeTrajectoryAnalysis> analyses = new List<EpisodeTrajectoryAnalysis>();
//     public float average_smoothness;
//     public float average_linearity;
//     public float average_overall_score;
//     public float success_rate;
//     public string analysis_completed_time;
// } 