using System.IO;
using UnityEngine;

public class TransparentObjectCapture : MonoBehaviour
{
    [Header("捕获设置")]
    public Camera captureCamera;       // 用于拍摄的相机
    public GameObject targetObject;    // 目标物体
    public string saveFileName = "capture.png"; // 保存文件名
    public int width = 1920;           // 输出宽度
    public int height = 1080;          // 输出高度

    private RenderTexture rt;
    private Texture2D tex;
    private int originalLayer;
    private int captureLayer;

    void Start()
    {
        if (captureCamera == null)
        {
            Debug.LogError("请指定捕获相机！");
            return;
        }

        if (targetObject == null)
        {
            Debug.LogError("请指定目标物体！");
            return;
        }

        // 创建专用层（如果不存在，用默认的 User Layer 8）
        captureLayer = LayerMask.NameToLayer("CaptureTarget");
        if (captureLayer == -1)
        {
            Debug.LogWarning("未找到 'CaptureTarget' Layer，请手动在 Unity 中添加！");
            captureLayer = 8; // 默认用第 8 层
        }

        // 创建 RenderTexture 和 Texture2D
        rt = new RenderTexture(width, height, 24, RenderTextureFormat.ARGB32);
        rt.Create();

        tex = new Texture2D(width, height, TextureFormat.ARGB32, false);

        // 配置相机
        captureCamera.clearFlags = CameraClearFlags.SolidColor;
        captureCamera.backgroundColor = new Color(0, 0, 0, 0); // 透明背景
        captureCamera.targetTexture = rt;
    }

    void Update()
    {
        if (Input.GetKeyDown(KeyCode.Space))
        {
            CaptureTransparent();
        }
    }

    public void CaptureTransparent()
    {
        if (targetObject == null || captureCamera == null) return;

        // 保存原始 Layer
        originalLayer = targetObject.layer;

        // 把目标物体放到 CaptureTarget 层
        SetLayerRecursively(targetObject, captureLayer);

        // 相机只渲染该层
        captureCamera.cullingMask = 1 << captureLayer;

        // 捕获
        RenderTexture.active = rt;
        captureCamera.Render();

        tex.ReadPixels(new Rect(0, 0, rt.width, rt.height), 0, 0);
        tex.Apply();

        // 保存 PNG
        byte[] bytes = tex.EncodeToPNG();
        File.WriteAllBytes(saveFileName, bytes);
        Debug.Log("透明背景图像已保存到：" + Path.GetFullPath(saveFileName));

        // 恢复设置
        RenderTexture.active = null;
        SetLayerRecursively(targetObject, originalLayer);
    }

    void OnDestroy()
    {
        if (rt != null) rt.Release();
        if (tex != null) Destroy(tex);
    }

    // 递归设置物体及其子物体 Layer
    void SetLayerRecursively(GameObject obj, int newLayer)
    {
        obj.layer = newLayer;
        foreach (Transform child in obj.transform)
        {
            SetLayerRecursively(child.gameObject, newLayer);
        }
    }
}