using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using System.IO;
using System;
using System.Linq;
using EpisodeControl;
using WebSocketSharp;

/// <summary>
/// VLA模型测试管理器 - WebSocket版
/// 通过WebSocket与Python端通信，支持混合配置文件，每个episode重复多次评估
/// </summary>
public class VLATestManager : MonoBehaviour
{
    [Header("测试配置")]
    public float actionApplyFPS = 20f; // 动作应用频率
    public float objectMotionFPS = 60f; // 物体运动频率
    public float positionThreshold = 0.3f; // 成功判定的位置阈值
    public int maxWaitFrames = 600; // 最大等待帧数（原10s * 60FPS）
    public int windowSize = 16; // 状态和动作数据的窗口大小
    public int actionDim = 18; // 动作维度，可以是18可以是3
    public bool use_camera_coordinate = true;  // 是否使用相机坐标系，true则使用相机坐标系，否则使用世界坐标系
    private Vector3 camPos = new Vector3(42.64241f, 8.79f, -0.94f);
    private Quaternion camRot = Quaternion.Euler(35.7056f, 359.7939f, 359.9761f); // 注意是度
    [Tooltip("摄像头画面宽度")]
    public int width;
    [Tooltip("摄像头画面高度")]
    public int height;


    [Header("WebSocket配置")]
    public string serverUrl = "ws://127.0.0.1:8765";

    [Header("手部结构引用")]
    public Transform finger_1_1;
    public Transform finger_1_2;
    public Transform finger_1_3;
    public Transform finger_2_1;
    public Transform finger_2_2;
    public Transform finger_2_3;
    public Transform finger_3_1;
    public Transform finger_3_2;
    public Transform finger_3_3;
    public Transform finger_4_1;
    public Transform finger_4_2;
    public Transform finger_4_3;
    public Transform finger_5_1;
    public Transform finger_5_2;
    public Transform finger_5_3;
    public Transform finger_nail_1;
    public Transform finger_nail_2;
    public Transform finger_nail_3;
    public Transform finger_nail_4;
    public Transform finger_nail_5;

    [Header("组件引用")]
    public Transform palmCenter;
    public Transform handsRoot;
    public Transform armRoot;
    public Camera testCamera;

    [Header("环境清理")]
    public Transform[] movableObjects; // 需要重置位置的可移动物体
    private Vector3[] initialObjectPositions; // 可移动物体的初始位置
    private Quaternion[] initialObjectRotations; // 可移动物体的初始旋转

    // 测试状态
    public enum TestState
    {
        Initializing,
        ApplyingAction,
        Evaluating,
        EpisodeComplete
    }

    [Header("当前状态")]
    public TestState currentState = TestState.Initializing;
    public int currentEpisode = 0;
    public int currentRepeat = 0;
    public int currentStep = 0;

    // Episode管理相关
    private BaseConfig currentConfig;
    private string currentTaskType;
    private float minDistanceToTarget = float.MaxValue; // 记录最小距离

    // WebSocket相关
    private WebSocket ws;
    private bool wsConnected = false;

    // 主线程消息队列与start_episode消息队列
    private Queue<WebSocketMessage> messageQueue = new Queue<WebSocketMessage>();
    private readonly object queueLock = new object();
    private Queue<EpisodeRequest> pendingEpisodes = new Queue<EpisodeRequest>();

    // 动作数据缓冲队列（帧对齐模式专用）
    // private Queue<float[][]> actionBufferQueue = new Queue<float[][]>();
    private Queue<float[]> actionBufferQueue = new Queue<float[]>();
    private readonly object actionBufferLock = new object();

    // 内部管理
    private string resultsDir;
    private Texture2D captureTexture;

    // 手部控制相关
    private List<Transform> allFingerJoints = new List<Transform>(); // 存15个可控关节点
    private List<Vector3> initialJointRotations = new List<Vector3>(); // 存15个关节点初始旋转
    private List<Transform> detectjoints = new List<Transform>(); // 存储所有用于碰撞检测的关节

    // 抓握检测相关
    private float minJointToSurfaceDistance = 1000f; // 关节到物体表面的最小距离

    // 帧对齐运动控制相关
    private Vector3[] presetTrajectory; // 预设轨迹点位数组（1200个点）
    private int currentFrameIndex = 0; // 当前帧索引（60FPS）
    private int currentHandMotionIndex = 0; // 当前手部运动索引（20FPS）
    private int totalTrajectoryFrames = 1200; // 总轨迹帧数（20秒*60FPS）
    private int actionFrameRatio = 3; // 动作帧比例（60FPS:20FPS = 3:1）
    private int waitStartFrameIndex = 0; // 评估开始帧索引
    private string trajectoryDir; // 轨迹文件目录
    private float hightOffset; // 高度偏移，调用cm、lm、hm的GetModelHeight得到物体高度的一半，用于把确保物体下表面接触桌面

    // 运动脚本相关（保留用于参考平面）
    private Vector3 initialHandPosition;
    private Vector3 initialCameraPosition;
    public Transform scriptLoader; // 挂载运动脚本的物体
    public HarmonicMotion hm;
    public CircularMotion cm;
    public LinearMotion lm;
    public Transform referencePlane; // 参考平面

    // 评估数据
    private EpisodeEvaluation currentEvaluation;
    private float waitStartTime;
    private Transform targetObject;

    // 动作应用相关
    private float[,] currentActionWindow; // [windowSize, actionDim] 动作窗口
    private int currentActionWindowSize = 0; // 当前实际接收到的动作数量
    private int currentActionIndex = 0;
    private bool actionDataReceived = false;
    // private bool inferenceComplete = false;

    // 时间测试
    private float time_per_window = 0f;

    // 累计帧数
    private int total_frame_count = 0;
    private int total_frame_episode = 0;

    // 手和手掌中心之间的偏移
    private Vector3 offset_hand2palm;

    // 提前成功标志 和 成功时候的帧索引
    private bool success_early = false;
    private int success_index = 0;


    void Start()
    {
        InitializeVLATest();
        offset_hand2palm = transform.position - palmCenter.position; // transform.position和palmCenter.position之间有一个偏移X；X + transform.position = palmCenter.position
    }

    void Update()
    {
        // 只处理WebSocket消息队列
        ProcessMessageQueue();
    }

    void FixedUpdate()
    {
        // 帧对齐模式的核心逻辑：60FPS同步执行
        UpdateFrameAlignedLogic();
    }

    void OnDestroy()
    {
        if (ws != null && ws.IsAlive)
            ws.Close();
    }

    /// <summary>
    /// 设置物理一致性参数，确保每次运行结果相同
    /// 帧对齐模式：使用60FPS固定时间步长
    /// </summary>
    private void SetupPhysicsConsistency()
    {
        // 设置固定的物理时间步长（60FPS = 0.01667秒/帧）
        Time.fixedDeltaTime = 1f / objectMotionFPS; // 0.01667f for 60 FPS

        // 设置物理迭代次数确保稳定性
        Physics.defaultSolverIterations = 6;
        Physics.defaultSolverVelocityIterations = 1;

        // 设置重力确保一致性
        Physics.gravity = new Vector3(0, -9.81f, 0);

        Debug.Log($"帧对齐模式物理一致性设置完成：FixedDeltaTime={Time.fixedDeltaTime:F4}s ({objectMotionFPS}FPS), 物理迭代次数={Physics.defaultSolverIterations}");
    }

    /// <summary>
    /// 初始化VLA测试系统
    /// </summary>
    private void InitializeVLATest()
    {
        resultsDir = Path.Combine(Application.dataPath, "..", "vla_test_results");
        Directory.CreateDirectory(resultsDir);

        // 设置固定的物理时间步长确保一致性
        SetupPhysicsConsistency();

        // 初始化手部关节列表
        InitializeHandJoints();

        // 记录初始状态
        RecordInitialStates();

        // 记录可移动物体的初始状态
        RecordMovableObjectsInitialStates();

        // 初始化相机捕获
        if (testCamera != null)
        {
            captureTexture = new Texture2D(width, height, TextureFormat.RGB24, false);
        }

        // 在帧对齐模式下，不需要计算actionInterval

        // 初始化WebSocket
        InitializeWebSocket();

        // 初始化动作窗口
        // currentActionWindow = new float[windowSize, 18];
        currentActionWindow = new float[windowSize, actionDim];
        Debug.Log($"初始化了 {windowSize} 帧的动作窗口，维度为 {actionDim}，帧对齐模式：{objectMotionFPS}FPS物体运动，{actionApplyFPS}FPS动作应用");

        // 初始化运动控制脚本
        hm = scriptLoader.GetComponent<HarmonicMotion>();
        cm = scriptLoader.GetComponent<CircularMotion>();
        lm = scriptLoader.GetComponent<LinearMotion>();

        // 初始化轨迹文件目录
        trajectoryDir = Path.Combine(Application.dataPath, "..", "PreProcessPoints");
        Directory.CreateDirectory(trajectoryDir);
        
        Debug.Log("VLA测试系统初始化完成，等待Python端发送episode请求");
    }

    /// <summary>
    /// 初始化手部关节列表
    /// </summary>
    private void InitializeHandJoints()
    {
        allFingerJoints.Clear();

        if (finger_1_1 != null) allFingerJoints.Add(finger_1_1);
        if (finger_1_2 != null) allFingerJoints.Add(finger_1_2);
        if (finger_1_3 != null) allFingerJoints.Add(finger_1_3);
        if (finger_2_1 != null) allFingerJoints.Add(finger_2_1);
        if (finger_2_2 != null) allFingerJoints.Add(finger_2_2);
        if (finger_2_3 != null) allFingerJoints.Add(finger_2_3);
        if (finger_3_1 != null) allFingerJoints.Add(finger_3_1);
        if (finger_3_2 != null) allFingerJoints.Add(finger_3_2);
        if (finger_3_3 != null) allFingerJoints.Add(finger_3_3);
        if (finger_4_1 != null) allFingerJoints.Add(finger_4_1);
        if (finger_4_2 != null) allFingerJoints.Add(finger_4_2);
        if (finger_4_3 != null) allFingerJoints.Add(finger_4_3);
        if (finger_5_1 != null) allFingerJoints.Add(finger_5_1);
        if (finger_5_2 != null) allFingerJoints.Add(finger_5_2);
        if (finger_5_3 != null) allFingerJoints.Add(finger_5_3);

        Debug.Log($"初始化了 {allFingerJoints.Count} 个手指关节");

        // 初始化检测关节列表，包含所有手指关节和指尖关节，但移除手指根关节
        detectjoints.Clear();
        detectjoints.AddRange(allFingerJoints);

        // 添加指尖关节
        if (finger_nail_1 != null) detectjoints.Add(finger_nail_1);
        if (finger_nail_2 != null) detectjoints.Add(finger_nail_2);
        if (finger_nail_3 != null) detectjoints.Add(finger_nail_3);
        if (finger_nail_4 != null) detectjoints.Add(finger_nail_4);
        if (finger_nail_5 != null) detectjoints.Add(finger_nail_5);

        // 移除手指根关节（第一关节）
        detectjoints.Remove(finger_1_1);
        detectjoints.Remove(finger_2_1);
        detectjoints.Remove(finger_3_1);
        detectjoints.Remove(finger_4_1);
        detectjoints.Remove(finger_5_1);

        Debug.Log($"初始化了 {detectjoints.Count} 个检测关节");
    }

    /// <summary>
    /// 加载预设轨迹文件
    /// 从{task_type}_{episode}.txt文件中读取1200个预设点位
    /// </summary>
    private bool LoadPresetTrajectory(string taskType, int episodeId)
    {
        string fileName = $"{taskType}_{episodeId}.txt";
        string filePath = Path.Combine(trajectoryDir, fileName);
        filePath = Path.GetFullPath(filePath);  // 规范化路径，去掉 Assets\.. 等

        if (!File.Exists(filePath))
        {
            Debug.LogError($"轨迹文件不存在: {filePath}");
            return false;
        }

        try
        {
            string[] lines = File.ReadAllLines(filePath);
            
            if (lines.Length != totalTrajectoryFrames+2)
            {
                Debug.LogWarning($"轨迹文件行数不匹配，期望{totalTrajectoryFrames+2}行，实际{lines.Length}行");
            }

            presetTrajectory = new Vector3[lines.Length];

            // 前 totalTrajectoryFrames 行是点
            for (int i = 0; i < totalTrajectoryFrames; i++)
            {
                string[] coords = lines[i].Trim().Split(' ');
                if (coords.Length >= 3)
                {
                    if (float.TryParse(coords[0], out float x) &&
                        float.TryParse(coords[1], out float y) &&
                        float.TryParse(coords[2], out float z))
                    {
                        // 存储参考平面坐标系下的点位
                        presetTrajectory[i] = new Vector3(x, y, z);
                    }
                    else
                    {
                        Debug.LogError($"轨迹文件第{i+1}行格式错误: {lines[i]}");
                        return false;
                    }
                }
                else
                {
                    Debug.LogError($"轨迹文件第{i+1}行坐标数量不足: {lines[i]}");
                    return false;
                }
            }

            // 最后两行是这个轨迹的observeTime和targetobject的name
            string lastLine = lines[lines.Length - 2].Trim();
            float observeTime;
            if (!float.TryParse(lastLine, out observeTime))
            {
                Debug.LogError($"最后一行不是有效的 observeTime: {lastLine}");
                return false;
            }
            maxWaitFrames = Mathf.RoundToInt(observeTime * objectMotionFPS);

            string targetObjectName = lines[lines.Length - 1].Trim();
            targetObject = GameObject.Find(targetObjectName).transform;
            Debug.Log($"找到目标物体: {targetObjectName}");
            hightOffset = hm.GetModelHeight(targetObject.gameObject) / 2;

            Debug.Log($"成功加载轨迹文件: {fileName}, 共{presetTrajectory.Length}个点位");
            return true;
        }
        catch (Exception ex)
        {
            Debug.LogError($"读取轨迹文件失败: {ex.Message}");
            return false;
        }
    }

    /// <summary>
    /// 处理单帧动作数据
    /// </summary>
    private void ProcessSingleActionFrame(float[] actionFrame)
    {
        // 将单帧动作数据转换为currentActionWindow格式
        currentActionWindow = new float[1, actionDim];
        for (int j = 0; j < Mathf.Min(actionFrame.Length, actionDim); j++)
        {
            currentActionWindow[0, j] = actionFrame[j];
        }
        currentActionWindowSize = 1;
        currentActionIndex = 0;

        Debug.Log($"处理单帧动作数据：维度{actionDim}");
    }

    /// <summary>
    /// 处理动作窗口数据，把接受到的actions读取到currentActionWindow
    /// </summary>
    private void ProcessActionWindow(float[][] actions)
    {
        int action_horizon = actions.Length;
        currentActionWindowSize = Mathf.Min(action_horizon, windowSize); // 记录实际动作数量

        for (int i = 0; i < currentActionWindowSize; i++)
        {
            if (actions[i].Length >= actionDim)
            {
                for (int j = 0; j < actionDim; j++)
                {
                    currentActionWindow[i, j] = actions[i][j];
                }
            }
        }

        currentActionIndex = 0;
        Debug.Log($"处理动作窗口：接收到 {action_horizon} 帧，实际处理 {currentActionWindowSize} 帧");
    }


    /// <summary>
    /// 记录初始状态
    /// </summary>
    private void RecordInitialStates()
    {
        initialHandPosition = transform.position;

        if (testCamera != null)
        {
            initialCameraPosition = testCamera.transform.position;
        }

        initialJointRotations.Clear();
        foreach (var joint in allFingerJoints)
        {
            if (joint != null)
            {
                initialJointRotations.Add(joint.localRotation.eulerAngles);
            }
            else
            {
                initialJointRotations.Add(Vector3.zero);
            }
        }

        Debug.Log($"记录了初始状态：位置={initialHandPosition}, 关节数={initialJointRotations.Count}");
    }

    /// <summary>
    /// 记录可移动物体的初始状态
    /// </summary>
    private void RecordMovableObjectsInitialStates()
    {
        if (movableObjects != null && movableObjects.Length > 0)
        {
            initialObjectPositions = new Vector3[movableObjects.Length];
            initialObjectRotations = new Quaternion[movableObjects.Length];

            for (int i = 0; i < movableObjects.Length; i++)
            {
                if (movableObjects[i] != null)
                {
                    initialObjectPositions[i] = movableObjects[i].position;
                    initialObjectRotations[i] = movableObjects[i].rotation;
                }
            }

            Debug.Log($"记录了 {movableObjects.Length} 个可移动物体的初始状态");
        }
    }

    /// <summary>
    /// 清理桌面，重置所有可移动物体到初始位置（排除当前目标物体）
    /// </summary>
    private void ClearDesktop()
    {
        if (movableObjects != null && initialObjectPositions != null && initialObjectRotations != null)
        {
            int resetCount = 0;
            for (int i = 0; i < movableObjects.Length && i < initialObjectPositions.Length; i++)
            {
                if (movableObjects[i] != null)
                {
                    // 如果当前物体是目标物体，跳过重置（让运动脚本来设置位置）
                    if (targetObject != null && movableObjects[i] == targetObject)
                    {
                        Debug.Log($"跳过重置当前目标物体: {targetObject.name}");
                        continue;
                    }

                    movableObjects[i].position = initialObjectPositions[i];
                    movableObjects[i].rotation = initialObjectRotations[i];

                    // 重置物体的物理状态
                    Rigidbody rb = movableObjects[i].GetComponent<Rigidbody>();
                    if (rb != null)
                    {
                        rb.velocity = Vector3.zero;
                        rb.angularVelocity = Vector3.zero;
                    }
                    resetCount++;
                }
            }

            Debug.Log($"已清理桌面，重置了 {resetCount} 个物体（排除目标物体）");
        }
    }

    /// <summary>
    /// 初始化WebSocket连接
    /// </summary>
    private void InitializeWebSocket()
    {
        ws = new WebSocket(serverUrl);

        ws.OnOpen += (sender, e) =>
        {
            wsConnected = true;
            Debug.Log("WebSocket连接成功");
        };

        ws.OnMessage += (sender, e) =>
        {
            HandleWebSocketMessage(e.Data);
        };

        ws.OnError += (sender, e) =>
        {
            Debug.LogError($"WebSocket错误: {e.Message}");
            wsConnected = false;
        };

        ws.OnClose += (sender, e) =>
        {
            wsConnected = false;
            Debug.Log("WebSocket连接关闭");
        };

        ws.Connect();
    }

    /// <summary>
    /// 处理WebSocket消息（将消息放入队列，在主线程中处理）
    /// </summary>
    private void HandleWebSocketMessage(string message)
    {
        try
        {
            var messageData = JsonUtility.FromJson<WebSocketMessage>(message);
            lock (queueLock)
            {
                messageQueue.Enqueue(messageData);
            }
        }
        catch (Exception ex)
        {
            Debug.LogError($"解析WebSocket消息失败: {ex.Message}");
        }
    }

    /// <summary>
    /// 在主线程中处理消息队列
    /// </summary>
    private void ProcessMessageQueue()
    {
        lock (queueLock)
        {
            while (messageQueue.Count > 0)
            {
                var messageData = messageQueue.Dequeue();
                ProcessWebSocketMessage(messageData);
            }
        }
    }

    /// <summary>
    /// 在主线程中处理WebSocket消息
    /// </summary>
    private void ProcessWebSocketMessage(WebSocketMessage messageData)
    {
        try
        {
            Debug.Log($"收到消息类型: {messageData.type}");
            Debug.Log($"收到消息数据: {messageData.data}");
            switch (messageData.type)
            {
                case "start_episode":
                    // 开始新episode
                    var episodeRequest = JsonUtility.FromJson<EpisodeRequest>(messageData.data);
                    Debug.Log($"收到start_episode请求: {episodeRequest.episode_id}, 任务类型: {episodeRequest.task_type}, 重试次数: {episodeRequest.retry_time}, 总帧数: {episodeRequest.total_frame_episode}, 窗口大小: {episodeRequest.windowSize}");
                    Debug.Log($"当前状态: {currentState}");

                    // 更严格的状态检查
                    if (currentState == TestState.Evaluating ||
                        currentState == TestState.ApplyingAction)
                    {
                        // 还在忙碌状态，放入队列
                        pendingEpisodes.Enqueue(episodeRequest);
                        Debug.Log($"当前忙碌状态，已将 episode {episodeRequest.episode_id} 加入待启动队列，队列长度: {pendingEpisodes.Count}");
                    }
                    else if (currentState == TestState.Initializing || currentState == TestState.EpisodeComplete)
                    {
                        // 空闲状态，直接启动
                        Debug.Log($"当前空闲，直接启动 episode {episodeRequest.episode_id}");
                        StartNewEpisode(episodeRequest.episode_id, episodeRequest.task_type, episodeRequest.retry_time, episodeRequest.total_frame_episode, episodeRequest.windowSize);
                    }
                    else
                    {
                        Debug.LogError($"未知状态 {currentState}，将 episode {episodeRequest.episode_id} 加入队列");
                        // 其他状态，也放入队列（安全起见）
                        pendingEpisodes.Enqueue(episodeRequest);
                        Debug.Log($"未知状态 {currentState}，将 episode {episodeRequest.episode_id} 加入队列");
                    }
                    break;

                case "action_data":
                    // 接收到动作数据 - N个18维或3维参数
                    var actionData = JsonUtility.FromJson<ActionWindowData>(messageData.data);
                    if (actionData != null && actionData.actions != null)
                    {
                        Debug.Log($"接收到动作数据，共 {actionData.actions.Length} 帧，逐动作放入缓冲队列");

                        lock (actionBufferLock)
                        {
                            for (int i = 0; i < actionData.actions.Length; i++)
                            {
                                float[] singleAction = actionData.actions[i].values;
                                actionBufferQueue.Enqueue(singleAction);
                            }
                        }

                        Debug.Log($"动作数据已放入缓冲队列，当前队列长度: {actionBufferQueue.Count}");
                    }
                    break;

                default:
                    Debug.Log($"收到未知消息类型: {messageData.type}");
                    break;
            }
        }
        catch (Exception ex)
        {
            Debug.LogError($"处理WebSocket消息失败: {ex.Message}");
        }
    }

    /// <summary>
    /// 发送WebSocket消息
    /// </summary>
    private void SendWebSocketMessage(string type, string data)
    {
        if (ws != null && wsConnected)
        {
            var message = new WebSocketMessage
            {
                type = type,
                data = data
            };
            string json = JsonUtility.ToJson(message);
            ws.Send(json);
        }
    }


    /// <summary>
    /// 开始新的Episode - 帧对齐版本
    /// </summary>
    private void StartNewEpisode(int episodeId, string taskType, int retryTime, int total_frame_episode, int windowSize)
    {
        currentEpisode = episodeId;
        currentTaskType = taskType;
        currentRepeat = retryTime;
        currentStep = 0;
        this.total_frame_episode = total_frame_episode;
        this.windowSize = windowSize;

        currentActionWindow = new float[windowSize, actionDim];

        if (!LoadPresetTrajectory(taskType, episodeId))
        {
            return;
        }

        SetupReferencePlane(taskType);

        currentEvaluation = new EpisodeEvaluation
        {
            episode_id = currentEpisode,
            repeat = currentRepeat,
            task_type = currentTaskType,
            start_time = Time.time
        };

        ClearDesktop();

        ResetHandState();

        currentFrameIndex = 0;
        waitStartFrameIndex = 0;
        actionDataReceived = false;
        minDistanceToTarget = float.MaxValue;
        currentActionWindowSize = 0;
        currentActionIndex = 0;
        total_frame_count = 0;
        success_early = false;
        success_index = 0;
        minJointToSurfaceDistance = 1000f;

        lock (actionBufferLock)
        {
            actionBufferQueue.Clear();
        }

        if (presetTrajectory.Length > 0)
        {
            SetObjectToTrajectoryPosition(0);
        }

        Time.fixedDeltaTime = 1f / objectMotionFPS; // 1/60 = 0.01667s


        SendImageAndState();

        currentState = TestState.ApplyingAction;
        
        Time.timeScale = 1f;
    }

    


    /// <summary>
    /// 帧对齐模式的核心逻辑 - 60FPS同步执行
    /// 统一处理所有状态：ApplyingAction、WaitingForAction、Evaluating
    /// </summary>
    private void UpdateFrameAlignedLogic()
    {
        // 初始状态是Initializing，startepisode之后变成ApplyingAction才开始做物体和人手的步进
        if (currentState != TestState.ApplyingAction && currentState != TestState.Evaluating){
            return;
        }
        currentFrameIndex = currentFrameIndex + 1; // 准备拿下一帧动作

        // 检查是否完成所有轨迹帧
        if (currentFrameIndex >= totalTrajectoryFrames)
        {
            if (currentState != TestState.Evaluating)
            {
                Debug.Log($"Episode {currentEpisode} 轨迹数据播放完成，开始评估阶段");
                StartEvaluating();
            }
            return;
        }

        // 更新物体位置
        if (currentFrameIndex < totalTrajectoryFrames)
        {
            SetObjectToTrajectoryPosition(currentFrameIndex);
        }
        
        // 如果现在是评估状态，那就调用评估状态的更新，不需要走下面的人手步进策略
        if (currentState== TestState.Evaluating){
            UpdateEvaluating();
            return;
        }

        // 检查是否需要应用手部动作（每3帧应用一次）
        bool shouldApplyAction = currentFrameIndex % actionFrameRatio == 0;

        if (shouldApplyAction)
        {
            // 检查是否有动作数据可以应用
            lock (actionBufferLock)
            {
                if (actionBufferQueue.Count > 0)
                {
                    // 有动作数据，从队列里那一个出来
                    float[] currentAction = actionBufferQueue.Dequeue(); 

                    // 应用指定帧的动作数据
                    //Apply18DimAction(0);
                    Apply18DimAction(currentAction);
                    currentHandMotionIndex = currentHandMotionIndex + 1;

                    Debug.Log($"全局帧为：{currentFrameIndex}: 手部动作帧为：{currentHandMotionIndex}，人手位置更新");
                }
                else
                {
                    // 该更新的时候没有动作数据，那就把现在移动索引往前提一个
                    currentFrameIndex = currentFrameIndex - 1;
                }
            }
        }
        else
        {
            Debug.Log($"全局帧为：{currentFrameIndex}: 手部动作帧为：{currentHandMotionIndex}，人手位置更新（无动作）");
        }


        // 检查成功条件
        if (CheckSuccess() && !success_early)
        {
            success_early = true;
            success_index = currentFrameIndex;
            Debug.Log($"提前成功，在第{success_index}帧检测到成功");
        }

        // 检查通信时机：在WaitingForAction状态下，每windowSize*actionFrameRatio帧发送一次图像
        int communicationFrameInterval = windowSize * actionFrameRatio;
        // 需要发state和image了
        if (currentFrameIndex % communicationFrameInterval == 0){
            if (actionBufferQueue.Count > 0) Debug.LogError($"此时的模拟步是{currentFrameIndex}，该发送图像，但动作数据队列不为空，但应该为空");
            else{
                SendImageAndState();
                int stepIndex = currentFrameIndex / communicationFrameInterval;
                Debug.Log($"帧{currentFrameIndex}: 发送图像和状态 (步骤{stepIndex})");
            }
        }
        
    }



    /// <summary>
    /// 重置手部状态到初始位置
    /// </summary>
    private void ResetHandState()
    {
        transform.position = initialHandPosition;

        for (int i = 0; i < allFingerJoints.Count && i < initialJointRotations.Count; i++)
        {
            if (allFingerJoints[i] != null)
            {
                allFingerJoints[i].localRotation = Quaternion.Euler(initialJointRotations[i]);
            }
        }

        if (testCamera != null)
        {
            testCamera.transform.position = initialCameraPosition;
        }

        Debug.Log("手部状态已重置到初始位置");
    }


    /// <summary>
    /// 设置参考平面（用于坐标转换）
    /// </summary>
    private void SetupReferencePlane(string taskType)
    {
        // 根据任务类型获取对应的参考平面
        switch (taskType.ToLower())
        {
            case "linear":
                if (lm != null && lm.referencePlane != null)
                {
                    referencePlane = lm.referencePlane;
                }
                break;
            case "circular":
                if (cm != null && cm.referencePlane != null)
                {
                    referencePlane = cm.referencePlane;
                }
                break;
            case "harmonic":
                if (hm != null && hm.referencePlane != null)
                {
                    referencePlane = hm.referencePlane;
                }
                break;
            default:
                Debug.LogError($"未知的任务类型: {taskType}");
                break;
        }

        if (referencePlane == null)
        {
            Debug.LogError($"无法获取{taskType}任务的参考平面");
        }
        else
        {
            Debug.Log($"使用参考平面: {referencePlane.name}");
        }
    }

    /// <summary>
    /// 将物体设置到轨迹指定位置
    /// </summary>
    private void SetObjectToTrajectoryPosition(int frameIndex)
    {
        if (targetObject == null || presetTrajectory == null || frameIndex >= presetTrajectory.Length)
        {
            return;
        }

        if (referencePlane == null)
        {
            Debug.LogError("参考平面未设置，无法进行坐标转换");
            return;
        }

        // 将参考平面坐标系下的点转换为世界坐标
        Vector3 localPosition = presetTrajectory[frameIndex];
        Vector3 worldPosition = referencePlane.TransformPoint(localPosition);
        
        // 直接设置物体位置（不使用物理模拟）
        targetObject.position = worldPosition;
        
        // 如果物体有Rigidbody，确保它不受物理影响
        Rigidbody rb = targetObject.GetComponent<Rigidbody>();
        if (rb != null)
        {
            rb.isKinematic = true; // 设为运动学模式，不受物理影响
        }
    }



    /// <summary>
    /// 应用18维动作参数
    /// </summary>
    private void Apply18DimAction(int frameIndex)
    {
        // 相机坐标系 转 世界坐标系
        if (use_camera_coordinate)
        {
            Vector3 palmpos_cam = new Vector3(currentActionWindow[frameIndex, 0], currentActionWindow[frameIndex, 1], currentActionWindow[frameIndex, 2]);
            Vector3 palmpos_world = camRot * palmpos_cam + camPos;
            transform.position = palmpos_world + offset_hand2palm;
        }

        else
        {
            Vector3 palmPosition = new Vector3(
                        currentActionWindow[frameIndex, 0],
                        currentActionWindow[frameIndex, 1],
                        currentActionWindow[frameIndex, 2]
                    );
            transform.position = palmPosition + offset_hand2palm; // p + x = hand
        }

        if (actionDim == 18)
        {
            // 后15维：15个关节旋转 (弧度制)
            for (int i = 0; i < 15 && i < allFingerJoints.Count; i++)
            {
                if (allFingerJoints[i] != null)
                {
                    float jointRotation = currentActionWindow[frameIndex, i + 3];
                    ApplyJointRotation(allFingerJoints[i], jointRotation, i);
                }
            }
        }
    }

    /// <summary>
    /// 应用18维动作参数
    /// </summary>
    private void Apply18DimAction(float [] action)
    {
        // 相机坐标系 转 世界坐标系
        if (use_camera_coordinate)
        {
            Vector3 palmpos_cam = new Vector3(action[0], action[1], action[2]);
            Vector3 palmpos_world = camRot * palmpos_cam + camPos;
            transform.position = palmpos_world + offset_hand2palm + new Vector3(0, hightOffset, 0);
        }

        else
        {
            Vector3 palmPosition = new Vector3(
                        action[0],
                        action[1],
                        action[2]
                    );
            transform.position = palmPosition + offset_hand2palm + new Vector3(0, hightOffset, 0); // p + x = hand
        }

        if (actionDim == 18)
        {
            // 后15维：15个关节旋转 (弧度制)
            for (int i = 0; i < 15 && i < allFingerJoints.Count; i++)
            {
                if (allFingerJoints[i] != null)
                {
                    float jointRotation = action[i + 3];
                    ApplyJointRotation(allFingerJoints[i], jointRotation, i);
                }
            }
        }
    }

    /// <summary>
    /// 应用单个关节旋转 (finger_1_2和finger_1_3用y轴，其余用z轴)
    /// </summary>
    private void ApplyJointRotation(Transform joint, float rotationRad, int jointIndex)
    {
        // Vector3 currentRotation = joint.localRotation.eulerAngles;
        Vector3 currentRotation = new Vector3(joint.eulerAngles.x, joint.eulerAngles.y, joint.eulerAngles.z);
        float rotationDeg = rotationRad * Mathf.Rad2Deg;

        // finger_1_2(索引1)和finger_1_3(索引2)使用y轴旋转
        if (jointIndex == 1 || jointIndex == 2) // finger_1_2, finger_1_3
        {
            currentRotation.y = rotationDeg;
        }
        else // 其他关节使用z轴旋转
        {
            currentRotation.z = rotationDeg;
        }

        // joint.localRotation = Quaternion.Euler(currentRotation);
        joint.transform.eulerAngles = currentRotation;
    }


    /// <summary>
    /// 开始评估阶段 - 帧对齐版本
    /// </summary>
    private void StartEvaluating()
    {
        currentState = TestState.Evaluating;
        Debug.Log("=== 开始评估阶段，检测是否成功抓取... ===");
        Time.timeScale = 1f; // 确保模拟器正常运行
        waitStartFrameIndex = currentFrameIndex; // 记录评估开始的帧索引
        Debug.Log($"评估开始帧: {waitStartFrameIndex}, 当前最小距离: {minDistanceToTarget:F4}, 成功阈值: {positionThreshold:F4}");
        Debug.Log($"最大等待帧数: {maxWaitFrames} (约{maxWaitFrames/objectMotionFPS:F1}秒)");
    }

    /// <summary>
    /// 更新评估状态 - 帧对齐版本
    /// </summary>
    private void UpdateEvaluating()
    {
        if (success_early)
        {
            float waitTime = (success_index - waitStartFrameIndex) / objectMotionFPS;
            CompleteEpisodeEvaluation(true, waitTime);
            return;
        }

        // 确保模拟器正常运行
        if (Time.timeScale != 1f)
        {
            Time.timeScale = 1f;
        }

        // 计算等待帧数和时间
        int elapsedWaitFrames = currentFrameIndex - waitStartFrameIndex;
        float elapsedWaitTime = elapsedWaitFrames / objectMotionFPS;

        // 每帧检查成功条件并更新最小距离
        bool success = CheckSuccess();

        // 检查成功条件
        if (success)
        {
            Debug.Log($"Episode {currentEpisode} (重试: {currentRepeat}) 成功！等待帧数: {elapsedWaitFrames}, 等待时间: {elapsedWaitTime:F2}s, 最小距离: {minDistanceToTarget:F4}");
            CompleteEpisodeEvaluation(true, elapsedWaitTime);
        }
        // 检查超时条件（使用帧数）
        else if (elapsedWaitFrames >= maxWaitFrames)
        {
            Debug.Log($"Episode {currentEpisode} (重试: {currentRepeat}) 超时失败，等待帧数: {elapsedWaitFrames}, 等待时间: {elapsedWaitTime:F2}s, 最小距离: {minDistanceToTarget:F4}");
            CompleteEpisodeEvaluation(false, elapsedWaitTime);
        }
        // 检查轨迹数据是否用完
        else if (currentFrameIndex >= totalTrajectoryFrames)
        {
            Debug.Log($"Episode {currentEpisode} (重试: {currentRepeat}) 轨迹数据用完，等待帧数: {elapsedWaitFrames}, 等待时间: {elapsedWaitTime:F2}s, 最小距离: {minDistanceToTarget:F4}");
            CompleteEpisodeEvaluation(false, elapsedWaitTime);
        }
    }

    /// <summary>
    /// 检查成功条件并更新最小距离
    /// </summary>
    private bool CheckSuccess()
    {
        if (palmCenter != null && targetObject != null)
        {
            Vector3 palmPos = palmCenter.position;
            Vector3 targetPos = targetObject.position;

            // 计算XZ平面的距离用于成功判定（忽略Y轴高度差）
            float distanceXZ = Vector3.Distance(
                new Vector3(palmPos.x, 0, palmPos.z),
                new Vector3(targetPos.x, 0, targetPos.z)
            );

            // 更新最小距离记录（同样使用XZ距离保持一致）
            if (distanceXZ < minDistanceToTarget)
            {
                minDistanceToTarget = distanceXZ;
                
                // 当距离达到最小值时，进行抓握检测
                CheckGraspDetection();
            }
            Debug.Log($"当前XZ距离: {distanceXZ:F4}, 最小距离: {minDistanceToTarget:F4}");

            // 成功条件：XZ平面距离小于阈值
            return distanceXZ < positionThreshold;
        }

        return false;
    }



    /// <summary>
    /// 完成Episode评估
    /// </summary>
    private void CompleteEpisodeEvaluation(bool success, float waitTime)
    {
        currentEvaluation.success = success;
        currentEvaluation.wait_time = waitTime;
        // currentEvaluation.final_palm_position = new SerializableVector3(palmCenter != null ? palmCenter.position : Vector3.zero);
        currentEvaluation.min_distance_to_target = minDistanceToTarget;
        currentEvaluation.end_time = Time.time;

        Debug.Log($"Episode {currentEpisode} (重试: {currentRepeat}) 完成，结果: {(success ? "成功" : "失败")}, 最小距离: {minDistanceToTarget:F4}");
        // Debug.Log($"评估数据: 等待时间={waitTime:F2}s, 掌心位置=({currentEvaluation.final_palm_position.x:F3}, {currentEvaluation.final_palm_position.y:F3}, {currentEvaluation.final_palm_position.z:F3}).");

        CalculateAndSendCurrentEvaluationResult();
        currentState = TestState.EpisodeComplete;
        Debug.Log($"结束{currentEpisode}的评估结果保存，现在pendingEpisodes: {pendingEpisodes}");

        // 检查是否有等待的 episode
        if (pendingEpisodes.Count > 0)
        {
            var nextEpisode = pendingEpisodes.Dequeue();
            Debug.Log($"评估完成，自动启动待启动 episode: {nextEpisode.episode_id}");
            StartNewEpisode(nextEpisode.episode_id, nextEpisode.task_type, nextEpisode.retry_time, nextEpisode.total_frame_episode, nextEpisode.windowSize);
        }
    }



    /// <summary>
    /// 获取当前状态 (掌心坐标 + 15个关节旋转)
    /// </summary>
    private float[] GetCurrentState()
    {
        // float[] state = new float[18];
        float[] state = new float[actionDim];

        // 前3维：掌心坐标
        if (palmCenter != null)
        {
            state[0] = palmCenter.position.x;
            state[1] = palmCenter.position.y;
            state[2] = palmCenter.position.z;
        }

        // 世界坐标转 相机坐标
        if (use_camera_coordinate)
        {
            Vector3 palmpos_world = new Vector3(state[0], state[1], state[2]);
            Vector3 palmpos_cam = Quaternion.Inverse(camRot) * (palmpos_world - camPos);
            state[0] = palmpos_cam.x;
            state[1] = palmpos_cam.y;
            state[2] = palmpos_cam.z;
        }

        if (actionDim == 18)
        {
            // 后15维：15个关节旋转
            for (int j = 0; j < 15 && j < allFingerJoints.Count; j++)
            {
                if (allFingerJoints[j] != null)
                {
                    Vector3 rotation = allFingerJoints[j].localRotation.eulerAngles;

                    // finger_1_2(索引1)和finger_1_3(索引2)使用y轴，其余用z轴
                    if (j == 1 || j == 2)
                        state[j + 3] = rotation.y * Mathf.Deg2Rad;
                    else
                        state[j + 3] = rotation.z * Mathf.Deg2Rad;
                }
            }
        }
        return state;
    }

    /// <summary>
    /// 发送图像和状态数据
    /// </summary>
    private void SendImageAndState()
    {
        if (testCamera != null && captureTexture != null)
        {
            // 发送图像数据
            RenderTexture renderTexture = new RenderTexture(width, height, 24);
            Debug.Log($"width and height renderTexture: {renderTexture.width} and {renderTexture.height}");
            testCamera.targetTexture = renderTexture;
            testCamera.Render();

            RenderTexture.active = renderTexture;
            captureTexture.ReadPixels(new Rect(0, 0, renderTexture.width, renderTexture.height), 0, 0);
            captureTexture.Apply();
            RenderTexture.active = null;

            byte[] imageBytes = captureTexture.EncodeToJPG(85);
            string imageBase64 = System.Convert.ToBase64String(imageBytes);

            // 获取当前状态
            float[] currentState = GetCurrentState();
            var imageAndStateData = new ImageAndStateMessage
            {
                episode_id = currentEpisode,
                repeat = currentRepeat,
                step = currentStep,
                image_data = imageBase64,
                state_data = currentState,
                task_type = currentTaskType
            };

            SendWebSocketMessage("image_and_state", JsonUtility.ToJson(imageAndStateData));
            Debug.Log($"发送图像和状态数据: Episode {currentEpisode}, Repeat {currentRepeat}, Step {currentStep}");
            currentStep++;

            testCamera.targetTexture = null;
            DestroyImmediate(renderTexture);
        }
    }

    private void SendMetrics(bool success, float waitTime, float score, float min_distance_to_target, int successIndex)
    {
        var metricsData = new Metrics
        {
            episode_id = currentEpisode,
            repeat = currentRepeat,
            success = success,
            waitTime = waitTime,
            score = score,
            min_distance_to_target = min_distance_to_target,
            successIndex = successIndex,
            minJointToSurfaceDistance = minJointToSurfaceDistance
        };

        SendWebSocketMessage("metrics", JsonUtility.ToJson(metricsData));
        Debug.Log($"发送评估结果到python端: waitTime={waitTime}, success={success}, min_distance_to_target={min_distance_to_target}, minJointToSurfaceDistance={minJointToSurfaceDistance:F4}");
    }

    /// <summary>
    /// 判定当前关节属于哪一根手指，根据关节Transform返回手指索引
    /// </summary>
    private int GetFingerIndex(Transform joint)
    {
        if (joint == finger_1_1 || joint == finger_1_2 || joint == finger_1_3 || joint == finger_nail_1) return 1;
        if (joint == finger_2_1 || joint == finger_2_2 || joint == finger_2_3 || joint == finger_nail_2) return 2;
        if (joint == finger_3_1 || joint == finger_3_2 || joint == finger_3_3 || joint == finger_nail_3) return 3;
        if (joint == finger_4_1 || joint == finger_4_2 || joint == finger_4_3 || joint == finger_nail_4) return 4;
        if (joint == finger_5_1 || joint == finger_5_2 || joint == finger_5_3 || joint == finger_nail_5) return 5;
        return 0;
    }

    /// <summary>
    /// 判断给定的Transform是否属于手部组件
    /// 用于在碰撞检测中排除手部自身的碰撞
    /// </summary>
    private bool IsHandComponent(Transform targetTransform)
    {
        // 检查是否是手部的任何一个关节
        foreach (var joint in allFingerJoints)
        {
            if (joint != null && (targetTransform == joint || targetTransform.IsChildOf(joint) || joint.IsChildOf(targetTransform)))
            {
                return true;
            }
        }

        // 检查是否是手部根对象的子对象
        if (targetTransform.IsChildOf(this.transform) || targetTransform == this.transform)
        {
            return true;
        }

        return false;
    }

    /// <summary>
    /// 检测关节到物体表面的最小距离
    /// 将15个关节按手指分成5组，计算每组内的最小距离，然后取5组间的平均值
    /// </summary>
    private void CheckGraspDetection()
    {
        if (targetObject == null || detectjoints == null || actionDim != 18) return;

        // 为每个手指初始化最小距离数组 (索引0-4对应手指1-5)
        float[] fingerMinDistances = new float[5];
        for (int i = 0; i < 5; i++) fingerMinDistances[i] = 1000f;

        foreach (var joint in detectjoints)
        {
            if (joint == null) continue;

            // 获取当前关节属于哪个手指
            int fingerIndex = GetFingerIndex(joint);
            if (fingerIndex <= 0 || fingerIndex > 5) continue; // 跳过无效的手指索引

            // 计算关节位置到目标物体表面的距离
            Vector3 jointPosition = joint.position;
            
            // 获取目标物体的所有碰撞器
            Collider[] targetColliders = targetObject.GetComponentsInChildren<Collider>();
            
            float jointMinDistance = 1000f; // 当前关节的最小距离
            
            foreach (var targetCollider in targetColliders)
            {
                if (targetCollider != null && !IsHandComponent(targetCollider.transform) &&
                    targetCollider.gameObject.layer != LayerMask.NameToLayer("Ground"))
                {
                    // 计算关节位置到物体表面的最近距离
                    Vector3 closestPointOnSurface = targetCollider.ClosestPoint(jointPosition);
                    float distanceToSurface = Vector3.Distance(jointPosition, closestPointOnSurface);
                    
                    // 更新当前关节的最小距离
                    if (distanceToSurface < jointMinDistance)
                    {
                        jointMinDistance = distanceToSurface;
                    }
                }
            }
            
            // 更新对应手指的最小距离
            if (jointMinDistance < fingerMinDistances[fingerIndex - 1]) // fingerIndex从1开始，数组从0开始
            {
                fingerMinDistances[fingerIndex - 1] = jointMinDistance;
            }
        }

        // 计算5个手指最小距离的平均值
        float totalDistance = 0f;
        int validFingers = 0;
        
        for (int i = 0; i < 5; i++)
        {
            if (fingerMinDistances[i] < 1000f) // 只计算有效的距离
            {
                totalDistance += fingerMinDistances[i];
                validFingers++;
            }
        }
        
        if (validFingers > 0)
        {
            float averageMinDistance = totalDistance / validFingers;
            
            // 更新全局最小距离
            if (averageMinDistance < minJointToSurfaceDistance)
            {
                minJointToSurfaceDistance = averageMinDistance;
                Debug.Log($"更新关节到物体表面最小距离（5指平均）: {minJointToSurfaceDistance:F4}，有效手指数: {validFingers}");
                Debug.Log($"各手指最小距离: 手指1:{fingerMinDistances[0]:F4}, 手指2:{fingerMinDistances[1]:F4}, 手指3:{fingerMinDistances[2]:F4}, 手指4:{fingerMinDistances[3]:F4}, 手指5:{fingerMinDistances[4]:F4}");
            }
        }
    }

    /// <summary>
    /// 为帧对齐模式获取目标物体（简化版本，不依赖配置文件）
    /// </summary>
    private Transform GetTargetObjectForFrameAlignment(string taskType)
    {
        // 在帧对齐模式下，我们根据任务类型从运动脚本中获取目标物体
        // 这样可以保持与原有系统的兼容性，同时不依赖具体的配置参数
        switch (taskType.ToLower())
        {
            case "linear":
                if (lm != null && lm.targetObject != null)
                {
                    return lm.targetObject;
                }
                break;
            case "circular":
                if (cm != null && cm.targetObject != null)
                {
                    return cm.targetObject;
                }
                break;
            case "harmonic":
                if (hm != null && hm.targetObject != null)
                {
                    return hm.targetObject;
                }
                break;
            default:
                Debug.LogError($"未知的任务类型: {taskType}");
                break;
        }
        
        Debug.LogWarning($"无法从{taskType}运动脚本获取目标物体，尝试查找默认物体");
        
        
        return null;
    }
    
    /// <summary>
    /// 从配置获取目标物体（保留用于兼容性）
    /// </summary>
    private Transform GetTargetObjectFromConfig()
    {
        if (currentConfig != null)
        {
            string objectName = currentConfig.objectName;
            GameObject obj = GameObject.Find(objectName);
            return obj?.transform;
        }
        return null;
    }

    /// <summary>
    /// 计算并且发送评估结果到Python端
    /// </summary>
    private void CalculateAndSendCurrentEvaluationResult()
    {
        if (currentEvaluation == null)
        {
            Debug.LogWarning("当前评估数据为空，无法保存");
            return;
        }

        if (success_early)
        {
            SendMetrics(true, 0f, 1f, minDistanceToTarget, success_index);
            return;
        }

        // 计算等待时间分
        float T = 0;
        float score = 0;

        if (!currentEvaluation.success) score = 0;
        else
        {
            switch (currentTaskType)
            {
                case "circular":
                    T = Mathf.Abs(360f / cm.angularSpeed);
                    break;
                case "harmonic":
                    T = Mathf.Abs((2 * Mathf.PI) / hm.angularFrequency);
                    break;
                case "linear":
                    T = 1f;
                    break;
            }
            score = Mathf.Max(0, 1 - (currentEvaluation.wait_time / T));
        }

        // 发送评估结果到python端
        SendMetrics(currentEvaluation.success, currentEvaluation.wait_time, score, currentEvaluation.min_distance_to_target, total_frame_episode);

    }
    void OnDrawGizmos()
    {
        // 在参考平面上绘制手掌心点
        Vector3 palmPosition = palmCenter.position;
        if (lm != null)
        {
            Transform referencePlane = lm.referencePlane;
            Vector3 palmXZ_plane = new Vector3(palmPosition.x, referencePlane.position.y, palmPosition.z);
            Gizmos.color = Color.red;
            Gizmos.DrawSphere(palmXZ_plane, 0.05f);

            if (targetObject != null)
            {
                Gizmos.color = Color.green;
                Vector3 targetObject_plane = new Vector3(targetObject.position.x, referencePlane.position.y, targetObject.position.z);
                Gizmos.DrawSphere(targetObject_plane, 0.05f);
            }
        }
    }

}

// WebSocket通信相关数据结构
[System.Serializable]
public class WebSocketMessage
{
    public string type;
    public string data;
}

[System.Serializable]
public class EpisodeRequest
{
    public int episode_id;
    public string task_type;
    public int retry_time;
    public int total_frame_episode;
    public int windowSize;
}


[System.Serializable]
public class ActionFrame
{
    public float[] values;  // 每一帧的 18 维 或3 维参数
}

[System.Serializable]
public class ActionWindowData
{
    public ActionFrame[] actions;
}


[System.Serializable]
public class ImageAndStateMessage
{
    public int episode_id;
    public int repeat;
    public int step;
    public string image_data;
    public float[] state_data; // 18维或3维状态数据
    public string task_type;
}

[System.Serializable]
public class Metrics
{
    public int episode_id;
    public int repeat;
    public float waitTime;
    public float score; // 等候时间分 = 1 - (waitTime / T)
    public bool success;
    public float min_distance_to_target;

    public int successIndex; // 成功时候的帧索引，当提前成功的时候被赋值，否则就是整个episode的总帧数

    // 关节到物体表面的最小距离
    public float minJointToSurfaceDistance; // 关节到物体表面的最小距离
}

[System.Serializable]
public class LinearConfigWrapper
{
    public List<LinearConfig> configs;
}

[System.Serializable]
public class CircularConfigWrapper
{
    public List<CircularConfig> configs;
}

[System.Serializable]
public class HarmonicConfigWrapper
{
    public List<HarmonicConfig> configs;
}

// 评估相关数据结构

[System.Serializable]
public class SerializableVector3
{
    public float x;
    public float y;
    public float z;

    public SerializableVector3(Vector3 vector)
    {
        x = vector.x;
        y = vector.y;
        z = vector.z;
    }

    public SerializableVector3(float x, float y, float z)
    {
        this.x = x;
        this.y = y;
        this.z = z;
    }
}

[System.Serializable]
public class EpisodeEvaluation
{
    public int episode_id;
    public int repeat;
    public string task_type;
    // public SerializableVector3 final_palm_position;
    public float wait_time;
    public bool success;
    public float min_distance_to_target;
    public float start_time;
    public float end_time;
}

[System.Serializable]
public class VLATestSummary
{
    public int total_episodes;
    public List<EpisodeEvaluation> evaluations = new List<EpisodeEvaluation>();
    public string test_completed_time;
}