using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using System.IO;
using System;
using EpisodeControl;

/// <summary>
/// VLA模型测试管理器 - 简化调试版
/// 只负责物体运动和图像捕获，用于调试基础功能
/// </summary>
public class VLATestManager2 : MonoBehaviour
{
    [Header("测试配置")]
    public float captureFPS = 20f; // 图像捕获频率
    [Tooltip("摄像头画面宽度")]
    public int width = 640;
    [Tooltip("摄像头画面高度")]
    public int height = 480;

    [Header("组件引用")]
    public Camera testCamera;

    [Header("环境清理")]
    public Transform[] movableObjects; // 需要重置位置的可移动物体
    private Vector3[] initialObjectPositions; // 可移动物体的初始位置
    private Quaternion[] initialObjectRotations; // 可移动物体的初始旋转

    // 简化的测试状态
    public enum TestState
    {
        Initializing,
        Running,
        Complete
    }

    [Header("当前状态")]
    public TestState currentState = TestState.Initializing;
    public int currentEpisode = 16724;
    public int currentStep = 0;

    // Episode管理相关
    private BaseConfig currentConfig;
    private string currentTaskType = "harmonic";

    // 内部管理
    private Texture2D captureTexture;

    // 运动脚本相关
    public Transform scriptLoader; // 挂载运动脚本的物体
    public HarmonicMotion hm;
    public CircularMotion cm;
    public LinearMotion lm;

    // 图像捕获相关
    private float captureTimer = 0f;
    private float captureInterval;
    private Transform targetObject;

    void Start()
    {
        InitializeSimpleTest();
    }

    void Update()
    {
        UpdateSimpleTest();
    }

    /// <summary>
    /// 初始化简化测试系统
    /// </summary>
    private void InitializeSimpleTest()
    {
        // 设置图像捕获间隔
        captureInterval = 1f / captureFPS;

        // 记录可移动物体的初始状态
        RecordMovableObjectsInitialStates();

        // 初始化相机捕获
        if (testCamera != null)
        {
            captureTexture = new Texture2D(width, height, TextureFormat.RGB24, false);
        }

        // 初始化运动控制脚本
        hm = scriptLoader.GetComponent<HarmonicMotion>();
        cm = scriptLoader.GetComponent<CircularMotion>();
        lm = scriptLoader.GetComponent<LinearMotion>();

        Debug.Log("简化测试系统初始化完成");

        // 自动开始测试
        StartSimpleEpisode();
    }

    /// <summary>
    /// 更新简化测试
    /// </summary>
    private void UpdateSimpleTest()
    {
        switch (currentState)
        {
            case TestState.Initializing:
                break;

            case TestState.Running:
                UpdateCapture();
                break;

            case TestState.Complete:
                break;
        }
    }

    /// <summary>
    /// 更新图像捕获
    /// </summary>
    private void UpdateCapture()
    {
        captureTimer += Time.deltaTime;

        if (captureTimer >= captureInterval)
        {
            CaptureImageAndSave();
            captureTimer = 0f;
            currentStep++;

            // 运行一定时间后停止（比如5秒 = 100帧）
            if (currentStep >= 100)
            {
                Debug.Log("图像捕获完成，停止测试");
                currentState = TestState.Complete;
            }
        }
    }

    /// <summary>
    /// 开始简化Episode
    /// </summary>
    private void StartSimpleEpisode()
    {
        Debug.Log($"开始简化Episode {currentEpisode}");

        // 动态加载配置
        currentConfig = LoadConfigByTaskType(currentTaskType, currentEpisode);
        if (currentConfig == null)
        {
            Debug.LogError($"无法加载Episode {currentEpisode} 的{currentTaskType}配置");
            return;
        }

        // 获取目标物体
        targetObject = GetTargetObjectFromConfig();
        Debug.Log($"目标物体: {targetObject?.name}");

        // 先清理桌面，重置所有可移动物体
        ClearDesktop();

        // 然后设置运动脚本
        SetupTargetObjectMotion();

        // 验证目标物体位置
        if (targetObject != null)
        {
            Debug.Log($"目标物体设置完成，位置: ({targetObject.position.x:F3}, {targetObject.position.y:F3}, {targetObject.position.z:F3})");
        }

        // 重置计时器
        captureTimer = 0f;
        currentStep = 0;

        // 开始运行
        currentState = TestState.Running;
        Debug.Log("Episode开始，开始图像捕获...");
    }

    /// <summary>
    /// 根据task_type和episode_id动态加载配置
    /// </summary>
    private BaseConfig LoadConfigByTaskType(string taskType, int episodeId)
    {
        string configFileName = $"level2_{taskType}_configs.json";
        string configPath = Path.Combine(Application.dataPath, "..", "ParaConfigs", configFileName);

        if (!File.Exists(configPath))
        {
            Debug.LogError($"配置文件不存在: {configPath}");
            return null;
        }

        try
        {
            string json = File.ReadAllText(configPath);
            string wrappedJson = "{\"configs\":" + json + "}";

            // 根据task_type解析对应的配置
            switch (taskType.ToLower())
            {
                case "linear":
                    var linearWrapper = JsonUtility.FromJson<LinearConfigWrapper>(wrappedJson);
                    var linearConfigs = linearWrapper.configs;
                    return linearConfigs.Find(c => c.index == episodeId);

                case "circular":
                    var circularWrapper = JsonUtility.FromJson<CircularConfigWrapper>(wrappedJson);
                    var circularConfigs = circularWrapper.configs;
                    return circularConfigs.Find(c => c.index == episodeId);

                case "harmonic":
                    var harmonicWrapper = JsonUtility.FromJson<HarmonicConfigWrapper>(wrappedJson);
                    var harmonicConfigs = harmonicWrapper.configs;
                    return harmonicConfigs.Find(c => c.index == episodeId);

                default:
                    Debug.LogError($"未知的task_type: {taskType}");
                    return null;
            }
        }
        catch (System.Exception e)
        {
            Debug.LogError($"加载配置文件失败: {e.Message}");
            return null;
        }
    }

    /// <summary>
    /// 设置目标物体的运动脚本
    /// </summary>
    private void SetupTargetObjectMotion()
    {
        if (currentConfig == null || targetObject == null) return;

        switch (currentConfig)
        {
            case LinearConfig linearConfig:
                SetupLinearMotion(linearConfig);
                Debug.Log($"设置直线运动，初始位置: {linearConfig.initial_position}, 运动方向: {linearConfig.direction}, 初始速度: {linearConfig.initial_speed}, 加速度: {linearConfig.acceleration}");
                break;
            case CircularConfig circularConfig:
                SetupCircularMotion(circularConfig);
                Debug.Log($"设置圆周运动，初始位置: {circularConfig.localCenter}, 半径: {circularConfig.radius}, 角速度: {circularConfig.angularSpeed}, 起始角度: {circularConfig.startAngleDeg}");
                break;
            case HarmonicConfig harmonicConfig:
                SetupHarmonicMotion(harmonicConfig);
                Debug.Log($"设置简谐运动，初始位置: {harmonicConfig.localCenter}, 方向: {harmonicConfig.localDirection}, 幅度: {harmonicConfig.amplitude}, 角频率: {harmonicConfig.angularFrequency}, 初始相位: {harmonicConfig.initialPhase}");
                break;
        }
    }

    /// <summary>
    /// 设置直线运动
    /// </summary>
    private void SetupLinearMotion(LinearConfig config)
    {
        if (lm == null) Debug.LogError("LinearMotion组件不存在");
        lm.enabled = true;
        hm.enabled = false;
        cm.enabled = false;

        lm.SetMotion(
            config.objectName,
            config.initial_position,
            config.direction.normalized,
            config.initial_speed,
            config.acceleration
        );
    }

    /// <summary>
    /// 设置圆周运动
    /// </summary>
    private void SetupCircularMotion(CircularConfig config)
    {
        if (cm == null) Debug.LogError("CircularMotion组件不存在");
        cm.enabled = true;
        lm.enabled = false;
        hm.enabled = false;

        cm.SetMotion(
            config.objectName,
            config.localCenter,
            config.radius,
            config.angularSpeed,
            config.startAngleDeg
        );
    }

    /// <summary>
    /// 设置简谐运动
    /// </summary>
    private void SetupHarmonicMotion(HarmonicConfig config)
    {
        if (hm == null) Debug.LogError("HarmonicMotion组件不存在");
        hm.enabled = true;
        lm.enabled = false;
        cm.enabled = false;

        hm.SetMotion(
            config.objectName,
            config.localCenter,
            config.localDirection,
            config.amplitude,
            config.angularFrequency,
            config.initialPhase
        );
    }

    /// <summary>
    /// 记录可移动物体的初始状态
    /// </summary>
    private void RecordMovableObjectsInitialStates()
    {
        if (movableObjects != null && movableObjects.Length > 0)
        {
            initialObjectPositions = new Vector3[movableObjects.Length];
            initialObjectRotations = new Quaternion[movableObjects.Length];

            for (int i = 0; i < movableObjects.Length; i++)
            {
                if (movableObjects[i] != null)
                {
                    initialObjectPositions[i] = movableObjects[i].position;
                    initialObjectRotations[i] = movableObjects[i].rotation;
                }
            }

            Debug.Log($"记录了 {movableObjects.Length} 个可移动物体的初始状态");
        }
    }

    /// <summary>
    /// 清理桌面，重置所有可移动物体到初始位置（排除当前目标物体）
    /// </summary>
    private void ClearDesktop()
    {
        if (movableObjects != null && initialObjectPositions != null && initialObjectRotations != null)
        {
            int resetCount = 0;
            for (int i = 0; i < movableObjects.Length && i < initialObjectPositions.Length; i++)
            {
                if (movableObjects[i] != null)
                {
                    // 如果当前物体是目标物体，跳过重置（让运动脚本来设置位置）
                    if (targetObject != null && movableObjects[i] == targetObject)
                    {
                        Debug.Log($"跳过重置当前目标物体: {targetObject.name}");
                        continue;
                    }

                    movableObjects[i].position = initialObjectPositions[i];
                    movableObjects[i].rotation = initialObjectRotations[i];

                    // 重置物体的物理状态
                    Rigidbody rb = movableObjects[i].GetComponent<Rigidbody>();
                    if (rb != null)
                    {
                        rb.velocity = Vector3.zero;
                        rb.angularVelocity = Vector3.zero;
                    }
                    resetCount++;
                }
            }

            Debug.Log($"已清理桌面，重置了 {resetCount} 个物体（排除目标物体）");
        }
    }

    /// <summary>
    /// 捕获图像并保存到本地
    /// </summary>
    private void CaptureImageAndSave()
    {
        if (testCamera != null && captureTexture != null)
        {
            // 渲染相机画面
            RenderTexture renderTexture = new RenderTexture(width, height, 24);
            testCamera.targetTexture = renderTexture;
            testCamera.Render();

            RenderTexture.active = renderTexture;
            captureTexture.ReadPixels(new Rect(0, 0, renderTexture.width, renderTexture.height), 0, 0);
            captureTexture.Apply();
            RenderTexture.active = null;

            // 转换为 JPG
            byte[] imageBytes = captureTexture.EncodeToJPG(85);

            // 保存目录：项目根目录下的 debug 文件夹
            string debugDir = Path.Combine(Application.dataPath, "..", "debug");
            Directory.CreateDirectory(debugDir);

            // 保存图片
            string imagePath = Path.Combine(debugDir, $"episode_{currentEpisode}_step_{currentStep:D4}.jpg");
            File.WriteAllBytes(imagePath, imageBytes);

            Debug.Log($"图像已保存: step_{currentStep:D4} -> {imagePath}");

            // 清理
            testCamera.targetTexture = null;
            DestroyImmediate(renderTexture);
        }
    }

    /// <summary>
    /// 从配置获取目标物体
    /// </summary>
    private Transform GetTargetObjectFromConfig()
    {
        if (currentConfig != null)
        {
            string objectName = currentConfig.objectName;
            GameObject obj = GameObject.Find(objectName);
            return obj?.transform;
        }
        return null;
    }
}
