import math
import threading
from collections import deque
from gurobipy import GRB


def _is_int_like(v) -> bool:
    """Best-effort check whether a solver variable is integer-like."""
    try:
        return bool(v.isBinary() or v.isIntegral())
    except Exception:
        pass
    t = str(getattr(v, "vtype", lambda: "C")()).upper()
    return t in {"B", "I", "BINARY", "INTEGER", "IMPLINT"}


def _name_map(m):
    """Return {var_name: var_obj} for a SCIP-like model."""
    return {v.name: v for v in m.getVars()}


def _clamp_to_orig_bounds(val, v):
    """Clamp value to original bounds if available."""
    try:
        lb, ub = v.getLbOriginal(), v.getUbOriginal()
        if val < lb:
            val = lb
        if val > ub:
            val = ub
    except Exception:
        pass
    return val


def _get_lb_curr(v):
    for attr in ("getLb", "getLbLocal", "getLbOriginal"):
        try:
            return getattr(v, attr)()
        except Exception:
            pass
    return -1e20


def _get_ub_curr(v):
    for attr in ("getUb", "getUbLocal", "getUbOriginal"):
        try:
            return getattr(v, attr)()
        except Exception:
            pass
    return 1e20


class SCIPPrimalHint:
    """
    Rolling in-memory primal-hint buffer for SCIP.

    Strategies
    ----------
    - "obj":    Directly inject full solutions (round integrals, clamp to original bounds).
    - "int_lp": Use integer assignments only -> temporarily fix them -> short-time optimize
                to repair continuous vars -> fallback fill -> inject.
    - "auto":   Try "obj" first; if nothing applied, fall back to "int_lp".
    """

    def __init__(
        self,
        k: int = 10,
        strategy: str = "obj",
        lp_repair_time: float = 5.0,
        int_coverage_min: float = 0.95,
    ):
        self._k = max(0, int(k))
        self._buf = deque(maxlen=self._k)  # list[dict[var_name -> value]]
        self._lk = threading.Lock()
        self.strategy = (strategy or "obj").lower().strip()
        self.lp_repair_time = float(lp_repair_time)
        self.int_coverage_min = float(int_coverage_min)

    def reset(self, k: int | None = None):
        """Reset ring buffer length (keeps most recent)."""
        if k is None:
            return
        with self._lk:
            self._k = max(0, int(k))
            self._buf = deque(self._buf, maxlen=self._k)

    # ---------- Public API ----------
    def load(self, model) -> int:
        """Inject historical solutions to the current model. Return # of addSol successes."""
        with self._lk:
            hist = list(self._buf)
        if not hist:
            return 0

        if self.strategy == "obj":
            return self._load_obj(model, hist)
        if self.strategy == "int_lp":
            return self._load_int_lp(model, hist)
        if self.strategy == "auto":
            applied = self._load_obj(model, hist)
            if applied > 0:
                return applied
            return self._load_int_lp(model, hist)
        return 0

    def store(self, model) -> None:
        """Capture incumbent solution as {name: value} and push to buffer."""
        try:
            sol = model.getBestSol()
            if sol is None:
                return
            mapping = {}
            for v in model.getVars():
                try:
                    val = model.getSolVal(sol, v)
                    if val is None or (isinstance(val, float) and math.isnan(val)):
                        continue
                    mapping[v.name] = float(val)
                except Exception:
                    continue
            if not mapping:
                return
            with self._lk:
                self._buf.append(mapping)
        except Exception:
            pass

    # ---------- Strategies ----------
    def _load_obj(self, model, hist) -> int:
        """Construct solutions directly and add via addSol()."""
        name2 = _name_map(model)
        applied = 0
        for mapping in hist[-self._k:]:
            newsol = model.createSol()
            setcnt = 0
            for name, val in mapping.items():
                v = name2.get(name)
                if v is None:
                    continue
                if _is_int_like(v):
                    try:
                        val = int(round(val))
                    except Exception:
                        continue
                val = _clamp_to_orig_bounds(val, v)
                try:
                    model.setSolVal(newsol, v, float(val))
                    setcnt += 1
                except Exception:
                    pass
            try:
                if setcnt > 0 and model.addSol(newsol):
                    applied += 1
            except Exception:
                pass
        return applied

    def _load_int_lp(self, model, hist) -> int:
        """Integer-only fix + short-time repair (continuous) + fallback fill + inject."""
        name2 = _name_map(model)
        tgt_int_names = [v.name for v in model.getVars() if _is_int_like(v)]
        requested = len(tgt_int_names)
        applied = 0

        for mapping in hist[-self._k:]:
            ints = {}
            found = 0
            for nm in tgt_int_names:
                if nm in mapping:
                    try:
                        ints[nm] = int(round(mapping[nm]))
                        found += 1
                    except Exception:
                        pass
            if requested == 0 or found == 0:
                continue
            coverage = found / requested
            if coverage < self.int_coverage_min:
                continue

            saved_bounds = {}
            try:
                model.freeTransform()
            except Exception:
                pass

            for nm, ival in ints.items():
                v = name2.get(nm)
                if v is None:
                    continue
                lb, ub = _get_lb_curr(v), _get_ub_curr(v)
                saved_bounds[nm] = (lb, ub)
                val = max(lb, min(ival, ub))
                try:
                    model.chgVarLb(v, val)
                    model.chgVarUb(v, val)
                except Exception:
                    saved_bounds.pop(nm, None)
                    continue

            try:
                model.setRealParam("limits/time", float(self.lp_repair_time))
            except Exception:
                pass
            try:
                model.optimize()
            except Exception:
                pass

            solvals = None
            try:
                if model.getNSols() > 0:
                    sol_rep = model.getBestSol()
                    solvals = {}
                    for name, v in name2.items():
                        try:
                            solvals[name] = model.getSolVal(sol_rep, v)
                        except Exception:
                            pass
            except Exception:
                solvals = None

            try:
                model.freeTransform()
            except Exception:
                pass
            for nm, (lb, ub) in saved_bounds.items():
                v = name2.get(nm)
                if v is None:
                    continue
                try:
                    model.chgVarLb(v, lb)
                    model.chgVarUb(v, ub)
                except Exception:
                    pass

            if solvals is None:
                continue

            newsol = model.createSol()
            setcnt = 0
            for name, v in name2.items():
                val = None
                if name in solvals:
                    val = solvals[name]
                if val is None and name in ints:
                    val = ints[name]
                if val is None:
                    try:
                        lb0, ub0 = v.getLbOriginal(), v.getUbOriginal()
                        if lb0 == ub0:
                            val = lb0
                    except Exception:
                        pass
                if val is None:
                    continue

                if _is_int_like(v):
                    try:
                        val = int(round(val))
                    except Exception:
                        continue
                val = _clamp_to_orig_bounds(val, v)
                try:
                    model.setSolVal(newsol, v, float(val))
                    setcnt += 1
                except Exception:
                    pass

            try:
                if setcnt > 0 and model.addSol(newsol):
                    applied += 1
            except Exception:
                pass

        return applied


# ---------------------- GurobiPrimalHint ----------------------
def _grb_varmap(model):
    return {v.VarName: v for v in model.getVars()}


def _grb_clamp_to_bounds(val, v):
    lb = v.LB if v.LB is not None else float("-inf")
    ub = v.UB if v.UB is not None else float("inf")
    if val < lb:
        val = lb
    if val > ub:
        val = ub
    return val


class GurobiPrimalHint:
    """
    Rolling in-memory primal-hint buffer for Gurobi, using MIP starts.

    Strategies
    ----------
    - "obj":    Set Start for all variables (round integrals; clamp to current LB/UB).
    - "int_lp": Set Start only for integer variables (continuous left unset).
    - "auto":   Alias of "obj".
    """

    def __init__(self, k: int = 10, strategy: str = "obj", **_kwargs):
        self._k = max(0, int(k))
        self._buf = deque(maxlen=self._k)  # list[dict[var_name -> value]]
        self._lk = threading.Lock()
        st = (strategy or "obj").lower().strip()
        self.strategy = "obj" if st == "auto" else st

    def reset(self, k: int | None = None):
        """Reset ring buffer length (keeps most recent)."""
        if k is None:
            return
        with self._lk:
            self._k = max(0, int(k))
            self._buf = deque(self._buf, maxlen=self._k)

    def load(self, model) -> int:
        """Inject historical mappings as multiple MIP starts. Return number of starts injected."""
        with self._lk:
            hist = list(self._buf)
        if not hist:
            return 0

        name2 = _grb_varmap(model)
        usable = hist[-self._k:]
        S = len(usable)
        if S <= 0:
            return 0

        model.NumStart = S
        model.update()

        ints_only = (self.strategy == "int_lp")

        for s, mapping in enumerate(usable):
            model.Params.StartNumber = s
            for name, val in mapping.items():
                v = name2.get(name)
                if v is None:
                    continue
                try:
                    if v.VType != GRB.CONTINUOUS:
                        val = round(val)
                    else:
                        if ints_only:
                            continue
                except Exception:
                    if ints_only:
                        continue
                try:
                    v.Start = float(_grb_clamp_to_bounds(float(val), v))
                except Exception:
                    pass

        return S

    def store(self, model) -> None:
        """Capture incumbent solution as {name: value} and push to buffer."""
        try:
            if model.SolCount <= 0:
                return
        except Exception:
            return

        mapping = {}
        for v in model.getVars():
            try:
                x = v.X
                if x is None or (isinstance(x, float) and math.isnan(x)):
                    continue
                mapping[v.VarName] = float(x)
            except Exception:
                continue
        if not mapping:
            return
        with self._lk:
            self._buf.append(mapping)
