import gurobipy as gp
import pyscipopt as pyopt
from pkg_resources import resource_filename
from argparse import Namespace
from typing import Optional
from evaluation.util import get_cutting_planes

# ---------------------------------
# Gurobi backend (seed-respecting)
# ---------------------------------

def _gurobi_solve(input: tuple, args: dict, fathom_time: float, hint_mgr=None) -> float:
    """
    input: (action, model_file, seed)
    args:  {"solver", "time_limit", "gap_limit"}
    Returns: solver work units (float)
    """
    action, file, seed = input
    with gp.Env(empty=True) as env:
        env.setParam("LogToConsole", 0)
        env.start()
        with gp.read(file, env=env) as model:
            model.setParam("Seed", int(seed))
            model.setParam("Threads", 1)
            model.setParam("MIPGap", args["gap_limit"])

            # Optional: last bit of action vector toggles primal-hint usage
            primal_flag = 0
            if action is not None:
                nseps = len(get_cutting_planes("gurobi"))
                if len(action) > nseps:
                    primal_flag = int(action[nseps] > 0.5)

            # Apply separator parameters from action (if provided)
            if action is not None:
                for i, sep in enumerate(get_cutting_planes("gurobi")):
                    model.setParam(sep, action[i])

            # Inject/store primal hints if enabled
            if primal_flag and hint_mgr is not None:
                try:
                    hint_mgr.load(model)
                except Exception:
                    pass

            model.setParam("WorkLimit", args["time_limit"])
            model.optimize()

            if primal_flag and hint_mgr is not None:
                try:
                    hint_mgr.store(model)
                except Exception:
                    pass

            work = model.Work
    return work


# ---------------------------------
# SCIP backend (seed-respecting)
# ---------------------------------

def _scip_solve(input: tuple, args: dict, fathom_time: float, hint_mgr=None) -> float:
    """
    input: (action, model_file, seed)
    args:  {"solver", "time_limit", "gap_limit"}
    Returns: wall-clock solving time (float)
    """
    action, file, seed = input
    model = pyopt.Model()
    model.setParam("display/verblevel", 0)
    model.readProblem(file)
    model.setIntParam("parallel/maxnthreads", 1)
    model.readParams(resource_filename(__name__, "randomness_control.set"))

    for p in ("randomization/randomseedshift", "randomization/permutationseed", "randomization/lpseed"):
        try:
            model.setIntParam(p, int(seed))
        except Exception:
            pass

    model.setParam("limits/gap", args["gap_limit"])

    # Interpret action: separators + (optional) primal hint flag + (optional) root-only flag
    primal_flag = 0
    root_only_flag = 0
    nseps = len(get_cutting_planes("scip"))

    if action is not None:
        if len(action) > nseps:
            primal_flag = int(action[nseps] > 0.5)
        if len(action) > nseps + 1:
            root_only_flag = int(action[nseps + 1] > 0.5)

        seps = get_cutting_planes("scip")
        for i, sep in enumerate(seps):
            enabled = bool(action[i] > 0.5)
            if not enabled:
                freq = -1
            else:
                freq = 0 if root_only_flag else 1
            model.setParam(f"separating/{sep}/freq", int(freq))

    # Inject/store primal hints if enabled
    if primal_flag and hint_mgr is not None:
        try:
            hint_mgr.load(model)
        except Exception:
            pass

    model.setParam("limits/time", args["time_limit"])
    model.optimize()

    if primal_flag and hint_mgr is not None:
        try:
            hint_mgr.store(model)
        except Exception:
            pass

    return model.getSolvingTime()


# --------------------------
# Front-end dispatch
# --------------------------

def solve(input: tuple, args: dict, fathom_time: float, hint_mgr=None):
    """Dispatch to the selected solver backend."""
    solve_fns = {"gurobi": _gurobi_solve, "scip": _scip_solve}
    return solve_fns[args["solver"]](input, args, fathom_time, hint_mgr)
