#!/bin/bash

# Move problematic response files to backup directories
# This handles patterns that had the duplicate issue: mixed_3, cap10_3, small_4, large_2

echo "Moving problematic response files to backup directories..."
echo "============================================"

# Define problematic patterns
PROBLEMATIC_PATTERNS=("mixed_3" "cap10_3" "small_4" "large_2")

# Counter variables
TOTAL_MOVED=0
TOTAL_DIRS_CREATED=0

# Function to move files matching a pattern
move_pattern_files() {
    local pattern=$1
    local count=0
    
    echo ""
    echo "Processing pattern: $pattern"
    echo "----------------------------"
    
    # Find all response directories
    for response_dir in datasets/*/*/responses/; do
        if [ -d "$response_dir" ]; then
            # Check if any files match this pattern
            matching_files=$(ls "$response_dir"*"$pattern"* 2>/dev/null | wc -l)
            
            if [ "$matching_files" -gt 0 ]; then
                # Create backup directory if it doesn't exist
                backup_dir="${response_dir%/}_old"
                if [ ! -d "$backup_dir" ]; then
                    mkdir -p "$backup_dir"
                    echo "  Created backup dir: $backup_dir"
                    ((TOTAL_DIRS_CREATED++))
                fi
                
                # Move all matching files
                for file in "$response_dir"*"$pattern"*; do
                    if [ -f "$file" ]; then
                        filename=$(basename "$file")
                        mv "$file" "$backup_dir/$filename"
                        echo "  Moved: $filename"
                        ((count++))
                        ((TOTAL_MOVED++))
                    fi
                done
            fi
        fi
    done
    
    echo "  Total files moved for $pattern: $count"
}

# Process each problematic pattern
for pattern in "${PROBLEMATIC_PATTERNS[@]}"; do
    move_pattern_files "$pattern"
done

echo ""
echo "============================================"
echo "SUMMARY:"
echo "  Directories created: $TOTAL_DIRS_CREATED"
echo "  Files moved: $TOTAL_MOVED"
echo "============================================"

# Optional: List backup directories created
echo ""
echo "Backup directories created:"
for backup_dir in datasets/*/*/responses_old/; do
    if [ -d "$backup_dir" ]; then
        file_count=$(ls "$backup_dir" 2>/dev/null | wc -l)
        echo "  $backup_dir ($file_count files)"
    fi
done