#!/bin/bash
# test_reformat_enhanced.sh
# Script to test the enhanced reformat_cluster_results.py with your actual data

echo "🧪 Testing Enhanced Cluster Results Reformatting with Token Analysis"
echo "=================================================================="

# Check if we have any cluster results files
CLUSTER_FILES=(llm-inference/results/qwen-3-32B_results_20250617_093913.json)

if [ ! -f "${CLUSTER_FILES[0]}" ]; then
    echo "❌ No cluster results files found in llm-inference/results/"
    echo "   Looking for files matching: llm-inference/results/*_results_*.json"
    echo "   Please make sure you have cluster results files in that directory."
    exit 1
fi

echo "📁 Found cluster results files:"
for file in "${CLUSTER_FILES[@]}"; do
    if [ -f "$file" ]; then
        echo "   - $(basename "$file")"
    fi
done
echo

# Pick the first file for testing
TEST_FILE="${CLUSTER_FILES[0]}"
echo "🎯 Testing with: $(basename "$TEST_FILE")"

# Show file info
echo "📊 File info:"
echo "   Size: $(du -h "$TEST_FILE" | cut -f1)"
echo "   Records: $(jq length "$TEST_FILE" 2>/dev/null || echo "Could not count")"
echo

# Test 1: Run with token analysis (default)
echo "🧠 Test 1: Running with token analysis enabled..."
echo "Command: python scripts/reformat_cluster_results.py \"$TEST_FILE\" --verbose"
echo "----------------------------------------"

if python -m scripts.reformat_cluster_results "$TEST_FILE" --verbose; then
    echo "✅ Test 1 passed!"
else
    echo "❌ Test 1 failed!"
    exit 1
fi

echo

# Check if token file was created
TOKEN_FILE="${TEST_FILE%.*}_tokens.json"
if [ -f "$TOKEN_FILE" ]; then
    echo "✅ Token data file created: $(basename "$TOKEN_FILE")"
    echo "   Size: $(du -h "$TOKEN_FILE" | cut -f1)"
    
    # Show a sample of the token data
    echo "   Sample content:"
    jq -r '.batch_statistics | to_entries[] | "     \(.key): \(.value)"' "$TOKEN_FILE" 2>/dev/null || echo "     Could not parse token file"
else
    echo "❌ Token data file not found: $(basename "$TOKEN_FILE")"
fi

echo

# Test 2: Check that individual response files were created
echo "🔍 Test 2: Checking individual response files..."

# Count response files created
RESPONSE_COUNT=$(find datasets/ -name "*.txt" -newer "$TEST_FILE" 2>/dev/null | wc -l)
echo "   Response files created: $RESPONSE_COUNT"

if [ $RESPONSE_COUNT -gt 0 ]; then
    echo "✅ Individual response files created successfully!"
    
    # Show a few examples
    echo "   Sample response files:"
    find datasets/ -name "*.txt" -newer "$TEST_FILE" 2>/dev/null | head -3 | while read file; do
        echo "     - $file"
    done
else
    echo "❌ No individual response files found!"
fi

echo

# Test 3: Compare with running without token analysis
echo "🚫 Test 3: Testing without token analysis..."
echo "Command: python scripts/reformat_cluster_results.py \"$TEST_FILE\" --no-token-analysis --verbose"
echo "----------------------------------------"

if python -m scripts.reformat_cluster_results "$TEST_FILE" --no-token-analysis --verbose; then
    echo "✅ Test 3 passed!"
else
    echo "❌ Test 3 failed!"
fi

echo

# Summary
echo "📋 Test Summary:"
echo "================"
echo "✅ Enhanced reformat script executed successfully"
echo "✅ Token analysis functionality working"
echo "✅ Individual response files created"
echo "✅ Backward compatibility maintained"

if [ -f "$TOKEN_FILE" ]; then
    echo "✅ Consolidated token data file: $(basename "$TOKEN_FILE")"
fi

echo
echo "🎉 All tests completed! The enhanced reformat script is ready to use."
echo
echo "💡 Next steps:"
echo "   1. Use the enhanced script for all your cluster results"
echo "   2. Check the consolidated *_tokens.json files"
echo "   3. Ready to modify evaluate_responses.py to use consolidated token data"
