import os


def encode_as_adjacency_list(
    G, save_path="datasets/temp/textual/adjacency.txt", verbose=False
):
    """
    Saves the adjacency list representation of the graph to a file, including node colors.

    Parameters:
    - G: networkx.Graph
    - save_path: str, file path for saving the encoding
    - verbose: bool, if True, lists all node colors explicitly, otherwise applies optimized descriptions.

    Format:
    - In an undirected graph, (i,j) means that node i and node j are connected with an undirected edge.
    - Node colors are listed with optimizations to avoid redundancy.
    """

    os.makedirs(os.path.dirname(save_path), exist_ok=True)

    nodes_list = ", ".join(map(str, G.nodes))
    edges_list = " ".join(f"({i},{j})" for i, j in G.edges)

    # Capture node colors (excluding grey)
    colored_nodes = {node: G.nodes[node].get("color", "grey") for node in G.nodes}
    non_grey_nodes = {
        node: color for node, color in colored_nodes.items() if color != "grey"
    }

    # Determine how to describe colored nodes
    unique_colors = set(non_grey_nodes.values())

    if not non_grey_nodes:
        color_description = "None of the nodes are colored."
    elif len(unique_colors) == 1:
        color_description = f"The following nodes are colored: {', '.join(map(str, non_grey_nodes.keys()))}."
    else:
        color_description = "\n".join(
            f"Node {node} is {color}." for node, color in non_grey_nodes.items()
        )

    # Grey node description
    grey_nodes = [node for node in G.nodes if node not in non_grey_nodes]
    grey_description = (
        "\n".join(f"Node {node} is uncolored." for node in grey_nodes)
        if verbose
        else ""
    )

    result = (
        "In an undirected graph, (i,j) means that node i\n"
        "and node j are connected with an undirected edge. G\n"
        f"describes a graph among nodes {nodes_list}.\n"
        f"The edges in G are: {edges_list}.\n\n"
        f"{color_description}\n{grey_description}"
    ).strip()

    with open(save_path, "w", encoding="utf-8") as f:
        f.write(result)


def encode_as_incident_list(
    G, save_path="datasets/temp/textual/incident.txt", verbose=False
):
    """
    Saves the incident list representation of the graph to a file, including node colors.

    Format:
    - Lists which nodes are connected to which others.
    - Lists node colors, avoiding redundant descriptions.
    """

    os.makedirs(os.path.dirname(save_path), exist_ok=True)

    nodes_list = ", ".join(map(str, G.nodes))
    incident_list = "\n".join(
        f"Node {node} is connected to nodes {', '.join(map(str, G.neighbors(node)))}."
        for node in G.nodes
    )

    # Capture node colors (excluding grey)
    colored_nodes = {node: G.nodes[node].get("color", "grey") for node in G.nodes}
    non_grey_nodes = {
        node: color for node, color in colored_nodes.items() if color != "grey"
    }
    unique_colors = set(non_grey_nodes.values())

    if not non_grey_nodes:
        color_description = "None of the nodes are colored."
    elif len(unique_colors) == 1:
        color_description = f"The following nodes are colored: {', '.join(map(str, non_grey_nodes.keys()))}."
    else:
        color_description = "\n".join(
            f"Node {node} is {color}." for node, color in non_grey_nodes.items()
        )

    grey_nodes = [node for node in G.nodes if node not in non_grey_nodes]
    grey_description = (
        "\n".join(f"Node {node} is uncolored." for node in grey_nodes)
        if verbose
        else ""
    )

    result = (
        f"G describes a graph among nodes {nodes_list}.\nIn this graph:\n{incident_list}\n\n"
        f"{color_description}\n{grey_description}"
    ).strip()

    with open(save_path, "w", encoding="utf-8") as f:
        f.write(result)
