import os

from scripts.utils.encode_graph import (encode_as_adjacency_list,
                                       encode_as_incident_list)
from scripts.utils.visualize_graph import visualize_graph
from scripts.utils.metadata import get_or_create_specs, update_specs_with_pair


def determine_size_category(num_nodes):
    """
    Determines the size category based on number of nodes.

    Parameters:
    - num_nodes: int, number of nodes in the graph

    Returns:
    - str: 'small', 'medium', 'large', or 'Xlarge'
    """
    if num_nodes <= 5:
        return "small"
    elif num_nodes <= 10:
        return "medium"
    elif num_nodes <= 15:
        return "large"
    else:
        return "Xlarge"


def generate_and_process_graph(
    name,
    graph_generator,
    transformation,
    num_nodes=9,
    seed=42,
    num=1,
    visualization_layouts=None,
    pre_generated_graph=None,
    task_description=None,
    property_requirements=None,
    transformation_parameters=None,
):
    """
    Generates a graph, applies a transformation, and saves encodings, visualizations, and metadata.
    
    Now with improved metadata handling to ensure all pairs are recorded.
    """
    if visualization_layouts is None:
        visualization_layouts = ["kamada_kawai"]  # Default to just Kamada-Kawai

    # Parse the name to extract graph_type and benchmark
    parts = name.split("_", 1)  # Split only at the first underscore
    if len(parts) == 2:
        graph_type, task_name = parts
    else:
        # Handle case where name doesn't have an underscore
        graph_type = "unknown"
        task_name = name

    # Generate input graph (some functions may return additional values)
    if pre_generated_graph is not None:
        # Use the pre-generated graph
        result = pre_generated_graph
    else:
        # Generate a new graph
        result = graph_generator(num_nodes, seed=seed)
        
    if isinstance(result, tuple):  # Unpack if multiple values are returned
        graph, *extra_values = result
    else:
        graph = result
        extra_values = []

    # New path structure: datasets/<task_name>/<graph_type>/<textual or visual>/<input or output>/<num_nodes>/<encoding>_<number>
    base_path = f"datasets/{task_name}/{graph_type}"
    
    # Initialize or load the specs file
    generator_name = graph_generator.__name__
    generator_params = {"num_nodes": num_nodes}
    
    # Get/create the specs file
    specs = get_or_create_specs(
        benchmark_path=base_path,
        task_name=task_name,
        graph_type=graph_type,
        base_seed=seed,
        generator_function=generator_name,
        generator_parameters=generator_params,
        property_requirements=property_requirements,
        transformation_parameters=transformation_parameters,
        description=task_description
    )

    # Save input graph encodings
    input_path_prefix = f"{base_path}/textual/input/{num_nodes}/"
    os.makedirs(input_path_prefix, exist_ok=True)
    
    adjacency_input_path = f"{input_path_prefix}adjacency{num}.txt"
    incident_input_path = f"{input_path_prefix}incident{num}.txt"
    
    encode_as_adjacency_list(graph, save_path=adjacency_input_path)
    encode_as_incident_list(graph, save_path=incident_input_path)

    # Save input graph visualizations
    for layout in visualization_layouts:
        visual_path = f"{base_path}/visual/input/{num_nodes}/{layout}_{num}.png"
        os.makedirs(os.path.dirname(visual_path), exist_ok=True)
        visualize_graph(
            graph,
            layout=layout,
            directed=False,
            seed=seed,
            save_path=visual_path,
        )

    # Apply transformation to generate output graph
    transformed_graph = transformation(
        graph, *extra_values
    )  # Pass extra values if needed

    # Save output graph encodings
    output_path_prefix = f"{base_path}/textual/output/{num_nodes}/"
    os.makedirs(output_path_prefix, exist_ok=True)
    
    adjacency_output_path = f"{output_path_prefix}adjacency{num}.txt"
    incident_output_path = f"{output_path_prefix}incident{num}.txt"
    
    encode_as_adjacency_list(
        transformed_graph, save_path=adjacency_output_path
    )
    encode_as_incident_list(
        transformed_graph, save_path=incident_output_path
    )

    # Save output graph visualizations
    for layout in visualization_layouts:
        visual_path = f"{base_path}/visual/output/{num_nodes}/{layout}_{num}.png"
        os.makedirs(os.path.dirname(visual_path), exist_ok=True)
        visualize_graph(
            transformed_graph,
            layout=layout,
            directed=False,
            seed=seed,
            save_path=visual_path,
        )
    
    # Prepare file paths for metadata
    file_paths = {
        "input": {
            "adjacency": f"textual/input/{num_nodes}/adjacency{num}.txt",
            "incident": f"textual/input/{num_nodes}/incident{num}.txt"
        },
        "output": {
            "adjacency": f"textual/output/{num_nodes}/adjacency{num}.txt",
            "incident": f"textual/output/{num_nodes}/incident{num}.txt"
        }
    }
    
    # Update specs with this pair's metadata, using improved function with file locking
    update_specs_with_pair(
        benchmark_path=base_path,
        specs=specs,
        num_nodes=num_nodes,
        pair_index=num,
        seed=seed,
        input_graph=graph,
        output_graph=transformed_graph,
        file_paths=file_paths
    )
