import time
import networkx as nx
from typing import Callable, Optional, Tuple, Union
from scripts.utils.properties import PropertyStatus, check_property
from scripts.utils.pretransformations import apply_pretransformations
from scripts.utils.task_compatibility import get_generator_properties


def generate_valid_graph(
    generator_func: Callable,
    required_properties: list,
    required_pretransforms: list = None,
    num_nodes: int = 9,
    seed: Optional[int] = None,
    max_attempts: int = 100,
    timeout_seconds: int = 30,
    debug_mode: bool = False,
    **generator_kwargs,
) -> Union[nx.Graph, Tuple]:
    """
    Generates a graph that satisfies the required properties, retrying if necessary.
    This version uses PropertyStatus enum to determine when to regenerate graphs.

    Parameters:
    - generator_func: Graph generator function
    - required_properties: List of property names the graph must satisfy
    - required_pretransforms: List of pretransformations to apply before checking
    - num_nodes: Number of nodes for the graph
    - seed: Random seed (will be incremented for each attempt)
    - max_attempts: Maximum number of generation attempts
    - timeout_seconds: Maximum time to spend trying to generate a valid graph
    - debug_mode: If True, prints detailed debugging information
    - generator_kwargs: Additional keyword arguments for the generator function

    Returns:
    - A NetworkX graph or tuple (graph, *extra_values) satisfying the required properties

    Raises:
    - TimeoutError: If a valid graph cannot be generated within the timeout period
    - ValueError: If max attempts are reached without finding a valid graph
                 or if a required property is guaranteed FALSE for this generator
    """

    start_time = time.time()
    current_seed = seed

    # Get the generator properties
    generator_props = get_generator_properties(generator_func)

    # Check if any required property is guaranteed to be FALSE for this generator
    incompatible_props = []
    for prop in required_properties:
        if prop in generator_props and generator_props[prop] == PropertyStatus.FALSE:
            incompatible_props.append(prop)

    if incompatible_props:
        raise ValueError(
            f"Generator cannot satisfy the following properties: {incompatible_props}. "
            "These properties are guaranteed to be FALSE for this generator."
        )

    # Identify properties that need to be checked dynamically (not guaranteed TRUE)
    properties_to_check = [
        prop
        for prop in required_properties
        if prop not in generator_props  # Not specified in generator
        or generator_props[prop] == PropertyStatus.MAYBE  # Or specified as MAYBE
    ]

    if debug_mode:
        guaranteed_props = [
            p
            for p in required_properties
            if p in generator_props and generator_props[p] == PropertyStatus.TRUE
        ]
        print(f"Properties guaranteed by generator: {guaranteed_props}")
        print(f"Properties that need verification: {properties_to_check}")

    for attempt in range(max_attempts):
        # Check if we've exceeded the timeout
        if time.time() - start_time > timeout_seconds:
            raise TimeoutError(
                f"Failed to generate a valid graph within {timeout_seconds} seconds"
            )

        # Generate a graph with the current seed
        if current_seed is not None:
            current_seed = seed + attempt  # Increment seed for each attempt

        # Handle generators that might return multiple values
        result = generator_func(num_nodes, seed=current_seed, **generator_kwargs)

        if isinstance(result, tuple):
            # Some generators return (graph, extra_values)
            graph = result[0]
        else:
            graph = result

        # Apply pretransformations if required, seeding random for determinism
        working_graph = graph.copy()
        if required_pretransforms:
            import random
            random.seed(current_seed)
            try:
                working_graph = apply_pretransformations(
                    working_graph, required_pretransforms
                )
            except (ValueError, TypeError) as e:
                if debug_mode:
                    print(f"Pretransformation failed on attempt {attempt+1}: {e}")
                continue

        # Only check properties that aren't guaranteed TRUE
        if all(check_property(working_graph, prop) for prop in properties_to_check):
            if debug_mode:
                print(f"✅ Valid graph generated after {attempt+1} attempts")
            # Return the pre-transformed graph (with pretransformations applied)
            if isinstance(result, tuple):
                return (working_graph,) + tuple(result[1:])
            return working_graph

        if debug_mode and attempt % 10 == 0:
            failed_props = [
                p for p in properties_to_check if not check_property(working_graph, p)
            ]
            print(f"Attempt {attempt+1} failed. Properties not met: {failed_props}")

    raise ValueError(f"Failed to generate a valid graph after {max_attempts} attempts")
