"""
Graph property framework for defining and verifying graph properties.

This module provides:
1. A registry of graph properties with verification functions
2. Functions to check if a graph has specific properties
3. Utilities for working with property requirements
"""

import networkx as nx
from typing import Dict, List, Callable, Any, Set
from enum import Enum, auto


class PropertyStatus(Enum):
    """Status of a property in a graph generator."""

    TRUE = auto()  # Property is guaranteed to be true
    FALSE = auto()  # Property is guaranteed to be false
    MAYBE = auto()  # Property may be true or false depending on random generation

    def __bool__(self):
        """Allow using the enum in boolean context."""
        return self == PropertyStatus.TRUE
    
    def needs_checking(self) -> bool:
        """Check if this property status requires verification."""
        return self == PropertyStatus.MAYBE
    
    def is_possible(self) -> bool:
        """Check if this property status is compatible (not FALSE)."""
        return self != PropertyStatus.FALSE


# Create global variables for easier usage
TRUE = PropertyStatus.TRUE
FALSE = PropertyStatus.FALSE
MAYBE = PropertyStatus.MAYBE

# Export these at the module level
__all__ = [
    "PropertyStatus",
    "TRUE",
    "FALSE",
    "MAYBE",
    # existing exports...
    "register_property",
    "check_property",
    "meets_requirements",
]

# Type definition for property verification functions
PropertyVerifier = Callable[[nx.Graph, Any], bool]

# Property registry
GRAPH_PROPERTIES: Dict[str, PropertyVerifier] = {}


def register_property(name: str):
    """Decorator to register a property verification function."""

    def decorator(func: PropertyVerifier):
        GRAPH_PROPERTIES[name] = func
        return func

    return decorator


# Basic property verification functions


@register_property("connected")
def is_connected(G: nx.Graph, _: Any = None) -> bool:
    """Check if the graph is connected."""
    return nx.is_connected(G)


@register_property("acyclic")
def is_acyclic(G: nx.Graph, _: Any = None) -> bool:
    """Check if the graph is acyclic (has no cycles)."""
    return nx.is_forest(G)

@register_property("cyclic")
def is_cyclic(G: nx.Graph, _: Any = None) -> bool:
    """Check if the graph contains at least one cycle."""
    # A graph has cycles if it's not a forest
    return not nx.is_forest(G)


@register_property("tree")
def is_tree(G: nx.Graph, _: Any = None) -> bool:
    """Check if the graph is a tree (connected and acyclic)."""
    return nx.is_tree(G)


@register_property("has_internal_node")
def has_internal_node(G: nx.Graph, _: Any = None) -> bool:
    """Check if the graph has at least one internal node, i.e., a node with degree > 1."""
    return any(degree > 1 for _, degree in G.degree())


@register_property("bipartite")
def is_bipartite(G: nx.Graph, _: Any = None) -> bool:
    """Check if the graph is bipartite."""
    return nx.is_bipartite(G)


@register_property("has_colored_node")
def has_colored_node(G: nx.Graph, color: str = None) -> bool:
    """Check if the graph has at least one colored node."""
    if color:
        return any(data.get("color", "grey") == color for _, data in G.nodes(data=True))
    else:
        return any(
            data.get("color", "grey") != "grey" for _, data in G.nodes(data=True)
        )


@register_property("bipartition_seeds_colored")
def has_bipartition_seeds_colored(G: nx.Graph, _: Any = None) -> bool:
    """
    Check if a bipartite graph has at least one colored node in each partition.

    This requires:
    1. The graph must be bipartite
    2. There must be at least two differently colored nodes, one in each partition
    """
    # First check if the graph is bipartite
    if not nx.is_bipartite(G):
        return False

    try:
        # Get the two partitions
        part0, part1 = nx.bipartite.sets(G)
    except nx.NetworkXError:
        return False

    # Check if there's at least one non-grey colored node in part0
    has_colored_node_part0 = any(
        G.nodes[node].get("color", "grey") != "grey" for node in part0
    )

    # Check if there's at least one non-grey colored node in part1
    has_colored_node_part1 = any(
        G.nodes[node].get("color", "grey") != "grey" for node in part1
    )

    # Find colors used in each partition
    part0_colors = {
        G.nodes[node].get("color")
        for node in part0
        if G.nodes[node].get("color", "grey") != "grey"
    }
    part1_colors = {
        G.nodes[node].get("color")
        for node in part1
        if G.nodes[node].get("color", "grey") != "grey"
    }

    # Ensure we have at least one color in each partition
    # and that the colors are different between partitions
    return (
        has_colored_node_part0
        and has_colored_node_part1
        and part0_colors.isdisjoint(part1_colors)
    )


@register_property("all_nodes_colored")
def all_nodes_colored(G: nx.Graph, _: Any = None) -> bool:
    """Check if all nodes in the graph are colored (non-grey)."""
    return all(data.get("color", "grey") != "grey" for _, data in G.nodes(data=True))


# Parameterized property generators


def has_degree(degree: int) -> str:
    """Generate a property name for having nodes with a specific degree."""
    prop_name = f"has_degree_{degree}"

    # Register the property dynamically if not already registered
    if prop_name not in GRAPH_PROPERTIES:

        @register_property(prop_name)
        def has_degree_n(G: nx.Graph, _: Any = None) -> bool:
            """Check if the graph has at least one node with the specified degree."""
            return any(d == degree for _, d in G.degree())

    return prop_name


def has_min_nodes(min_nodes: int) -> str:
    """Generate a property name for having at least a minimum number of nodes."""
    prop_name = f"has_min_nodes_{min_nodes}"

    if prop_name not in GRAPH_PROPERTIES:

        @register_property(prop_name)
        def has_min_nodes_n(G: nx.Graph, _: Any = None) -> bool:
            """Check if the graph has at least min_nodes nodes."""
            return len(G.nodes) >= min_nodes

    return prop_name


def has_colored_leaves(count: int) -> str:
    """Generate a property name for having a specific number of colored leaves.

    Parameters:
    - count: Minimum number of colored leaves required
    - color: Color to look for (default: "blue")

    Returns:
    - Property name string
    """
    prop_name = f"has_colored_leaves_{count}"

    if prop_name not in GRAPH_PROPERTIES:

        @register_property(prop_name)
        def has_n_colored_leaves(G: nx.Graph, params: Any = None) -> bool:
            """Check if the graph has at least 'count' colored leaves."""
            # Determine which color to check for
            check_color = "blue"  # Default color

            # If params is specified and is a string, use it as the color
            if isinstance(params, str):
                check_color = params
            # If params is a dict with a 'color' key, use that
            elif isinstance(params, dict) and "color" in params:
                check_color = params["color"]

            # Find colored leaves
            colored_leaves = [
                node
                for node, data in G.nodes(data=True)
                if data.get("color", "grey") == check_color and G.degree[node] == 1
            ]

            # Debug output
            if len(colored_leaves) < count:
                print(
                    f"Looking for {count} leaves with color '{check_color}', found {len(colored_leaves)}: {colored_leaves}"
                )

            return len(colored_leaves) >= count

    return prop_name


def has_components(count: int) -> str:
    """Generate a property name for having a specific number of connected components."""
    prop_name = f"has_components_{count}"

    if prop_name not in GRAPH_PROPERTIES:

        @register_property(prop_name)
        def has_n_components(G: nx.Graph, _: Any = None) -> bool:
            """Check if the graph has exactly 'count' connected components."""
            return nx.number_connected_components(G) == count

    return prop_name


def has_colored_nodes(count: int, color: str = "blue") -> str:
    """
    Generate a property name for having exactly a specific number of nodes with a color.
    
    Parameters:
    - count: Exact number of colored nodes required
    - color: The color to check for
    
    Returns:
    - Property name string
    """
    prop_name = f"has_colored_nodes_{count}_{color}"
    
    if prop_name not in GRAPH_PROPERTIES:
        @register_property(prop_name)
        def has_n_colored_nodes(G: nx.Graph, _: Any = None) -> bool:
            """Check if the graph has exactly 'count' nodes with the specified color."""
            colored_nodes = [
                node for node, data in G.nodes(data=True)
                if data.get("color", "grey") == color
            ]
            return len(colored_nodes) == count
    
    return prop_name


def has_multiple_colors(min_colors: int = 2) -> str:
    """
    Generate a property name for having at least a minimum number of different colors.
    
    Parameters:
    - min_colors: Minimum number of different colors required
    
    Returns:
    - Property name string
    """
    prop_name = f"has_multiple_colors_{min_colors}"
    
    if prop_name not in GRAPH_PROPERTIES:
        @register_property(prop_name)
        def has_min_colors(G: nx.Graph, _: Any = None) -> bool:
            """Check if the graph uses at least 'min_colors' different colors."""
            colors = set()
            for _, data in G.nodes(data=True):
                color = data.get("color", "grey")
                if color != "grey":
                    colors.add(color)
            return len(colors) >= min_colors
    
    return prop_name

def has_equidistant_node() -> str:
    """
    Generate a property name for having at least one node equidistant from two blue nodes.
    """
    prop_name = "has_equidistant_node"
    if prop_name not in GRAPH_PROPERTIES:

        @register_property(prop_name)
        def _has_equidistant_node(G: nx.Graph, _: Any = None) -> bool:
            blue_nodes = [n for n, d in G.nodes(data=True) if d.get("color") == "blue"]
            if len(blue_nodes) != 2:
                return False
            v, w = blue_nodes
            for node in G.nodes:
                if node in blue_nodes:
                    continue
                try:
                    dist_v = nx.shortest_path_length(G, v, node)
                    dist_w = nx.shortest_path_length(G, w, node)
                except nx.NetworkXNoPath:
                    continue
                if dist_v == dist_w:
                    return True
            return False

    return prop_name


def all_nodes_colored_with(min_colors: int = 2) -> str:
    """
    Generate a property name for having all nodes colored with at least min_colors different colors.
    
    Parameters:
    - min_colors: Minimum number of different colors required
    
    Returns:
    - Property name string
    """
    prop_name = f"all_nodes_colored_with_{min_colors}"
    
    if prop_name not in GRAPH_PROPERTIES:
        @register_property(prop_name)
        def all_colored_with_min(G: nx.Graph, _: Any = None) -> bool:
            """Check if all nodes are colored and at least 'min_colors' different colors are used."""
            colors = set()
            for _, data in G.nodes(data=True):
                color = data.get("color", "grey")
                if color == "grey":
                    return False  # Found an uncolored node
                colors.add(color)
            return len(colors) >= min_colors
    
    return prop_name


# Functions to check properties


def check_property(G: nx.Graph, property_name: str, params: Any = None) -> bool:
    """
    Check if a graph has a specific property.

    Parameters:
    - G: NetworkX graph to check
    - property_name: Name of the property to verify
    - params: Optional parameters for the property verification

    Returns:
    - bool: Whether the graph has the specified property
    """
    if property_name not in GRAPH_PROPERTIES:
        raise ValueError(f"Unknown property: {property_name}")

    verifier = GRAPH_PROPERTIES[property_name]
    result = verifier(G, params)
    if not result:
        print(f"Property '{property_name}' failed for the graph.")
    return result


def check_properties(G: nx.Graph, properties: List[str]) -> Dict[str, bool]:
    """
    Check multiple properties on a graph.

    Parameters:
    - G: NetworkX graph to check
    - properties: List of property names to verify

    Returns:
    - Dict mapping property names to verification results
    """
    return {prop: check_property(G, prop) for prop in properties}


def meets_requirements(G: nx.Graph, required_properties: List[str]) -> bool:
    """
    Check if a graph meets all the required properties.

    Parameters:
    - G: NetworkX graph to check
    - required_properties: List of property names required

    Returns:
    - bool: Whether the graph meets all requirements
    """
    return all(check_property(G, prop) for prop in required_properties)


# Functions to get property information


def get_graph_properties(G: nx.Graph) -> Set[str]:
    """
    Get the set of properties that a graph has.

    Parameters:
    - G: NetworkX graph to analyze

    Returns:
    - Set of property names that the graph satisfies
    """
    return {prop for prop in GRAPH_PROPERTIES if check_property(G, prop)}


def list_all_properties() -> List[str]:
    """List all registered property names."""
    return list(GRAPH_PROPERTIES.keys())
