import networkx as nx
from typing import Dict, List, Callable, Any, Tuple
from scripts.utils.pretransformations import apply_pretransformations

# Type definitions
GraphTransformer = Callable[[nx.Graph, Dict[str, Any]], nx.Graph]
PropertyList = List[str]
PreTransformList = List[Tuple[str, Dict[str, Any]]]

class Task:
    """
    A graph transformation task.
    """
    def __init__(
        self,
        name: str,
        transformation: GraphTransformer,
        required_properties: PropertyList = None,
        required_pretransforms: PreTransformList = None,
        preferred_generators: List[str] = None,
        parameter_schema: Dict[str, Any] = None,
        description: str = None
    ):
        self.name = name
        self.transformation = transformation
        self.required_properties = required_properties or []
        self.required_pretransforms = required_pretransforms or []
        self.preferred_generators = preferred_generators or []
        self.parameter_schema = parameter_schema or {}
        self.description = description or ""
    
    def apply(self, G: nx.Graph, params: Dict[str, Any] = None) -> nx.Graph:
        """
        Apply the task to a graph, including any required pre-transformations.
        
        Parameters:
        - G: NetworkX graph to transform
        - params: Parameters for the transformation
        
        Returns:
        - Transformed NetworkX graph
        """
        
        result = G.copy()
        params = params or {}
        
        # Apply pre-transformations if needed
        if self.required_pretransforms:
            result = apply_pretransformations(result, self.required_pretransforms)
        
        # Apply main transformation
        return self.transformation(result, params)
    
    def __str__(self) -> str:
        return f"Task(name={self.name})"
    
    def __repr__(self) -> str:
        return self.__str__()
