from scripts.utils.task_base import Task
from scripts.utils.task_definition import get_task, TASKS
from scripts.utils.pretransformations import get_pretransformation
from typing import Dict, List, Callable
from scripts.utils.properties import PropertyStatus, TRUE, FALSE, MAYBE

from scripts.utils.create_graph import (
    generate_random_2_component_graph,
    generate_random_bipartite_graph,
    generate_random_connected_graph,
    generate_random_graph,
    generate_random_tree,
    generate_star_graph,
)

# Graph types with their properties
GRAPH_TYPES = {
    "random": "Random Erdős-Rényi graph",
    "randomConnected": "Random connected graph",
    "random2Component": "Random graph with exactly 2 components, one with a blue node and the other with an orange node",
    "randomTree": "Random tree structure",
    "star": "Star-shaped graph with center hub",
    "bipartite": "Bipartite graph which is 2-colorable",
}

# Graph generator properties
# key: generator name, value: dict of properties (property_name -> TRUE, FALSE, or MAYBE)
GRAPH_GENERATORS = {
    "random": {
        "connected": MAYBE,
        "acyclic": MAYBE,
        "cyclic": MAYBE,
        "bipartite": MAYBE,
        "has_degree_1": MAYBE,
        "has_degree_2": MAYBE,
        "has_degree_3": MAYBE,
        "has_internal_node": MAYBE,
    },
    "randomConnected": {
        "connected": TRUE,
        "acyclic": MAYBE,
        "cyclic": MAYBE,
        "bipartite": MAYBE,
        "has_degree_1": MAYBE,
        "has_degree_2": MAYBE,
        "has_degree_3": MAYBE,
        "has_internal_node": MAYBE,
    },
    "random2Component": {
        "connected": FALSE,
        "has_colored_node": TRUE,
        "has_components_2": TRUE,
        "has_colored_nodes_1_blue": TRUE,      
        "has_colored_nodes_1_orange": TRUE,
    },
    "randomTree": {
        "connected": TRUE,
        "acyclic": TRUE,
        "tree": TRUE,
        "cyclic": FALSE,
        "has_degree_1": TRUE,
        "has_degree_2": MAYBE,
        "has_degree_3": MAYBE,
        "has_internal_node": TRUE,
    },
    "star": {
        "connected": TRUE,
        "acyclic": TRUE,
        "tree": TRUE,
        "has_center": TRUE,
        "has_leaves": TRUE,
        "has_degree_1": TRUE,
        "has_internal_node": TRUE,
        "has_colored_leaves_2": MAYBE,
    },
    "bipartite": {
        "bipartite": TRUE,
    },
}

GRAPH_GENERATORS_FUNCTIONS = {
    "random": generate_random_graph,
    "randomConnected": generate_random_connected_graph,
    "random2Component": generate_random_2_component_graph,
    "randomTree": generate_random_tree,
    "star": generate_star_graph,
    "bipartite": generate_random_bipartite_graph,
}


def is_generator_compatible_with_task(generator_name: str, task: Task) -> bool:
    """
    Check if a generator is compatible with a task based on properties.
    Now uses the PropertyStatus enum values for clear distinction between
    guaranteed, potential, and impossible properties.

    Parameters:
    - generator_name: Name of the graph generator
    - task: Task object to check compatibility with

    Returns:
    - bool: Whether the generator is compatible with the task
    """

    # Check if the generator is in the preferred list (if specified)
    if task.preferred_generators and generator_name not in task.preferred_generators:
        return False

    # Get the generator's properties
    generator_props = GRAPH_GENERATORS.get(generator_name, {})

    # Create a set of all required properties
    required_properties = set(task.required_properties)

    # Create a set of properties that will be added by pre-transformations
    provided_properties = set()

    # Add properties that would be added by pre-transformations
    if task.required_pretransforms:
        for pretransform_name, params in task.required_pretransforms:
            pretransform = get_pretransformation(pretransform_name)
            if pretransform:
                # Add these properties to the provided set
                provided_properties.update(pretransform.get_provided_properties(params))

    # Check if the generator has all the required properties that aren't provided by pre-transformations
    for required_prop in required_properties:
        # Skip properties that will be added by pre-transformations
        if required_prop in provided_properties:
            continue

        # Check if the generator provides this property
        if required_prop in generator_props:
            prop_status = generator_props[required_prop]
            # If the property is guaranteed to be FALSE, the generator is incompatible
            if prop_status == FALSE:
                return False
        # If the property is not listed, we have to assume it's MAYBE
        # since we don't know if the generator provides it

    return True


def find_compatible_generators_for_task(task_name: str) -> List[str]:
    """
    Find all graph generators that are compatible with a task.

    Parameters:
    - task_name: Name of the task

    Returns:
    - List of compatible generator names
    """
    task = get_task(task_name)
    if not task:
        raise ValueError(f"Unknown task: {task_name}")

    compatible = []

    for generator_name in GRAPH_GENERATORS:
        if is_generator_compatible_with_task(generator_name, task):
            compatible.append(generator_name)

    return compatible


def find_compatible_tasks_for_generator(generator_name: str) -> List[str]:
    """
    Find all tasks that are compatible with a graph generator.

    Parameters:
    - generator_name: Name of the graph generator

    Returns:
    - List of compatible task names
    """

    if generator_name not in GRAPH_GENERATORS:
        raise ValueError(f"Unknown generator: {generator_name}")

    compatible = []

    for task_name, task in TASKS.items():
        if is_generator_compatible_with_task(generator_name, task):
            compatible.append(task_name)

    return compatible


def build_compatibility_matrix() -> Dict[str, List[str]]:
    """
    Build a dictionary mapping each task to its compatible generators.
    This is a dynamic replacement for the hardcoded BENCHMARK_COMPATIBILITY.

    Returns:
    - Dict mapping task names to lists of compatible generator names
    """

    compatibility = {}

    for task_name in TASKS:
        compatibility[task_name] = find_compatible_generators_for_task(task_name)

    return compatibility


def get_generator_function(generator_name: str):
    """
    Get the generator function for a given generator name.

    Parameters:
    - generator_name: Name of the graph generator

    Returns:
    - Generator function

    Raises:
    - ValueError if the generator is not found
    """
    if generator_name not in GRAPH_GENERATORS_FUNCTIONS:
        raise ValueError(f"Unknown generator: {generator_name}")

    return GRAPH_GENERATORS_FUNCTIONS[generator_name]


def get_generator_properties(generator_func: Callable) -> Dict[str, PropertyStatus]:
    """
    Get the properties dictionary for a generator function.

    Parameters:
    - generator_func: Graph generator function

    Returns:
    - Dictionary of property names to PropertyStatus values
    """
    for name, func in GRAPH_GENERATORS_FUNCTIONS.items():
        if func == generator_func:
            return GRAPH_GENERATORS.get(name, {})

    # If not found, return empty dict
    return {}
