"""
Task definition framework for graph transformation tasks.

This module provides:
1. A structure for defining graph transformation tasks
2. A registry of tasks with their properties and transformations
3. Utilities for compatibility checking and task application
"""

import networkx as nx
from typing import Dict, List, Any, Optional

from scripts.utils.task_base import (
    Task,
    GraphTransformer,
    PropertyList,
    PreTransformList,
)
from scripts.utils.properties import meets_requirements


# Task registry
TASKS: Dict[str, Task] = {}


def register_task(
    name: str,
    required_properties: PropertyList = None,
    required_pretransforms: PreTransformList = None,
    preferred_generators: List[str] = None,
    parameter_schema: Dict[str, Any] = None,
    description: str = None,
):
    """
    Decorator to register a task function.

    Parameters:
    - name: Name of the task
    - required_properties: List of properties required for the task
    - required_pretransforms: List of pre-transformations to apply
    - preferred_generators: List of preferred graph generators
    - parameter_schema: Dictionary describing the parameters for the task
    - description: Human-readable description of the task
    """

    def decorator(func: GraphTransformer):
        TASKS[name] = Task(
            name=name,
            transformation=func,
            required_properties=required_properties,
            required_pretransforms=required_pretransforms,
            preferred_generators=preferred_generators,
            parameter_schema=parameter_schema,
            description=description,
        )
        return func

    return decorator


# Functions to work with tasks


def apply_task(G: nx.Graph, task_name: str, params: Dict[str, Any] = None) -> nx.Graph:
    """
    Apply a named task to a graph.

    Parameters:
    - G: NetworkX graph to transform
    - task_name: Name of the registered task
    - params: Parameters for the task

    Returns:
    - Transformed NetworkX graph
    """
    if task_name not in TASKS:
        raise ValueError(f"Unknown task: {task_name}")

    task = TASKS[task_name]

    if not meets_requirements(G, task.required_properties):
        raise ValueError(f"Task '{task_name}' is not applicable to the given graph")

    return task.apply(G, params)


def list_tasks() -> List[str]:
    """List all registered task names."""
    return list(TASKS.keys())


def get_task(name: str) -> Optional[Task]:
    """Get a task by name."""
    return TASKS.get(name)


def find_compatible_tasks(G: nx.Graph) -> List[str]:
    """
    Find all tasks that are compatible with a given graph.

    Parameters:
    - G: NetworkX graph to check

    Returns:
    - List of compatible task names
    """

    return [
        name
        for name, task in TASKS.items()
        if meets_requirements(G, task.required_properties)
    ]


def get_task_metadata(task_name: str) -> Dict[str, Any]:
    """
    Get metadata about a task for use in specs.json.

    Parameters:
    - task_name: Name of the registered task

    Returns:
    - Dictionary with task metadata including description and requirements
    """
    task = get_task(task_name)
    if not task:
        return {
            "description": f"Unknown task: {task_name}",
            "property_requirements": [],
            "transformation_parameters": {},
        }

    return {
        "description": task.description,
        "property_requirements": task.required_properties,
        "transformation_parameters": task.parameter_schema,
    }
