import argparse
import os
import sys
import traceback
import pandas as pd
from .generators.overview_generator import generate_overview_visualizations
from .generators.detailed_generator import generate_detailed_visualizations
from .generators.summary_generator import generate_summary_visualizations
from .generators.prompt_token_generator import generate_prompt_token_visualizations
from .generators.presentation_generator import generate_presentation_visualizations

from .core.data_loader import (
    find_latest_evaluation_file,
    load_evaluation_data,
    validate_data,
    get_available_values,
)
from .core.utils import setup_plot_style

VISUALIZATION_PROFILES = {
    "legacy": {
        "exclude_patterns": ["cap10_3", "cap25_3", "cap50_3", "cap100_3", "cap250_3"],
        "output_suffix": "_legacy",
        "description": "All patterns except the new scaling patterns (cap series)",
    },
    "scaling": {
        "include_patterns": ["cap10_3", "cap25_3", "cap50_3", "cap100_3", "cap250_3"],
        "output_suffix": "_scaling",
        "description": "Only the new scaling patterns (cap10_3 through cap250_3)",
    },
    "all": {"output_suffix": "_complete", "description": "All patterns included"},
}


def main():
    """Main entry point for visualization system."""
    parser = argparse.ArgumentParser(
        description="Generate visualizations for graph-based ARC evaluation results"
    )

    parser.add_argument(
        "evaluation_file",
        nargs="?",
        help="Path to evaluation results JSON file (default: latest in evaluation_data/). Not required for token analysis.",
    )

    parser.add_argument(
        "--tasks", nargs="+", help="Specific tasks to visualize (default: all)"
    )

    parser.add_argument(
        "--models", nargs="+", help="Specific models to include (default: all)"
    )

    parser.add_argument(
        "--question-types",
        nargs="+",
        help="Specific question types to include (default: all)",
    )

    parser.add_argument(
        "--profile",
        choices=list(VISUALIZATION_PROFILES.keys()),
        help="Use predefined visualization profile. Available profiles: "
        + ", ".join(
            [f"{k} ({v['description']})" for k, v in VISUALIZATION_PROFILES.items()]
        ),
    )

    parser.add_argument(
        "--include-patterns",
        nargs="+",
        help="Only include these size patterns (default: all). Overrides profile settings.",
    )

    parser.add_argument(
        "--exclude-patterns",
        nargs="+",
        help="Exclude these size patterns (default: none). Overrides profile settings.",
    )

    parser.add_argument(
        "--output-suffix",
        help="Suffix to add to output directory name (default: from profile or empty). Overrides profile settings.",
    )

    parser.add_argument(
        "--visualizations",
        nargs="+",
        choices=["overview", "detailed", "summary", "prompt-token", "presentation"],  # Added presentation
        default=["overview", "detailed", "summary"],
        help="Which visualization types to generate (default: overview, detailed, summary)",
    )

    parser.add_argument(
        "--output-dir",
        default="visualizations",
        help="Output directory for visualizations (default: visualizations/)",
    )

    parser.add_argument(
        "--datasets-dir",
        default="datasets",
        help="Path to datasets directory for token analysis (default: datasets/)",
    )

    parser.add_argument(
        "--verbose", action="store_true", help="Print detailed progress information"
    )

    parser.add_argument(
        "--no-titles", action="store_true", help="Generate all figures without titles"
    )

    args = parser.parse_args()

    # APPLY PROFILE SETTINGS
    profile_config = {}
    if args.profile:
        profile_config = VISUALIZATION_PROFILES[args.profile].copy()
        if args.verbose:
            print(f"📋 Using profile '{args.profile}': {profile_config['description']}")

    # Manual arguments override profile settings
    include_patterns = args.include_patterns or profile_config.get("include_patterns")
    exclude_patterns = args.exclude_patterns or profile_config.get("exclude_patterns")
    output_suffix = args.output_suffix or profile_config.get("output_suffix", "")

    # Modify output directory with suffix
    if output_suffix:
        args.output_dir = args.output_dir.rstrip("/") + output_suffix
        if args.verbose:
            print(f"📁 Output directory: {args.output_dir}")

    # Print no-titles info if verbose
    if args.no_titles and args.verbose:
        print("🚫 Generating figures without titles")

    # Set up plotting style
    setup_plot_style()

    # Check if we need evaluation data (not required for prompt-token-only analysis)
    needs_evaluation_data = any(
        viz in args.visualizations for viz in ["overview", "detailed", "summary", "presentation"]  # Added presentation
    )

    data = pd.DataFrame()  # Initialize empty DataFrame
    validation_info = {}
    available_values = {}

    if needs_evaluation_data:
        # Find evaluation file
        if args.evaluation_file:
            eval_file = args.evaluation_file
        else:
            eval_file = find_latest_evaluation_file()
            if not eval_file:
                print("❌ No evaluation results file found.")
                print("   Please run evaluation first or specify a file explicitly.")
                sys.exit(1)

            if args.verbose:
                print(f"📊 Using latest evaluation file: {eval_file}")

        if not os.path.exists(eval_file):
            print(f"❌ Evaluation file not found: {eval_file}")
            sys.exit(1)

        # Load and validate data
        try:
            if args.verbose:
                print("📖 Loading evaluation data...")

            filters = {}
            if args.models:
                filters["models"] = args.models
            if args.tasks:
                filters["tasks"] = args.tasks
            if args.question_types:
                filters["question_types"] = args.question_types

            # ADD PATTERN FILTERING
            if include_patterns:
                filters["include_patterns"] = include_patterns
                if args.verbose:
                    print(f"   Including patterns: {', '.join(include_patterns)}")
            if exclude_patterns:
                filters["exclude_patterns"] = exclude_patterns
                if args.verbose:
                    print(f"   Excluding patterns: {', '.join(exclude_patterns)}")

            data = load_evaluation_data(eval_file, filters)

            if data.empty:
                print("❌ No data found with the specified filters.")
                sys.exit(1)

            # Validate data
            validation_info = validate_data(data)

            if args.verbose:
                print(f"✅ Loaded {validation_info['total_records']} records")
                print(f"   Models: {validation_info['unique_models']}")
                print(f"   Tasks: {validation_info['unique_tasks']}")
                print(f"   Question types: {validation_info['unique_question_types']}")
                print(f"   Overall accuracy: {validation_info['overall_accuracy']:.3f}")

            # Get available values for summary
            available_values = get_available_values(data)

            if args.verbose:
                print("\n📋 Available data:")
                for key, values in available_values.items():
                    print(
                        f"   {key}: {len(values)} ({', '.join(values[:3])}{'...' if len(values) > 3 else ''})"
                    )

        except (FileNotFoundError, ValueError, KeyError, OSError) as e:
            print(f"❌ Error loading evaluation data: {e}")
            if args.verbose:
                traceback.print_exc()
            sys.exit(1)

    # Create output directory
    os.makedirs(args.output_dir, exist_ok=True)

    # Track all results
    all_visualization_results = {
        "evaluation_file": getattr(args, "evaluation_file", None),
        "datasets_dir": args.datasets_dir,
        "profile_used": args.profile,
        "filters_applied": filters if needs_evaluation_data else {},
        "data_summary": validation_info,
        "available_values": available_values,
        "overview_results": None,
        "detailed_results": None,
        "summary_results": None,
        "token_results": None,
        "presentation_results": None,  # NEW
        "total_files_generated": 0,
        "no_titles": args.no_titles,  # Store the no_titles setting
    }

    # Generate visualizations
    try:
        if "overview" in args.visualizations:
            print("🎯 Generating overview visualizations...")
            overview_results = generate_overview_visualizations(
                data, args.output_dir, args.verbose, no_titles=args.no_titles
            )
            all_visualization_results["overview_results"] = overview_results
            all_visualization_results["total_files_generated"] += overview_results[
                "total_files"
            ]

        if "detailed" in args.visualizations:
            print("🔍 Generating detailed visualizations...")
            detailed_results = generate_detailed_visualizations(
                data, args.output_dir, args.verbose, no_titles=args.no_titles
            )
            all_visualization_results["detailed_results"] = detailed_results
            all_visualization_results["total_files_generated"] += detailed_results[
                "total_files"
            ]

        if "summary" in args.visualizations:
            print("📈 Generating summary visualizations...")
            summary_results = generate_summary_visualizations(
                data, args.output_dir, args.verbose, no_titles=args.no_titles
            )
            all_visualization_results["summary_results"] = summary_results
            all_visualization_results["total_files_generated"] += summary_results[
                "total_files"
            ]

        if "presentation" in args.visualizations:  # NEW
            print("🎤 Generating presentation visualizations...")
            presentation_results = generate_presentation_visualizations(
                data, args.output_dir, args.verbose, no_titles=args.no_titles
            )
            all_visualization_results["presentation_results"] = presentation_results
            all_visualization_results["total_files_generated"] += presentation_results[
                "total_files"
            ]

        if "prompt-token" in args.visualizations:
            print("🔤 Generating prompt token analysis visualizations...")

            # Check if datasets directory exists
            if not os.path.exists(args.datasets_dir):
                print(f"❌ Datasets directory not found: {args.datasets_dir}")
                print(
                    "   Prompt token analysis requires access to the datasets directory."
                )
                sys.exit(1)

            token_results = generate_prompt_token_visualizations(
                args.datasets_dir,
                args.output_dir,
                args.verbose,
                no_titles=args.no_titles,
            )
            all_visualization_results["token_results"] = token_results
            all_visualization_results["total_files_generated"] += token_results[
                "total_files"
            ]

        # Create master summary
        create_master_summary(all_visualization_results, args.output_dir, args.verbose)

        print(
            f"✅ Generated {all_visualization_results['total_files_generated']} visualizations in {args.output_dir}/"
        )
        print(f"📋 See {args.output_dir}/README.md for a complete overview")

    except (FileNotFoundError, ValueError, KeyError, OSError) as e:
        print(f"❌ Error generating visualizations: {e}")
        if args.verbose:
            traceback.print_exc()
        sys.exit(1)


def create_master_summary(results: dict, output_dir: str, verbose: bool = False):
    """
    Create a master README summarizing all generated visualizations.

    Parameters:
    - results: Dictionary with all visualization results
    - output_dir: Output directory
    - verbose: Whether to print progress
    """
    if verbose:
        print("📋 Creating master summary...")

    readme_content = """# Graph-Based ARC Evaluation Visualizations

"""

    # Add profile info if used
    profile_used = results.get("profile_used")
    if profile_used:
        profile_config = VISUALIZATION_PROFILES[profile_used]
        readme_content += (
            f"**Profile used**: `{profile_used}` - {profile_config['description']}\n\n"
        )

        if "include_patterns" in profile_config:
            readme_content += f"**Patterns included**: {', '.join(profile_config['include_patterns'])}\n\n"
        elif "exclude_patterns" in profile_config:
            readme_content += f"**Patterns excluded**: {', '.join(profile_config['exclude_patterns'])}\n\n"

    # Add no-titles info if used
    if results.get("no_titles"):
        readme_content += "**Note**: All figures generated without titles.\n\n"

    # Add evaluation file info if available
    if results.get("evaluation_file"):
        readme_content += f"Generated from: `{results['evaluation_file']}`\n\n"

    # Add datasets info if prompt token analysis was run
    if results.get("token_results"):
        readme_content += (
            f"Prompt token analysis from: `{results['datasets_dir']}/`\n\n"
        )

    # Data summary (if evaluation data was used)
    if results["data_summary"]:
        readme_content += f"""## Data Summary

- **Total records**: {results['data_summary']['total_records']:,}
- **Overall accuracy**: {results['data_summary']['overall_accuracy']:.4f}
- **Unique models**: {results['data_summary']['unique_models']}
- **Unique tasks**: {results['data_summary']['unique_tasks']}
- **Unique question types**: {results['data_summary']['unique_question_types']}

## Applied Filters

"""

        if results["filters_applied"]:
            for filter_type, filter_values in results["filters_applied"].items():
                readme_content += f"- **{filter_type}**: {', '.join(filter_values)}\n"
        else:
            readme_content += "- No filters applied (all data included)\n"

    readme_content += f"""
## Generated Visualizations

**Total files generated**: {results['total_files_generated']}

"""

    # Presentation section (NEW - put first for prominence)
    if results["presentation_results"]:
        presentation = results["presentation_results"]
        readme_content += f"""### 🎤 Presentation Visualizations ({presentation['total_files']} files)

Located in: `{os.path.relpath(presentation['presentation_dir'], output_dir)}/`

**Purpose**: Clean, high-impact visualizations optimized for talks and presentations

**Key Features**:
- Combined model performance (full output vs. question-based tasks)
- Large fonts and clear visual hierarchy
- Minimal clutter, no floating annotations
- Different bar patterns for clear distinction
- PDF format for crisp projection

"""

    # Overview section
    if results["overview_results"]:
        overview = results["overview_results"]
        readme_content += f"""### 🎯 Overview Visualizations ({overview['total_files']} files)

Located in: `{os.path.relpath(overview['overview_dir'], output_dir)}/`

**Purpose**: High-level performance insights and comparisons

**Key Charts**:
- Overall model performance (full output tasks)
- Question-based task performance  
- Input vs output comparison
- Question type breakdown
- System prompt impact analysis
- Task performance summary
- Token usage overview (if available)

"""

    # Detailed section
    if results["detailed_results"]:
        detailed = results["detailed_results"]
        readme_content += f"""### 🔍 Detailed Visualizations ({detailed['total_files']} files)

Located in: `{os.path.relpath(detailed['detailed_dir'], output_dir)}/`

**Purpose**: Deep-dive analysis per task and error patterns

**Structure**:
"""
        if detailed["task_analyses"]:
            readme_content += f"- **Per-task analysis**: {len(detailed['task_analyses'])} tasks analyzed\n"
            for task in list(detailed["task_analyses"].keys())[:5]:  # Show first 5
                readme_content += f"  - `task_{task}/` - Question types, input/output analysis, system prompts, graph types\n"
            if len(detailed["task_analyses"]) > 5:
                readme_content += (
                    f"  - ... and {len(detailed['task_analyses']) - 5} more tasks\n"
                )

        readme_content += """- **Error analysis** - Success rates, target difficulty, challenging tasks
"""

    # Summary section
    if results["summary_results"]:
        summary = results["summary_results"]
        readme_content += f"""### 📈 Summary Reports ({summary['total_files']} files)

Located in: `{os.path.relpath(summary['summary_dir'], output_dir)}/`

**Purpose**: Statistical analysis and data quality assessment

**Reports**:
- **Performance summaries** (`.txt`, `.json`) - Comprehensive statistics
- **Data coverage analysis** - Sample distribution heatmaps
- **Sample size analysis** - Statistical reliability assessment  
- **Pattern impact analysis** - Performance by size pattern and model
- **Example size analysis** - Model performance vs number of examples
- **Cross-modal analysis** - Full output vs question-based comparison

"""

    # Prompt token analysis section
    if results["token_results"]:
        token = results["token_results"]
        readme_content += f"""### 🔤 Prompt Token Analysis ({token['total_files']} files)

Located in: `{os.path.relpath(token['token_dir'], output_dir)}/`

**Purpose**: Comprehensive analysis of token requirements for graph descriptions and prompt construction

**Key Analyses**:
- **Encoding Efficiency**: Adjacency vs Incident token comparison
- **Token Scaling**: How token requirements grow with graph size (all sizes discovered dynamically)
- **Task Complexity**: Token requirements by task and encoding
- **Prompt Construction**: Token costs for 1, 2, 3 examples
- **Strategic Insights**: Encoding decisions and budget optimization

**Data Files**:
- `prompt_token_data.csv` - Individual file token counts
- `prompt_construction_data.csv` - Prompt-level statistics  
- `prompt_token_summary.json` - Summary statistics with dataset structure
- `PROMPT_TOKEN_ANALYSIS_README.md` - Detailed analysis guide

"""

    # Usage instructions
    readme_content += """## How to Use These Visualizations

### Quick Start
1. **Start with Presentation** - Use `presentation/` for talks and high-level summaries  
2. **Overview for Analysis** - Get the big picture from `overview/`
3. **Dive into Details** - Explore specific tasks in `detailed/task_*/`
4. **Check Data Quality** - Review `summary/` for statistical confidence
5. **Optimize Prompt Token Usage** - Use `prompt_token_analysis/` for LLM cost planning

### Key Files to Check First
"""

    # Add presentation files if available (NEW - show these first)
    if results.get("presentation_results"):
        readme_content += """- `presentation/01_combined_model_performance.pdf` - Main results for presentations
"""
    
    readme_content += """- `overview/01_overall_model_performance.pdf` - Main results
- `overview/04_question_type_breakdown.pdf` - Question difficulty
- `summary/performance_summary.txt` - Statistical overview
- `summary/data_coverage_report.txt` - Sample sizes
"""

    # Add prompt token analysis key files if available
    if results.get("token_results"):
        readme_content += """- `prompt_token_analysis/01_encoding_efficiency_comparison.pdf` - Encoding comparison
- `prompt_token_analysis/05_prompt_construction_analysis.pdf` - Prompt optimization
- `prompt_token_analysis/PROMPT_TOKEN_ANALYSIS_README.md` - Complete prompt token guide
"""

    readme_content += """
### Understanding the Charts
- **Bar charts**: Higher bars = better performance  
- **Sample counts**: `n=X` shows number of evaluations
- **Accuracy scale**: 0.0 (0%) to 1.0 (100%)
- **Token counts**: Number of words/tokens in graph descriptions
- **Error bars**: Show statistical confidence where applicable
- **Bar patterns**: Solid = full output tasks, Striped/hatched = question-based tasks

## Available Data Dimensions

"""

    for dimension, values in results["available_values"].items():
        if values:
            readme_content += f"**{dimension.title()}**: {', '.join(values[:10])}"
            if len(values) > 10:
                readme_content += f" (and {len(values) - 10} more)"
            readme_content += "\n\n"

    readme_content += f"""## Generation Details

- **Generated on**: {pd.Timestamp.now().strftime('%Y-%m-%d %H:%M:%S')}
- **Visualization system**: Graph-Based ARC v2.0
- **Total processing time**: See console output for timing details

---

*For questions about these visualizations, refer to the main repository documentation.*
"""

    # Save README
    readme_path = os.path.join(output_dir, "README.md")
    with open(readme_path, "w", encoding="utf-8") as f:
        f.write(readme_content)

    if verbose:
        print(f"📋 Master summary saved to {readme_path}")


if __name__ == "__main__":
    main()